//
// MessagePack for Java
//
// Copyright (C) 2009-2011 FURUHASHI Sadayuki
//
//    Licensed under the Apache License, Version 2.0 (the "License");
//    you may not use this file except in compliance with the License.
//    You may obtain a copy of the License at
//
//        http://www.apache.org/licenses/LICENSE-2.0
//
//    Unless required by applicable law or agreed to in writing, software
//    distributed under the License is distributed on an "AS IS" BASIS,
//    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//    See the License for the specific language governing permissions and
//    limitations under the License.
//
package org.msgpack.template.builder;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import org.msgpack.*;
import org.msgpack.template.*;

import javassist.CannotCompileException;
import javassist.CtClass;
import javassist.CtConstructor;
import javassist.CtNewConstructor;
import javassist.NotFoundException;

public class BeansBuildContext extends BuildContextBase<BeansFieldEntry> {
	protected BeansFieldEntry[] entries;
	protected Class<?> origClass;
	protected String origName;
	protected Template[] templates;
	protected int minimumArrayLength;

	public BeansBuildContext(JavassistTemplateBuilder director) {
		super(director);
	}
	
	public Template buildTemplate(Class<?> targetClass, BeansFieldEntry[] entries, Template[] templates) {
		this.entries = entries;
		this.templates = templates;
		this.origClass = targetClass;
		this.origName = this.origClass.getName();
		return build(this.origName);
	}

	protected void setSuperClass() throws CannotCompileException, NotFoundException {
		this.tmplCtClass.setSuperclass(
				director.getCtClass(JavassistTemplateBuilder.JavassistTemplate.class.getName()));
	}

	protected void buildConstructor() throws CannotCompileException, NotFoundException {
		// Constructor(Class targetClass, Template[] templates)
		CtConstructor newCtCons = CtNewConstructor.make(
			new CtClass[] {
				director.getCtClass(Class.class.getName()),
				director.getCtClass(Template.class.getName()+"[]")
			},
			new CtClass[0],
			this.tmplCtClass);
		this.tmplCtClass.addConstructor(newCtCons);
	}

	protected Template buildInstance(Class<?> c) throws NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException {
		Constructor<?> cons = c.getConstructor(new Class[] {
				Class.class,
				Template[].class
			});
		Object tmpl = cons.newInstance(new Object[] {
				this.origClass,
				this.templates
			});
		return (Template)tmpl;
	}

	protected void buildMethodInit() {
		this.minimumArrayLength = 0;
		for(int i=0; i < entries.length; i++) {
			IFieldEntry e = entries[i];
			if(e.isRequired() || e.isNullable()) {
				this.minimumArrayLength = i+1;
			}
		}
	}

	protected String buildPackMethodBody() {
		resetStringBuilder();
		buildString("{");
		buildString("%s _$$_t = (%s)$2;", this.origName, this.origName);
		buildString("$1.packArray(%d);", entries.length);
		for(int i=0; i < entries.length; i++) {
			BeansFieldEntry e = entries[i];
			if(!e.isAvailable()) {
				buildString("$1.packNil();");
				continue;
			}
			Class<?> type = e.getType();
			if(type.isPrimitive()) {
				buildString("$1.%s(_$$_t.%s());", primitivePackName(type), e.getGetterName());
			} else {
				buildString("if(_$$_t.%s() == null) {", e.getGetterName());
				if(!e.isNullable() && !e.isOptional()) {
					buildString("throw new %s();", MessageTypeException.class.getName());
				} else {
					buildString("$1.packNil();");
				}
				buildString("} else {");
				buildString("  this.templates[%d].pack($1, _$$_t.%s());", i, e.getGetterName());
				buildString("}");
			}
		}
		buildString("}");
		return getBuiltString();
	}

	protected String buildUnpackMethodBody() {
		resetStringBuilder();
		buildString("{ ");

		buildString("%s _$$_t;", this.origName);
		buildString("if($2 == null) {");
		buildString("  _$$_t = new %s();", this.origName);
		buildString("} else {");
		buildString("  _$$_t = (%s)$2;", this.origName);
		buildString("}");

		buildString("int length = $1.unpackArray();");
		buildString("if(length < %d) {", this.minimumArrayLength);
		buildString("  throw new %s();", MessageTypeException.class.getName());
		buildString("}");

		int i;
		for(i=0; i < this.minimumArrayLength; i++) {
			BeansFieldEntry e = entries[i];
			if(!e.isAvailable()) {
				buildString("$1.unpackObject();");
				continue;
			}

			buildString("if($1.tryUnpackNull()) {");
				if(e.isRequired()) {
					// Required + nil => exception
					buildString("throw new %s();", MessageTypeException.class.getName());
				} else if(e.isOptional()) {
					// Optional + nil => keep default value
				} else {  // Nullable
					// Nullable + nil => set null
					buildString("_$$_t.%s(null);", e.getSetterName());
				}
			buildString("} else {");
				Class<?> type = e.getType();
				if(type.isPrimitive()) {
					buildString("_$$_t.set%s( $1.%s() );", e.getName(), primitiveUnpackName(type));
				} else {
					buildString("_$$_t.set%s( (%s)this.templates[%d].unpack($1, _$$_t.get%s()) );", e.getName(), e.getJavaTypeName(), i, e.getName());
				}
			buildString("}");
		}

		for(; i < entries.length; i++) {
			buildString("if(length <= %d) { return _$$_t; }", i);

			BeansFieldEntry e = entries[i];
			if(!e.isAvailable()) {
				buildString("$1.unpackObject();");
				continue;
			}

			buildString("if($1.tryUnpackNull()) {");
				// this is Optional field becaue i >= minimumArrayLength
				// Optional + nil => keep default value
			buildString("} else {");
				Class<?> type = e.getType();
				if(type.isPrimitive()) {
					buildString("_$$_t.%s( $1.%s() );", e.getSetterName(), primitiveUnpackName(type));
				} else {
					buildString("_$$_t.%s( (%s)this.templates[%d].unpack($1, _$$_t.%s()) );", e.getSetterName(), e.getJavaTypeName(), i, e.getGetterName());
				}
			buildString("}");
		}

		// latter entries are all Optional + nil => keep default value

		buildString("for(int i=%d; i < length; i++) {", i);
		buildString("  $1.unpackObject();");
		buildString("}");

		buildString("return _$$_t;");

		buildString("}");
		return getBuiltString();
	}

	protected String buildConvertMethodBody() {
		resetStringBuilder();
		buildString("{ ");

		buildString("%s _$$_t;", this.origName);
		buildString("if($2 == null) {");
		buildString("  _$$_t = new %s();", this.origName);
		buildString("} else {");
		buildString("  _$$_t = (%s)$2;", this.origName);
		buildString("}");

		buildString("%s[] array = $1.asArray();", MessagePackObject.class.getName());
		buildString("int length = array.length;");
		buildString("if(length < %d) {", this.minimumArrayLength);
		buildString("  throw new %s();", MessageTypeException.class.getName());
		buildString("}");

		buildString("%s obj;", MessagePackObject.class.getName());

		int i;
		for(i=0; i < this.minimumArrayLength; i++) {
			BeansFieldEntry e = entries[i];
			if(!e.isAvailable()) {
				continue;
			}

			buildString("obj = array[%d];", i);
			buildString("if(obj.isNil()) {");
				if(e.isRequired()) {
					// Required + nil => exception
					buildString("throw new %s();", MessageTypeException.class.getName());
				} else if(e.isOptional()) {
					// Optional + nil => keep default value
				} else {  // Nullable
					// Nullable + nil => set null
					buildString("_$$_t.%s( null );", e.getSetterName());
				}
			buildString("} else {");
				Class<?> type = e.getType();
				if(type.isPrimitive()) {
					buildString("_$$_t.%s( obj.%s() );", e.getSetterName(), primitiveConvertName(type));
				} else {
					buildString("_$$_t.%s( (%s)this.templates[%d].convert(obj, _$$_t.%s()) );", e.getSetterName(), e.getJavaTypeName(), i, e.getGetterName());
				}
			buildString("}");
		}

		for(; i < entries.length; i++) {
			buildString("if(length <= %d) { return _$$_t; }", i);

			BeansFieldEntry e = entries[i];
			if(!e.isAvailable()) {
				continue;
			}

			buildString("obj = array[%d];", i);
			buildString("if(obj.isNil()) {");
				// this is Optional field becaue i >= minimumArrayLength
				// Optional + nil => keep default value
			buildString("} else {");
				Class<?> type = e.getType();
				if(type.isPrimitive()) {
					buildString("_$$_t.%s( obj.%s() );", e.getSetterName(), primitiveConvertName(type));
				} else {
					buildString("_$$_t.%s( (%s)this.templates[%d].convert(obj, _$$_t.%s()) );", e.getSetterName(), e.getJavaTypeName(), i, e.getGetterName());
				}
			buildString("}");
		}

		// latter entries are all Optional + nil => keep default value

		buildString("return _$$_t;");

		buildString("}");
		return getBuiltString();
	}

	@Override
	public void writeTemplate(Class<?> targetClass, BeansFieldEntry[] entries,
			Template[] templates, String directoryName) {
		throw new UnsupportedOperationException(targetClass.getName());
	}

	@Override
	public Template loadTemplate(Class<?> targetClass, BeansFieldEntry[] entries, Template[] templates) {
		return null;
	}
}