var path = require('path');
var nroonga = require('../../wrapped-nroonga');
var Domain = require('../../database').Domain;
var dateFormat = require('dateformat');
var xmlbuilder = require('../../xmlbuilder');
var ipv4 = require('../../ipv4');
var uuid = require('node-uuid');

exports.version = path.basename(__dirname);

var XMLNS = 'http://cloudsearch.amazonaws.com/doc/2011-02-01';

function createCommonErrorResponse(errorCode, error) {
  if (error.message) {
    error = error.message;
  } else {
    error = error.toString();
  }
  var doc = xmlbuilder.create();
  doc.begin('Response', { version: '1.0' })
    .element('Errors')
      .element('Error')
        .element('Code').text(errorCode).up()
        .element('Message').text(error).up()
      .up()
    .up()
    .element('RequestID').text(uuid.v4()).up();
  return doc.toString();
}

var handlers = Object.create(null);

var defaultHttpPort = 80;
var defaultBaseHost = '127.0.0.1.xip.io';

function getBaseHostAndPort(config, request) {
  var host = config.baseHost ||
             request.headers.http_x_forwarded_host ||
             request.headers.host ||
             defaultBaseHost + ':' + config.port;
  var port = defaultHttpPort;

  var portMatching = host.match(/:(\d+)$/);
  if (portMatching) {
    host = host.replace(portMatching[0], '');
    port = parseInt(portMatching[1]);
  }

  if (port == defaultHttpPort)
    return host;
  else
    return host + ':' + port;
}

function createGenericResponse(action, result) {
  var doc = xmlbuilder.create();
  var root = doc.begin(action + 'Response', { version: '1.0' })
                .attribute('xmlns', XMLNS);

  var resultContainer = root.element(action + 'Result');
  if (result) resultContainer.importXMLBuilder(result);

  root.element('ResponseMetadata')
      .element('RequestId').text(uuid.v4());

  return doc.toString();
}

function createDomainStatus(options) {
  var domainStatus = xmlbuilder.create();
  domainStatus.begin(options.element || 'DomainStatus', { version: '1.0' })
    .element('Created').text(options.created || 'false').up()
    .element('Deleted').text(options.deleted || 'false').up()
    .element('DocService')
      .element('Endpoint').text(options.domain.getDocumentsEndpoint(options.hostAndPort)).up()
    .up()
    .element('DomainId')
      .text(options.domain.domainId)
    .up()
    .element('DomainName').text(options.domain.name).up()
    .element('NumSearchableDocs')
      .text(options.domain.searchableDocumentsCount)
    .up()
    .element('RequiresIndexDocuments')
      .text(options.domain.requiresIndexDocuments)
    .up()
    .element('SearchInstanceCount')
      .text(options.domain.searchInstanceCount)
    .up()
    .element('SearchPartitionCount')
      .text(options.domain.searchPartitionCount)
    .up()
    .element('SearchService')
      .element('Endpoint').text(options.domain.getSearchEndpoint(options.hostAndPort)).up()
    .up();
  return domainStatus;
}

handlers.CreateDomain = function(context, request, response, config) {
  var domain = new Domain(request.query.DomainName, context);
  domain.createSync();
  var result = createDomainStatus({
        domain:      domain,
        hostAndPort: getBaseHostAndPort(config, request),
        created:     true
      });
  response.contentType('application/xml');
  response.send(createGenericResponse('CreateDomain', result));
};

handlers.DeleteDomain = function(context, request, response, config) {
  var domain = new Domain(request.query.DomainName, context);
  domain.deleteSync();
  var result = createDomainStatus({
        domain:      domain,
        hostAndPort: getBaseHostAndPort(config, request),
        deleted:     true
      });
  response.contentType('application/xml');
  response.send(createGenericResponse('DeleteDomain', result));
};

function createDomainStatusList(options) {
  var doc = xmlbuilder.create();
  var domainStatusList = doc.begin('DomainStatusList', {version: '1.0'});
  options.domains.forEach(function(domain) {
    domainStatusList.importXMLBuilder(createDomainStatus({
                       domain:      domain,
                       hostAndPort: options.hostAndPort,
                       element:     'member'
                     }));
  });
  return doc;
}

handlers.DescribeDomains = function(context, request, response, config) {
  var keys = Object.keys(request.query).filter(function(key) {
        return /^DomainNames\.member\.\d+$/.test(key);
      });
  var domainNames = keys.sort().map(function(key) {
        return request.query[key];
      });
  var domains = domainNames.length ?
                  domainNames.map(function(name) {
                    var domain = new Domain(name, context);
                    return domain.exists() ? domain : null ;
                  }).filter(function(domain) {
                    return domain;
                  }) :
                  Domain.getAll(context) ;
  var result = createDomainStatusList({
        domains:     domains,
        hostAndPort: getBaseHostAndPort(config, request)
      });
  response.contentType('application/xml');
  response.send(createGenericResponse('DescribeDomains', result));
};

function createIndexFieldOptionStatus(options) {
  switch (options.field.type) {
    case 'text':
      var textOptions = xmlbuilder.create();
      textOptions.begin('TextOptions', { version: '1.0' })
        .element('DefaultValue').up()
        .element('FacetEnabled').text(options.field.facetEnabled).up()
        .element('ResultEnabled').text(options.field.resultEnabled);
      return textOptions;
    case 'uint':
      var uintOptions = xmlbuilder.create();
      uintOptions.begin('UIntOptions', { version: '1.0' })
        .element('DefaultValue');
      return uintOptions;
    case 'literal':
      var literalOptions = xmlbuilder.create();
      literalOptions.begin('LiteralOptions', { version: '1.0' })
        .element('DefaultValue').up()
        .element('FacetEnabled').text(options.field.facetEnabled).up()
        .element('ResultEnabled').text(options.field.resultEnabled).up()
        .element('SearchEnabled').text(options.field.searchEnabled);
      return literalOptions;
  }
}

function createOptionStatus(options) {
  var optionStatus = xmlbuilder.create();
  optionStatus.begin(options.element || 'Status', { version: '1.0' })
    .element('CreationDate').text(dateFormat(options.createdAt,
                                               'isoUtcDateTime')).up()
    .element('State').text(options.state || 'RequiresIndexDocuments').up()
    .element('UpdateDate').text(dateFormat(options.updatedAt,
                                             'isoUtcDateTime')).up()
    .element('UpdateVersion').text(options.updateVersion || '0');
  return optionStatus;
}

function createIndexFieldStatus(options) {
  var indexFieldStatus = xmlbuilder.create();
  indexFieldStatus.begin(options.element || 'IndexField', { version: '1.0' })
    .element('Options')
      .element('IndexFieldName').text(options.field.name).up()
      .element('IndexFieldType').text(options.field.type).up()
      .importXMLBuilder(createIndexFieldOptionStatus(options))
    .up()
    .importXMLBuilder(createOptionStatus({ createdAt:     options.createdAt,
                                           state:         options.field.state,
                                           updatedAt:     options.updatedAt,
                                           updateVersion: options.updateVersion,
                                           element:       'Status' }));
  return indexFieldStatus;
}

function getFieldOption(option, request, type) {
  if (type == 'text')
    return request.query['IndexField.TextOptions.' + option];
  if (type == 'literal')
    return request.query['IndexField.LiteralOptions.' + option];
  else
    return request.query['IndexField.UIntOptions.' + option];
}

handlers.DefineIndexField = function(context, request, response, config) {
  var domain = new Domain(request.query.DomainName, context);

  var fieldName = request.query['IndexField.IndexFieldName'] || '';
  var fieldType = request.query['IndexField.IndexFieldType'] || 'text';
  var field = domain.getIndexField(fieldName);

  var createdAt = new Date();

  if (!field.exists())
    field.type = fieldType;

  var facetEnabled = getFieldOption('FacetEnabled', request, fieldType);
  if (facetEnabled !== undefined)
    field.facetEnabled = facetEnabled.toLowerCase() == 'true';

  var resultEnabled = getFieldOption('ResultEnabled', request, fieldType);
  if (resultEnabled !== undefined)
    field.resultEnabled = resultEnabled.toLowerCase() == 'true';

  var searchEnabled = getFieldOption('SearchEnabled', request, fieldType);
  if (searchEnabled !== undefined)
    field.searchEnabled = searchEnabled.toLowerCase() == 'true';

  if (!field.exists())
    field.createSync();
  else
    field.saveOptionsSync();

  var result = createIndexFieldStatus({
        field: field,
        createdAt: createdAt,
        updatedAt: createdAt
      });
  response.contentType('application/xml');
  response.send(createGenericResponse('DefineIndexField', result));
};

handlers.DeleteIndexField = function(context, request, response, config) {
  var domain = new Domain(request.query.DomainName, context);

  var fieldName = request.query['IndexFieldName'] || '';
  var field = domain.getIndexField(fieldName);
  field.deleteSync();
  response.contentType('application/xml');
  response.send(createGenericResponse('DeleteIndexField'));
};

function createIndexFields(fields) {
  var doc = xmlbuilder.create();
  var indexFields = doc.begin('IndexFields', {version: '1.0'});
  fields.forEach(function(field) {
    indexFields.importXMLBuilder(createIndexFieldStatus({
                  field:   field,
                  element: 'member'
                }));
  });
  return doc;
}

handlers.DescribeIndexFields = function(context, request, response, config) {
  var domain = new Domain(request.query.DomainName, context);

  var keys = Object.keys(request.query).filter(function(key) {
        return /^FieldNames\.member\.\d+$/.test(key);
      });
  var fieldNames = keys.sort().map(function(key) {
        return request.query[key];
      });
  var fields = fieldNames.length ?
                  fieldNames.map(function(name) {
                    var field = domain.getIndexField(name);
                    return field.exists() ? field : null ;
                  }).filter(function(field) {
                    return field;
                  }) :
                  domain.indexFields ;
  var result = createIndexFields(fields);
  response.contentType('application/xml');
  response.send(createGenericResponse('DescribeIndexFields', result));
};

function createFieldNames(domain) {
  var doc = xmlbuilder.create();
  var fieldNames = doc.begin('FieldNames', {version: '1.0'});
  domain.indexFields
    .map(function(field) {
      return field.name;
    })
    .sort()
    .forEach(function(fieldName) {
      var member = xmlbuilder.create();
      member.begin('member', { version: '1.0' })
        .text(fieldName);
     fieldNames.importXMLBuilder(member);
    });
  return doc;
}

handlers.IndexDocuments = function(context, request, response, config) {
  var domain = new Domain(request.query.DomainName, context);
  domain.reindexSync();
  var result = createFieldNames(domain);
  response.contentType('application/xml');
  response.send(createGenericResponse('IndexDocuments', result));
};

function createSynonymOptionsStatus(options) {
  var synonyms = options.domain.getSynonymsSync();
  var synonymOptions = { synonyms: synonyms };

  var synonymOptionsStatus = xmlbuilder.create();
  synonymOptionsStatus.begin('Synonyms', { version: '1.0' })
    .element('Options')
      .text(JSON.stringify(synonymOptions))
    .up()
    .importXMLBuilder(createOptionStatus({ createdAt:     options.createdAt,
                                           state:         options.state,
                                           updatedAt:     options.updatedAt,
                                           updateVersion: options.updateVersion,
                                           element:       'Status' }));
  return synonymOptionsStatus;
}

handlers.UpdateSynonymOptions = function(context, request, response, config) {
  var domain = new Domain(request.query.DomainName, context);
  var synonymOptionsJson = request.query.Synonyms;
  var synonymOptions = JSON.parse(synonymOptionsJson);
  domain.updateSynonymsSync(synonymOptions.synonyms);

  var updatedAt = new Date();
  var result = createSynonymOptionsStatus({
        domain:    domain,
        updatedAt: updatedAt,
        createdAt: updatedAt
      });
  response.contentType('application/xml');
  response.send(createGenericResponse('UpdateSynonymOptions', result));
};

handlers.DescribeSynonymOptions = function(context, request, response, config) {
  var domain = new Domain(request.query.DomainName, context);
  var updatedAt = new Date();
  var result = createSynonymOptionsStatus({
        domain:    domain,
        updatedAt: updatedAt,
        createdAt: updatedAt
      });
  response.contentType('application/xml');
  response.send(createGenericResponse('DescribeSynonymOptions', result));
};

function createDefaultSearchFieldStatus(options) {
  var defaultSearchFieldStatus = xmlbuilder.create();
  defaultSearchFieldStatus.begin('DefaultSearchField', { version: '1.0' })
    .element('Options').text(options.fieldName).up()
    .importXMLBuilder(createOptionStatus({
       createdAt:     options.createdAt,
       state:         options.state,
       updatedAt:     options.updatedAt,
       updateVersion: options.updateVersion,
       element:       'Status'
    }));
  return defaultSearchFieldStatus;
}

handlers.UpdateDefaultSearchField = function(context, request, response, config) {
  var domain = new Domain(request.query.DomainName, context);
  var fieldName = request.query.DefaultSearchField;
  domain.defaultSearchField = fieldName;

  var updatedAt = new Date();
  var result = createDefaultSearchFieldStatus({
        fieldName: fieldName,
        createdAt: updatedAt,
        updatedAt: updatedAt
      });
  response.contentType('application/xml');
  response.send(createGenericResponse('UpdateDefaultSearchField', result));
};

handlers.DescribeDefaultSearchField = function(context, request, response, config) {
  var domain = new Domain(request.query.DomainName, context);
  var field = domain.defaultSearchField;
  var updatedAt = new Date();
  var result = createDefaultSearchFieldStatus({
        fieldName: field ? field.name : '',
        createdAt: updatedAt,
        updatedAt: updatedAt
      });
  response.contentType('application/xml');
  response.send(createGenericResponse('DescribeDefaultSearchField', result));
};

function getClientIp(request) {
  var forwardedIps = request.header('x-forwarded-for');
  if (forwardedIps) {
    var ip = forwardedIps.split(',')[0];
    if (ip)
      return ip;
  }
  return request.connection.remoteAddress;
};

exports.createHandler = function(context, config) {
  var privilegedRanges = config &&
                         config.privilegedRanges &&
                         config.privilegedRanges.split(/[,\| ]/);
  return function(request, response, next) {
    var message, body;

    // GCS specific behaviour: prevent to access this API from specific IP
    // range.
    if (privilegedRanges && privilegedRanges.length) {
      if (!privilegedRanges.some(function(privilegedRange) {
            return ipv4.isInRange(getClientIp(request), privilegedRange);
          })) {
        message = 'Permission denied.';
        body = createCommonErrorResponse('InvalidClientIpRange', message);
        response.contentType('application/xml');
        return response.send(body, 403);
      }
    }

    // GCS specific behaviour: fallback to other handlers for the endpoint
    // if no action is given.
    var action = request.query.Action || '';
    if (!action)
      return next();

    // ACS document says "The API version must be specified in all requests."
    // but actual implementation of ACS accepts configuration requests
    // without Version specification.
    // See http://docs.amazonwebservices.com/cloudsearch/latest/developerguide/ConfigAPI.html
    var version = request.query.Version;
    if (version && version != exports.version) {
      message = 'A bad or out-of-range value "' + version + '" was supplied ' +
                'for the "Version" input parameter.';
      body = createCommonErrorResponse('InvalidParameterValue', message);
      response.contentType('application/xml');
      return response.send(body, 400);
    }

    if (!(action in handlers)) {
      message = 'The action ' + action + ' is not valid for this web service.';
      body = createCommonErrorResponse('InvalidAction', message);
      response.contentType('application/xml');
      return response.send(body, 400);
    }

    var handler = handlers[action];
    try {
      handler(context, request, response, config);
    } catch (error) {
      var body = createCommonErrorResponse('InternalFailure', error);
      response.contentType('application/xml');
      response.send(body, 400);
    }
  };
};
