(function(root) {

	/** Use a single `load` function */
	var load = typeof require == 'function' ? require : root.load,

	/** The unit testing framework */
	QUnit =
		root.QUnit ||
		(root.QUnit = load('../vendor/qunit/qunit/qunit.js') || root.QUnit) &&
		(load('../vendor/qunit-clib/qunit-clib.js'), root.QUnit),

	/** The `punycode` object to test */
	punycode = root.punycode || load('../punycode.js') || root.punycode,

	/** Data that will be used in the tests */
	testData = {
		'strings': [
			{
				'description': 'a single basic code point',
				'decoded': 'Bach',
				'encoded': 'Bach-'
			},
			{
				'description': 'a single non-ASCII character',
				'decoded': '\xFC',
				'encoded': 'tda'
			},
			{
				'description': 'multiple non-ASCII characters',
				'decoded': '\xFC\xEB\xE4\xF6\u2665',
				'encoded': '4can8av2009b'
			},
			{
				'description': 'mix of ASCII and non-ASCII characters',
				'decoded': 'b\xFCcher',
				'encoded': 'bcher-kva'
			},
			{
				'description': 'long string with both ASCII and non-ASCII characters',
				'decoded': 'Willst du die Bl\xFCthe des fr\xFChen, die Fr\xFCchte des sp\xE4teren Jahres',
				'encoded': 'Willst du die Blthe des frhen, die Frchte des spteren Jahres-x9e96lkal'
			},
			// http://tools.ietf.org/html/rfc3492#section-7.1
			{
				'description': 'Arabic (Egyptian)',
				'decoded': '\u0644\u064A\u0647\u0645\u0627\u0628\u062A\u0643\u0644\u0645\u0648\u0634\u0639\u0631\u0628\u064A\u061F',
				'encoded': 'egbpdaj6bu4bxfgehfvwxn'
			},
			{
				'description': 'Chinese (simplified)',
				'decoded': '\u4ED6\u4EEC\u4E3A\u4EC0\u4E48\u4E0D\u8BF4\u4E2d\u6587',
				'encoded': 'ihqwcrb4cv8a8dqg056pqjye'
			},
			{
				'description': 'Chinese (traditional)',
				'decoded': '\u4ED6\u5011\u7232\u4EC0\u9EBD\u4E0D\u8AAA\u4E2D\u6587',
				'encoded': 'ihqwctvzc91f659drss3x8bo0yb'
			},
			{
				'description': 'Czech',
				'decoded': 'Pro\u010Dprost\u011Bnemluv\xED\u010Desky',
				'encoded': 'Proprostnemluvesky-uyb24dma41a'
			},
			{
				'description': 'Hebrew',
				'decoded': '\u05DC\u05DE\u05D4\u05D4\u05DD\u05E4\u05E9\u05D5\u05D8\u05DC\u05D0\u05DE\u05D3\u05D1\u05E8\u05D9\u05DD\u05E2\u05D1\u05E8\u05D9\u05EA',
				'encoded': '4dbcagdahymbxekheh6e0a7fei0b'
			},
			{
				'description': 'Hindi (Devanagari)',
				'decoded': '\u092F\u0939\u0932\u094B\u0917\u0939\u093F\u0928\u094D\u0926\u0940\u0915\u094D\u092F\u094B\u0902\u0928\u0939\u0940\u0902\u092C\u094B\u0932\u0938\u0915\u0924\u0947\u0939\u0948\u0902',
				'encoded': 'i1baa7eci9glrd9b2ae1bj0hfcgg6iyaf8o0a1dig0cd'
			},
			{
				'description': 'Japanese (kanji and hiragana)',
				'decoded': '\u306A\u305C\u307F\u3093\u306A\u65E5\u672C\u8A9E\u3092\u8A71\u3057\u3066\u304F\u308C\u306A\u3044\u306E\u304B',
				'encoded': 'n8jok5ay5dzabd5bym9f0cm5685rrjetr6pdxa'
			},
			{
				'description': 'Korean (Hangul syllables)',
				'decoded': '\uC138\uACC4\uC758\uBAA8\uB4E0\uC0AC\uB78C\uB4E4\uC774\uD55C\uAD6D\uC5B4\uB97C\uC774\uD574\uD55C\uB2E4\uBA74\uC5BC\uB9C8\uB098\uC88B\uC744\uAE4C',
				'encoded': '989aomsvi5e83db1d2a355cv1e0vak1dwrv93d5xbh15a0dt30a5jpsd879ccm6fea98c'
			},
			/**
			 * As there's no way to do it in JavaScript, Punycode.js doesn't support
			 * mixed-case annotation (which is entirely optional as per the RFC).
			 * So, while the RFC sample string encodes to:
			 * `b1abfaaepdrnnbgefbaDotcwatmq2g4l`
			 * Without mixed-case annotation it has to encode to:
			 * `b1abfaaepdrnnbgefbadotcwatmq2g4l`
			 * https://github.com/bestiejs/punycode.js/issues/3
			 */
			{
				'description': 'Russian (Cyrillic)',
				'decoded': '\u043F\u043E\u0447\u0435\u043C\u0443\u0436\u0435\u043E\u043D\u0438\u043D\u0435\u0433\u043E\u0432\u043E\u0440\u044F\u0442\u043F\u043E\u0440\u0443\u0441\u0441\u043A\u0438',
				'encoded': 'b1abfaaepdrnnbgefbadotcwatmq2g4l'
			},
			{
				'description': 'Spanish',
				'decoded': 'Porqu\xE9nopuedensimplementehablarenEspa\xF1ol',
				'encoded': 'PorqunopuedensimplementehablarenEspaol-fmd56a'
			},
			{
				'description': 'Vietnamese',
				'decoded': 'T\u1EA1isaoh\u1ECDkh\xF4ngth\u1EC3ch\u1EC9n\xF3iti\u1EBFngVi\u1EC7t',
				'encoded': 'TisaohkhngthchnitingVit-kjcr8268qyxafd2f1b9g'
			},
			{
				'decoded': '3\u5E74B\u7D44\u91D1\u516B\u5148\u751F',
				'encoded': '3B-ww4c5e180e575a65lsy2b'
			},
			{
				'decoded': '\u5B89\u5BA4\u5948\u7F8E\u6075-with-SUPER-MONKEYS',
				'encoded': '-with-SUPER-MONKEYS-pc58ag80a8qai00g7n9n'
			},
			{
				'decoded': 'Hello-Another-Way-\u305D\u308C\u305E\u308C\u306E\u5834\u6240',
				'encoded': 'Hello-Another-Way--fc4qua05auwb3674vfr0b'
			},
			{
				'decoded': '\u3072\u3068\u3064\u5C4B\u6839\u306E\u4E0B2',
				'encoded': '2-u9tlzr9756bt3uc0v'
			},
			{
				'decoded': 'Maji\u3067Koi\u3059\u308B5\u79D2\u524D',
				'encoded': 'MajiKoi5-783gue6qz075azm5e'
			},
			{
				'decoded': '\u30D1\u30D5\u30A3\u30FCde\u30EB\u30F3\u30D0',
				'encoded': 'de-jg4avhby1noc0d'
			},
			{
				'decoded': '\u305D\u306E\u30B9\u30D4\u30FC\u30C9\u3067',
				'encoded': 'd9juau41awczczp'
			},
			/**
			 * This example is an ASCII string that breaks the existing rules for host
			 * name labels. (It's not a realistic example for IDNA, because IDNA never
			 * encodes pure ASCII labels.)
			 */
			{
				'description': 'ASCII string that breaks the existing rules for host-name labels',
				'decoded': '-> $1.00 <-',
				'encoded': '-> $1.00 <--'
			}
		],
		'ucs2': [
			{
				'description': 'printable ASCII characters',
				'decoded': [32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126],
				'encoded': ' !"#$%&\'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~'
			},
			{
				'description': 'Apple logo (private range)',
				'decoded': [63743],
				'encoded': '\uF8FF'
			},
			{
				'decoded': [102, 3232, 246, 32, 98, 97, 114, 32, 98, 229, 122, 32, 113, 252, 120],
				'encoded': 'f\u0CA0\xF6 bar b\xE5\x7A q\xFCx'
			},
			{
				'decoded': [127829, 119808, 119558, 119638],
				'encoded': '\uD83C\uDF55\uD835\uDC00\uD834\uDF06\uD834\uDF56'
			},
			{
				'description': 'Unicode character category [Cc]: other, control',
				'decoded': [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 127, 128, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155, 156, 157, 158, 159],
				'encoded': '\x00\x01\x02\x03\x04\x05\x06\x07\b\t\n\x0B\f\r\x0E\x0F\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1A\x1B\x1C\x1D\x1E\x1F\x7F\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8A\x8B\x8C\x8D\x8E\x8F\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9A\x9B\x9C\x9D\x9E\x9F'
			},
			{
				'description': 'Unicode character category [Cf]: other, format',
				'decoded': [173, 1536, 1537, 1538, 1539, 1757, 1807, 6068, 6069, 8203, 32, 8205, 8206, 8207, 8234, 8235, 8236, 8237, 8238, 8288, 8289, 8290, 8291, 8292, 8298, 8299, 8300, 8301, 8302, 8303, 65279, 65529, 65530, 65531, 69821, 119155, 119156, 119157, 119158, 119159, 119160, 119161, 119162, 917505, 917536, 917537, 917538, 917539, 917540, 917541, 917542, 917543, 917544, 917545, 917546, 917547, 917548, 917549, 917550, 917551, 917552, 917553, 917554, 917555, 917556, 917557, 917558, 917559, 917560, 917561, 917562, 917563, 917564, 917565, 917566, 917567, 917568, 917569, 917570, 917571, 917572, 917573, 917574, 917575, 917576, 917577, 917578, 917579, 917580, 917581, 917582, 917583, 917584, 917585, 917586, 917587, 917588, 917589, 917590, 917591, 917592, 917593, 917594, 917595, 917596, 917597, 917598, 917599, 917600, 917601, 917602, 917603, 917604, 917605, 917606, 917607, 917608, 917609, 917610, 917611, 917612, 917613, 917614, 917615, 917616, 917617, 917618, 917619, 917620, 917621, 917622, 917623, 917624, 917625, 917626, 917627, 917628, 917629, 917630, 917631],
				'encoded': '\xAD\u0600\u0601\u0602\u0603\u06DD\u070F\u17B4\u17B5\u200B \u200D\u200E\u200F\u202A\u202B\u202C\u202D\u202E\u2060\u2061\u2062\u2063\u2064\u206A\u206B\u206C\u206D\u206E\u206F\uFEFF\uFFF9\uFFFA\uFFFB\uD804\uDCBD\uD834\uDD73\uD834\uDD74\uD834\uDD75\uD834\uDD76\uD834\uDD77\uD834\uDD78\uD834\uDD79\uD834\uDD7A\uDB40\uDC01\uDB40\uDC20\uDB40\uDC21\uDB40\uDC22\uDB40\uDC23\uDB40\uDC24\uDB40\uDC25\uDB40\uDC26\uDB40\uDC27\uDB40\uDC28\uDB40\uDC29\uDB40\uDC2A\uDB40\uDC2B\uDB40\uDC2C\uDB40\uDC2D\uDB40\uDC2E\uDB40\uDC2F\uDB40\uDC30\uDB40\uDC31\uDB40\uDC32\uDB40\uDC33\uDB40\uDC34\uDB40\uDC35\uDB40\uDC36\uDB40\uDC37\uDB40\uDC38\uDB40\uDC39\uDB40\uDC3A\uDB40\uDC3B\uDB40\uDC3C\uDB40\uDC3D\uDB40\uDC3E\uDB40\uDC3F\uDB40\uDC40\uDB40\uDC41\uDB40\uDC42\uDB40\uDC43\uDB40\uDC44\uDB40\uDC45\uDB40\uDC46\uDB40\uDC47\uDB40\uDC48\uDB40\uDC49\uDB40\uDC4A\uDB40\uDC4B\uDB40\uDC4C\uDB40\uDC4D\uDB40\uDC4E\uDB40\uDC4F\uDB40\uDC50\uDB40\uDC51\uDB40\uDC52\uDB40\uDC53\uDB40\uDC54\uDB40\uDC55\uDB40\uDC56\uDB40\uDC57\uDB40\uDC58\uDB40\uDC59\uDB40\uDC5A\uDB40\uDC5B\uDB40\uDC5C\uDB40\uDC5D\uDB40\uDC5E\uDB40\uDC5F\uDB40\uDC60\uDB40\uDC61\uDB40\uDC62\uDB40\uDC63\uDB40\uDC64\uDB40\uDC65\uDB40\uDC66\uDB40\uDC67\uDB40\uDC68\uDB40\uDC69\uDB40\uDC6A\uDB40\uDC6B\uDB40\uDC6C\uDB40\uDC6D\uDB40\uDC6E\uDB40\uDC6F\uDB40\uDC70\uDB40\uDC71\uDB40\uDC72\uDB40\uDC73\uDB40\uDC74\uDB40\uDC75\uDB40\uDC76\uDB40\uDC77\uDB40\uDC78\uDB40\uDC79\uDB40\uDC7A\uDB40\uDC7B\uDB40\uDC7C\uDB40\uDC7D\uDB40\uDC7E\uDB40\uDC7F'
			},
			{
				// This category contains no characters
				'description': 'Unicode character category [Cn]: other, not assigned',
				'decoded': [],
				'encoded': ''
			},
			{
				'description': 'Unicode character category [Co]: other, private use',
				'decoded': [57344, 63744, 983040, 1048574, 1048576, 1114110],
				'encoded': '\uE000\uF900\uDB80\uDC00\uDBBF\uDFFE\uDBC0\uDC00\uDBFF\uDFFE'
			},
			{
				// This category contains no characters
				'description': 'Unicode character category [LC]: letter, cased',
				'decoded': [],
				'encoded': ''
			},
			{
				'description': 'Unicode character category [Ll]: letter, lowercase',
				'decoded': [97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 170, 181, 186, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 248, 249, 250, 251, 252, 253, 254, 255, 257, 259, 261, 263, 265, 267, 269, 271, 273, 275, 277, 279, 281, 283, 285, 287, 289, 291, 293, 295, 297, 299, 301, 303, 305, 307, 309, 311, 312, 314, 316, 318, 320, 322, 324, 326, 328, 329, 331, 333, 335, 337, 339, 341, 343, 345, 347, 349, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 378, 380, 382, 383, 384, 387, 389, 392, 396, 397, 402, 405, 409, 410, 411, 414, 417, 419, 421, 424, 426, 427, 429, 432, 436, 438, 441, 442, 445, 446, 447, 454, 457, 460, 462, 464, 466, 468, 470, 472, 474, 476, 477, 479, 481, 483, 485, 487, 489, 491, 493, 495, 496, 499, 501, 505, 507, 509, 511, 513, 515, 517, 519, 521, 523, 525, 527, 529, 531, 533, 535, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 564, 565, 566, 567, 568, 569, 572, 575, 576, 578, 583, 585, 587, 589, 591, 592, 593, 594, 595, 596, 597, 598, 599, 600, 601, 602, 603, 604, 605, 606, 607, 608, 609, 610, 611, 612, 613, 614, 615, 616, 617, 618, 619, 620, 621, 622, 623, 624, 625, 626, 627, 628, 629, 630, 631, 632, 633, 634, 635, 636, 637, 638, 639, 640, 641, 642, 643, 644, 645, 646, 647, 648, 649, 650, 651, 652, 653, 654, 655, 656, 657, 658, 659, 661, 662, 663, 664, 665, 666, 667, 668, 669, 670, 671, 672, 673, 674, 675, 676, 677, 678, 679, 680, 681, 682, 683, 684, 685, 686, 687, 881, 883, 887, 891, 892, 893, 912, 940, 941, 942, 943, 944, 945, 946, 947, 948, 949, 950, 951, 952, 953, 954, 955, 956, 957, 958, 959, 960, 961, 962, 963, 964, 965, 966, 967, 968, 969, 970, 971, 972, 973, 974, 976, 977, 981, 982, 983, 985, 987, 989, 991, 993, 995, 997, 999, 1001, 1003, 1005, 1007, 1008, 1009, 1010, 1011, 1013, 1016, 1019, 1020, 1072, 1073, 1074, 1075, 1076, 1077, 1078, 1079, 1080, 1081, 1082, 1083, 1084, 1085, 1086, 1087, 1088, 1089, 1090, 1091, 1092, 1093, 1094, 1095, 1096, 1097, 1098, 1099, 1100, 1101, 1102, 1103, 1104, 1105, 1106, 1107, 1108, 1109, 1110, 1111, 1112, 1113, 1114, 1115, 1116, 1117, 1118, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1135, 1137, 1139, 1141, 1143, 1145, 1147, 1149, 1151, 1153, 1163, 1165, 1167, 1169, 1171, 1173, 1175, 1177, 1179, 1181, 1183, 1185, 1187, 1189, 1191, 1193, 1195, 1197, 1199, 1201, 1203, 1205, 1207, 1209, 1211, 1213, 1215, 1218, 1220, 1222, 1224, 1226, 1228, 1230, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1269, 1271, 1273, 1275, 1277, 1279, 1281, 1283, 1285, 1287, 1289, 1291, 1293, 1295, 1297, 1299, 1301, 1303, 1305, 1307, 1309, 1311, 1313, 1315, 1317, 1319, 1377, 1378, 1379, 1380, 1381, 1382, 1383, 1384, 1385, 1386, 1387, 1388, 1389, 1390, 1391, 1392, 1393, 1394, 1395, 1396, 1397, 1398, 1399, 1400, 1401, 1402, 1403, 1404, 1405, 1406, 1407, 1408, 1409, 1410, 1411, 1412, 1413, 1414, 1415, 7424, 7425, 7426, 7427, 7428, 7429, 7430, 7431, 7432, 7433, 7434, 7435, 7436, 7437, 7438, 7439, 7440, 7441, 7442, 7443, 7444, 7445, 7446, 7447, 7448, 7449, 7450, 7451, 7452, 7453, 7454, 7455, 7456, 7457, 7458, 7459, 7460, 7461, 7462, 7463, 7464, 7465, 7466, 7467, 7522, 7523, 7524, 7525, 7526, 7527, 7528, 7529, 7530, 7531, 7532, 7533, 7534, 7535, 7536, 7537, 7538, 7539, 7540, 7541, 7542, 7543, 7545, 7546, 7547, 7548, 7549, 7550, 7551, 7552, 7553, 7554, 7555, 7556, 7557, 7558, 7559, 7560, 7561, 7562, 7563, 7564, 7565, 7566, 7567, 7568, 7569, 7570, 7571, 7572, 7573, 7574, 7575, 7576, 7577, 7578, 7681, 7683, 7685, 7687, 7689, 7691, 7693, 7695, 7697, 7699, 7701, 7703, 7705, 7707, 7709, 7711, 7713, 7715, 7717, 7719, 7721, 7723, 7725, 7727, 7729, 7731, 7733, 7735, 7737, 7739, 7741, 7743, 7745, 7747, 7749, 7751, 7753, 7755, 7757, 7759, 7761, 7763, 7765, 7767, 7769, 7771, 7773, 7775, 7777, 7779, 7781, 7783, 7785, 7787, 7789, 7791, 7793, 7795, 7797, 7799, 7801, 7803, 7805, 7807, 7809, 7811, 7813, 7815, 7817, 7819, 7821, 7823, 7825, 7827, 7829, 7830, 7831, 7832, 7833, 7834, 7835, 7836, 7837, 7839, 7841, 7843, 7845, 7847, 7849, 7851, 7853, 7855, 7857, 7859, 7861, 7863, 7865, 7867, 7869, 7871, 7873, 7875, 7877, 7879, 7881, 7883, 7885, 7887, 7889, 7891, 7893, 7895, 7897, 7899, 7901, 7903, 7905, 7907, 7909, 7911, 7913, 7915, 7917, 7919, 7921, 7923, 7925, 7927, 7929, 7931, 7933, 7935, 7936, 7937, 7938, 7939, 7940, 7941, 7942, 7943, 7952, 7953, 7954, 7955, 7956, 7957, 7968, 7969, 7970, 7971, 7972, 7973, 7974, 7975, 7984, 7985, 7986, 7987, 7988, 7989, 7990, 7991, 8000, 8001, 8002, 8003, 8004, 8005, 8016, 8017, 8018, 8019, 8020, 8021, 8022, 8023, 8032, 8033, 8034, 8035, 8036, 8037, 8038, 8039, 8048, 940, 8050, 941, 8052, 942, 8054, 943, 8056, 972, 8058, 973, 8060, 974, 8064, 8065, 8066, 8067, 8068, 8069, 8070, 8071, 8080, 8081, 8082, 8083, 8084, 8085, 8086, 8087, 8096, 8097, 8098, 8099, 8100, 8101, 8102, 8103, 8112, 8113, 8114, 8115, 8116, 8118, 8119, 953, 8130, 8131, 8132, 8134, 8135, 8144, 8145, 8146, 912, 8150, 8151, 8160, 8161, 8162, 944, 8164, 8165, 8166, 8167, 8178, 8179, 8180, 8182, 8183, 8458, 8462, 8463, 8467, 8495, 8500, 8505, 8508, 8509, 8518, 8519, 8520, 8521, 8526, 8580, 11312, 11313, 11314, 11315, 11316, 11317, 11318, 11319, 11320, 11321, 11322, 11323, 11324, 11325, 11326, 11327, 11328, 11329, 11330, 11331, 11332, 11333, 11334, 11335, 11336, 11337, 11338, 11339, 11340, 11341, 11342, 11343, 11344, 11345, 11346, 11347, 11348, 11349, 11350, 11351, 11352, 11353, 11354, 11355, 11356, 11357, 11358, 11361, 11365, 11366, 11368, 11370, 11372, 11377, 11379, 11380, 11382, 11383, 11384, 11385, 11386, 11387, 11388, 11393, 11395, 11397, 11399, 11401, 11403, 11405, 11407, 11409, 11411, 11413, 11415, 11417, 11419, 11421, 11423, 11425, 11427, 11429, 11431, 11433, 11435, 11437, 11439, 11441, 11443, 11445, 11447, 11449, 11451, 11453, 11455, 11457, 11459, 11461, 11463, 11465, 11467, 11469, 11471, 11473, 11475, 11477, 11479, 11481, 11483, 11485, 11487, 11489, 11491, 11492, 11500, 11502, 11520, 11521, 11522, 11523, 11524, 11525, 11526, 11527, 11528, 11529, 11530, 11531, 11532, 11533, 11534, 11535, 11536, 11537, 11538, 11539, 11540, 11541, 11542, 11543, 11544, 11545, 11546, 11547, 11548, 11549, 11550, 11551, 11552, 11553, 11554, 11555, 11556, 11557, 42561, 42563, 42565, 42567, 42569, 42571, 42573, 42575, 42577, 42579, 42581, 42583, 42585, 42587, 42589, 42591, 42593, 42595, 42597, 42599, 42601, 42603, 42605, 42625, 42627, 42629, 42631, 42633, 42635, 42637, 42639, 42641, 42643, 42645, 42647, 42787, 42789, 42791, 42793, 42795, 42797, 42799, 42800, 42801, 42803, 42805, 42807, 42809, 42811, 42813, 42815, 42817, 42819, 42821, 42823, 42825, 42827, 42829, 42831, 42833, 42835, 42837, 42839, 42841, 42843, 42845, 42847, 42849, 42851, 42853, 42855, 42857, 42859, 42861, 42863, 42865, 42866, 42867, 42868, 42869, 42870, 42871, 42872, 42874, 42876, 42879, 42881, 42883, 42885, 42887, 42892, 42894, 42897, 42913, 42915, 42917, 42919, 42921, 43002, 64256, 64257, 64258, 64259, 64260, 64261, 64262, 64275, 64276, 64277, 64278, 64279, 65345, 65346, 65347, 65348, 65349, 65350, 65351, 65352, 65353, 65354, 65355, 65356, 65357, 65358, 65359, 65360, 65361, 65362, 65363, 65364, 65365, 65366, 65367, 65368, 65369, 65370, 66600, 66601, 66602, 66603, 66604, 66605, 66606, 66607, 66608, 66609, 66610, 66611, 66612, 66613, 66614, 66615, 66616, 66617, 66618, 66619, 66620, 66621, 66622, 66623, 66624, 66625, 66626, 66627, 66628, 66629, 66630, 66631, 66632, 66633, 66634, 66635, 66636, 66637, 66638, 66639, 119834, 119835, 119836, 119837, 119838, 119839, 119840, 119841, 119842, 119843, 119844, 119845, 119846, 119847, 119848, 119849, 119850, 119851, 119852, 119853, 119854, 119855, 119856, 119857, 119858, 119859, 119886, 119887, 119888, 119889, 119890, 119891, 119892, 119894, 119895, 119896, 119897, 119898, 119899, 119900, 119901, 119902, 119903, 119904, 119905, 119906, 119907, 119908, 119909, 119910, 119911, 119938, 119939, 119940, 119941, 119942, 119943, 119944, 119945, 119946, 119947, 119948, 119949, 119950, 119951, 119952, 119953, 119954, 119955, 119956, 119957, 119958, 119959, 119960, 119961, 119962, 119963, 119990, 119991, 119992, 119993, 119995, 119997, 119998, 119999, 120000, 120001, 120002, 120003, 120005, 120006, 120007, 120008, 120009, 120010, 120011, 120012, 120013, 120014, 120015, 120042, 120043, 120044, 120045, 120046, 120047, 120048, 120049, 120050, 120051, 120052, 120053, 120054, 120055, 120056, 120057, 120058, 120059, 120060, 120061, 120062, 120063, 120064, 120065, 120066, 120067, 120094, 120095, 120096, 120097, 120098, 120099, 120100, 120101, 120102, 120103, 120104, 120105, 120106, 120107, 120108, 120109, 120110, 120111, 120112, 120113, 120114, 120115, 120116, 120117, 120118, 120119, 120146, 120147, 120148, 120149, 120150, 120151, 120152, 120153, 120154, 120155, 120156, 120157, 120158, 120159, 120160, 120161, 120162, 120163, 120164, 120165, 120166, 120167, 120168, 120169, 120170, 120171, 120198, 120199, 120200, 120201, 120202, 120203, 120204, 120205, 120206, 120207, 120208, 120209, 120210, 120211, 120212, 120213, 120214, 120215, 120216, 120217, 120218, 120219, 120220, 120221, 120222, 120223, 120250, 120251, 120252, 120253, 120254, 120255, 120256, 120257, 120258, 120259, 120260, 120261, 120262, 120263, 120264, 120265, 120266, 120267, 120268, 120269, 120270, 120271, 120272, 120273, 120274, 120275, 120302, 120303, 120304, 120305, 120306, 120307, 120308, 120309, 120310, 120311, 120312, 120313, 120314, 120315, 120316, 120317, 120318, 120319, 120320, 120321, 120322, 120323, 120324, 120325, 120326, 120327, 120354, 120355, 120356, 120357, 120358, 120359, 120360, 120361, 120362, 120363, 120364, 120365, 120366, 120367, 120368, 120369, 120370, 120371, 120372, 120373, 120374, 120375, 120376, 120377, 120378, 120379, 120406, 120407, 120408, 120409, 120410, 120411, 120412, 120413, 120414, 120415, 120416, 120417, 120418, 120419, 120420, 120421, 120422, 120423, 120424, 120425, 120426, 120427, 120428, 120429, 120430, 120431, 120458, 120459, 120460, 120461, 120462, 120463, 120464, 120465, 120466, 120467, 120468, 120469, 120470, 120471, 120472, 120473, 120474, 120475, 120476, 120477, 120478, 120479, 120480, 120481, 120482, 120483, 120484, 120485, 120514, 120515, 120516, 120517, 120518, 120519, 120520, 120521, 120522, 120523, 120524, 120525, 120526, 120527, 120528, 120529, 120530, 120531, 120532, 120533, 120534, 120535, 120536, 120537, 120538, 120540, 120541, 120542, 120543, 120544, 120545, 120572, 120573, 120574, 120575, 120576, 120577, 120578, 120579, 120580, 120581, 120582, 120583, 120584, 120585, 120586, 120587, 120588, 120589, 120590, 120591, 120592, 120593, 120594, 120595, 120596, 120598, 120599, 120600, 120601, 120602, 120603, 120630, 120631, 120632, 120633, 120634, 120635, 120636, 120637, 120638, 120639, 120640, 120641, 120642, 120643, 120644, 120645, 120646, 120647, 120648, 120649, 120650, 120651, 120652, 120653, 120654, 120656, 120657, 120658, 120659, 120660, 120661, 120688, 120689, 120690, 120691, 120692, 120693, 120694, 120695, 120696, 120697, 120698, 120699, 120700, 120701, 120702, 120703, 120704, 120705, 120706, 120707, 120708, 120709, 120710, 120711, 120712, 120714, 120715, 120716, 120717, 120718, 120719, 120746, 120747, 120748, 120749, 120750, 120751, 120752, 120753, 120754, 120755, 120756, 120757, 120758, 120759, 120760, 120761, 120762, 120763, 120764, 120765, 120766, 120767, 120768, 120769, 120770, 120772, 120773, 120774, 120775, 120776, 120777, 120779],
				'encoded': 'abcdefghijklmnopqrstuvwxyz\xAA\xB5\xBA\xDF\xE0\xE1\xE2\xE3\xE4\xE5\xE6\xE7\xE8\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF0\xF1\xF2\xF3\xF4\xF5\xF6\xF8\xF9\xFA\xFB\xFC\xFD\xFE\xFF\u0101\u0103\u0105\u0107\u0109\u010B\u010D\u010F\u0111\u0113\u0115\u0117\u0119\u011B\u011D\u011F\u0121\u0123\u0125\u0127\u0129\u012B\u012D\u012F\u0131\u0133\u0135\u0137\u0138\u013A\u013C\u013E\u0140\u0142\u0144\u0146\u0148\u0149\u014B\u014D\u014F\u0151\u0153\u0155\u0157\u0159\u015B\u015D\u015F\u0161\u0163\u0165\u0167\u0169\u016B\u016D\u016F\u0171\u0173\u0175\u0177\u017A\u017C\u017E\u017F\u0180\u0183\u0185\u0188\u018C\u018D\u0192\u0195\u0199\u019A\u019B\u019E\u01A1\u01A3\u01A5\u01A8\u01AA\u01AB\u01AD\u01B0\u01B4\u01B6\u01B9\u01BA\u01BD\u01BE\u01BF\u01C6\u01C9\u01CC\u01CE\u01D0\u01D2\u01D4\u01D6\u01D8\u01DA\u01DC\u01DD\u01DF\u01E1\u01E3\u01E5\u01E7\u01E9\u01EB\u01ED\u01EF\u01F0\u01F3\u01F5\u01F9\u01FB\u01FD\u01FF\u0201\u0203\u0205\u0207\u0209\u020B\u020D\u020F\u0211\u0213\u0215\u0217\u0219\u021B\u021D\u021F\u0221\u0223\u0225\u0227\u0229\u022B\u022D\u022F\u0231\u0233\u0234\u0235\u0236\u0237\u0238\u0239\u023C\u023F\u0240\u0242\u0247\u0249\u024B\u024D\u024F\u0250\u0251\u0252\u0253\u0254\u0255\u0256\u0257\u0258\u0259\u025A\u025B\u025C\u025D\u025E\u025F\u0260\u0261\u0262\u0263\u0264\u0265\u0266\u0267\u0268\u0269\u026A\u026B\u026C\u026D\u026E\u026F\u0270\u0271\u0272\u0273\u0274\u0275\u0276\u0277\u0278\u0279\u027A\u027B\u027C\u027D\u027E\u027F\u0280\u0281\u0282\u0283\u0284\u0285\u0286\u0287\u0288\u0289\u028A\u028B\u028C\u028D\u028E\u028F\u0290\u0291\u0292\u0293\u0295\u0296\u0297\u0298\u0299\u029A\u029B\u029C\u029D\u029E\u029F\u02A0\u02A1\u02A2\u02A3\u02A4\u02A5\u02A6\u02A7\u02A8\u02A9\u02AA\u02AB\u02AC\u02AD\u02AE\u02AF\u0371\u0373\u0377\u037B\u037C\u037D\u0390\u03AC\u03AD\u03AE\u03AF\u03B0\u03B1\u03B2\u03B3\u03B4\u03B5\u03B6\u03B7\u03B8\u03B9\u03BA\u03BB\u03BC\u03BD\u03BE\u03BF\u03C0\u03C1\u03C2\u03C3\u03C4\u03C5\u03C6\u03C7\u03C8\u03C9\u03CA\u03CB\u03CC\u03CD\u03CE\u03D0\u03D1\u03D5\u03D6\u03D7\u03D9\u03DB\u03DD\u03DF\u03E1\u03E3\u03E5\u03E7\u03E9\u03EB\u03ED\u03EF\u03F0\u03F1\u03F2\u03F3\u03F5\u03F8\u03FB\u03FC\u0430\u0431\u0432\u0433\u0434\u0435\u0436\u0437\u0438\u0439\u043A\u043B\u043C\u043D\u043E\u043F\u0440\u0441\u0442\u0443\u0444\u0445\u0446\u0447\u0448\u0449\u044A\u044B\u044C\u044D\u044E\u044F\u0450\u0451\u0452\u0453\u0454\u0455\u0456\u0457\u0458\u0459\u045A\u045B\u045C\u045D\u045E\u045F\u0461\u0463\u0465\u0467\u0469\u046B\u046D\u046F\u0471\u0473\u0475\u0477\u0479\u047B\u047D\u047F\u0481\u048B\u048D\u048F\u0491\u0493\u0495\u0497\u0499\u049B\u049D\u049F\u04A1\u04A3\u04A5\u04A7\u04A9\u04AB\u04AD\u04AF\u04B1\u04B3\u04B5\u04B7\u04B9\u04BB\u04BD\u04BF\u04C2\u04C4\u04C6\u04C8\u04CA\u04CC\u04CE\u04CF\u04D1\u04D3\u04D5\u04D7\u04D9\u04DB\u04DD\u04DF\u04E1\u04E3\u04E5\u04E7\u04E9\u04EB\u04ED\u04EF\u04F1\u04F3\u04F5\u04F7\u04F9\u04FB\u04FD\u04FF\u0501\u0503\u0505\u0507\u0509\u050B\u050D\u050F\u0511\u0513\u0515\u0517\u0519\u051B\u051D\u051F\u0521\u0523\u0525\u0527\u0561\u0562\u0563\u0564\u0565\u0566\u0567\u0568\u0569\u056A\u056B\u056C\u056D\u056E\u056F\u0570\u0571\u0572\u0573\u0574\u0575\u0576\u0577\u0578\u0579\u057A\u057B\u057C\u057D\u057E\u057F\u0580\u0581\u0582\u0583\u0584\u0585\u0586\u0587\u1D00\u1D01\u1D02\u1D03\u1D04\u1D05\u1D06\u1D07\u1D08\u1D09\u1D0A\u1D0B\u1D0C\u1D0D\u1D0E\u1D0F\u1D10\u1D11\u1D12\u1D13\u1D14\u1D15\u1D16\u1D17\u1D18\u1D19\u1D1A\u1D1B\u1D1C\u1D1D\u1D1E\u1D1F\u1D20\u1D21\u1D22\u1D23\u1D24\u1D25\u1D26\u1D27\u1D28\u1D29\u1D2A\u1D2B\u1D62\u1D63\u1D64\u1D65\u1D66\u1D67\u1D68\u1D69\u1D6A\u1D6B\u1D6C\u1D6D\u1D6E\u1D6F\u1D70\u1D71\u1D72\u1D73\u1D74\u1D75\u1D76\u1D77\u1D79\u1D7A\u1D7B\u1D7C\u1D7D\u1D7E\u1D7F\u1D80\u1D81\u1D82\u1D83\u1D84\u1D85\u1D86\u1D87\u1D88\u1D89\u1D8A\u1D8B\u1D8C\u1D8D\u1D8E\u1D8F\u1D90\u1D91\u1D92\u1D93\u1D94\u1D95\u1D96\u1D97\u1D98\u1D99\u1D9A\u1E01\u1E03\u1E05\u1E07\u1E09\u1E0B\u1E0D\u1E0F\u1E11\u1E13\u1E15\u1E17\u1E19\u1E1B\u1E1D\u1E1F\u1E21\u1E23\u1E25\u1E27\u1E29\u1E2B\u1E2D\u1E2F\u1E31\u1E33\u1E35\u1E37\u1E39\u1E3B\u1E3D\u1E3F\u1E41\u1E43\u1E45\u1E47\u1E49\u1E4B\u1E4D\u1E4F\u1E51\u1E53\u1E55\u1E57\u1E59\u1E5B\u1E5D\u1E5F\u1E61\u1E63\u1E65\u1E67\u1E69\u1E6B\u1E6D\u1E6F\u1E71\u1E73\u1E75\u1E77\u1E79\u1E7B\u1E7D\u1E7F\u1E81\u1E83\u1E85\u1E87\u1E89\u1E8B\u1E8D\u1E8F\u1E91\u1E93\u1E95\u1E96\u1E97\u1E98\u1E99\u1E9A\u1E9B\u1E9C\u1E9D\u1E9F\u1EA1\u1EA3\u1EA5\u1EA7\u1EA9\u1EAB\u1EAD\u1EAF\u1EB1\u1EB3\u1EB5\u1EB7\u1EB9\u1EBB\u1EBD\u1EBF\u1EC1\u1EC3\u1EC5\u1EC7\u1EC9\u1ECB\u1ECD\u1ECF\u1ED1\u1ED3\u1ED5\u1ED7\u1ED9\u1EDB\u1EDD\u1EDF\u1EE1\u1EE3\u1EE5\u1EE7\u1EE9\u1EEB\u1EED\u1EEF\u1EF1\u1EF3\u1EF5\u1EF7\u1EF9\u1EFB\u1EFD\u1EFF\u1F00\u1F01\u1F02\u1F03\u1F04\u1F05\u1F06\u1F07\u1F10\u1F11\u1F12\u1F13\u1F14\u1F15\u1F20\u1F21\u1F22\u1F23\u1F24\u1F25\u1F26\u1F27\u1F30\u1F31\u1F32\u1F33\u1F34\u1F35\u1F36\u1F37\u1F40\u1F41\u1F42\u1F43\u1F44\u1F45\u1F50\u1F51\u1F52\u1F53\u1F54\u1F55\u1F56\u1F57\u1F60\u1F61\u1F62\u1F63\u1F64\u1F65\u1F66\u1F67\u1F70\u03AC\u1F72\u03AD\u1F74\u03AE\u1F76\u03AF\u1F78\u03CC\u1F7A\u03CD\u1F7C\u03CE\u1F80\u1F81\u1F82\u1F83\u1F84\u1F85\u1F86\u1F87\u1F90\u1F91\u1F92\u1F93\u1F94\u1F95\u1F96\u1F97\u1FA0\u1FA1\u1FA2\u1FA3\u1FA4\u1FA5\u1FA6\u1FA7\u1FB0\u1FB1\u1FB2\u1FB3\u1FB4\u1FB6\u1FB7\u03B9\u1FC2\u1FC3\u1FC4\u1FC6\u1FC7\u1FD0\u1FD1\u1FD2\u0390\u1FD6\u1FD7\u1FE0\u1FE1\u1FE2\u03B0\u1FE4\u1FE5\u1FE6\u1FE7\u1FF2\u1FF3\u1FF4\u1FF6\u1FF7\u210A\u210E\u210F\u2113\u212F\u2134\u2139\u213C\u213D\u2146\u2147\u2148\u2149\u214E\u2184\u2C30\u2C31\u2C32\u2C33\u2C34\u2C35\u2C36\u2C37\u2C38\u2C39\u2C3A\u2C3B\u2C3C\u2C3D\u2C3E\u2C3F\u2C40\u2C41\u2C42\u2C43\u2C44\u2C45\u2C46\u2C47\u2C48\u2C49\u2C4A\u2C4B\u2C4C\u2C4D\u2C4E\u2C4F\u2C50\u2C51\u2C52\u2C53\u2C54\u2C55\u2C56\u2C57\u2C58\u2C59\u2C5A\u2C5B\u2C5C\u2C5D\u2C5E\u2C61\u2C65\u2C66\u2C68\u2C6A\u2C6C\u2C71\u2C73\u2C74\u2C76\u2C77\u2C78\u2C79\u2C7A\u2C7B\u2C7C\u2C81\u2C83\u2C85\u2C87\u2C89\u2C8B\u2C8D\u2C8F\u2C91\u2C93\u2C95\u2C97\u2C99\u2C9B\u2C9D\u2C9F\u2CA1\u2CA3\u2CA5\u2CA7\u2CA9\u2CAB\u2CAD\u2CAF\u2CB1\u2CB3\u2CB5\u2CB7\u2CB9\u2CBB\u2CBD\u2CBF\u2CC1\u2CC3\u2CC5\u2CC7\u2CC9\u2CCB\u2CCD\u2CCF\u2CD1\u2CD3\u2CD5\u2CD7\u2CD9\u2CDB\u2CDD\u2CDF\u2CE1\u2CE3\u2CE4\u2CEC\u2CEE\u2D00\u2D01\u2D02\u2D03\u2D04\u2D05\u2D06\u2D07\u2D08\u2D09\u2D0A\u2D0B\u2D0C\u2D0D\u2D0E\u2D0F\u2D10\u2D11\u2D12\u2D13\u2D14\u2D15\u2D16\u2D17\u2D18\u2D19\u2D1A\u2D1B\u2D1C\u2D1D\u2D1E\u2D1F\u2D20\u2D21\u2D22\u2D23\u2D24\u2D25\uA641\uA643\uA645\uA647\uA649\uA64B\uA64D\uA64F\uA651\uA653\uA655\uA657\uA659\uA65B\uA65D\uA65F\uA661\uA663\uA665\uA667\uA669\uA66B\uA66D\uA681\uA683\uA685\uA687\uA689\uA68B\uA68D\uA68F\uA691\uA693\uA695\uA697\uA723\uA725\uA727\uA729\uA72B\uA72D\uA72F\uA730\uA731\uA733\uA735\uA737\uA739\uA73B\uA73D\uA73F\uA741\uA743\uA745\uA747\uA749\uA74B\uA74D\uA74F\uA751\uA753\uA755\uA757\uA759\uA75B\uA75D\uA75F\uA761\uA763\uA765\uA767\uA769\uA76B\uA76D\uA76F\uA771\uA772\uA773\uA774\uA775\uA776\uA777\uA778\uA77A\uA77C\uA77F\uA781\uA783\uA785\uA787\uA78C\uA78E\uA791\uA7A1\uA7A3\uA7A5\uA7A7\uA7A9\uA7FA\uFB00\uFB01\uFB02\uFB03\uFB04\uFB05\uFB06\uFB13\uFB14\uFB15\uFB16\uFB17\uFF41\uFF42\uFF43\uFF44\uFF45\uFF46\uFF47\uFF48\uFF49\uFF4A\uFF4B\uFF4C\uFF4D\uFF4E\uFF4F\uFF50\uFF51\uFF52\uFF53\uFF54\uFF55\uFF56\uFF57\uFF58\uFF59\uFF5A\uD801\uDC28\uD801\uDC29\uD801\uDC2A\uD801\uDC2B\uD801\uDC2C\uD801\uDC2D\uD801\uDC2E\uD801\uDC2F\uD801\uDC30\uD801\uDC31\uD801\uDC32\uD801\uDC33\uD801\uDC34\uD801\uDC35\uD801\uDC36\uD801\uDC37\uD801\uDC38\uD801\uDC39\uD801\uDC3A\uD801\uDC3B\uD801\uDC3C\uD801\uDC3D\uD801\uDC3E\uD801\uDC3F\uD801\uDC40\uD801\uDC41\uD801\uDC42\uD801\uDC43\uD801\uDC44\uD801\uDC45\uD801\uDC46\uD801\uDC47\uD801\uDC48\uD801\uDC49\uD801\uDC4A\uD801\uDC4B\uD801\uDC4C\uD801\uDC4D\uD801\uDC4E\uD801\uDC4F\uD835\uDC1A\uD835\uDC1B\uD835\uDC1C\uD835\uDC1D\uD835\uDC1E\uD835\uDC1F\uD835\uDC20\uD835\uDC21\uD835\uDC22\uD835\uDC23\uD835\uDC24\uD835\uDC25\uD835\uDC26\uD835\uDC27\uD835\uDC28\uD835\uDC29\uD835\uDC2A\uD835\uDC2B\uD835\uDC2C\uD835\uDC2D\uD835\uDC2E\uD835\uDC2F\uD835\uDC30\uD835\uDC31\uD835\uDC32\uD835\uDC33\uD835\uDC4E\uD835\uDC4F\uD835\uDC50\uD835\uDC51\uD835\uDC52\uD835\uDC53\uD835\uDC54\uD835\uDC56\uD835\uDC57\uD835\uDC58\uD835\uDC59\uD835\uDC5A\uD835\uDC5B\uD835\uDC5C\uD835\uDC5D\uD835\uDC5E\uD835\uDC5F\uD835\uDC60\uD835\uDC61\uD835\uDC62\uD835\uDC63\uD835\uDC64\uD835\uDC65\uD835\uDC66\uD835\uDC67\uD835\uDC82\uD835\uDC83\uD835\uDC84\uD835\uDC85\uD835\uDC86\uD835\uDC87\uD835\uDC88\uD835\uDC89\uD835\uDC8A\uD835\uDC8B\uD835\uDC8C\uD835\uDC8D\uD835\uDC8E\uD835\uDC8F\uD835\uDC90\uD835\uDC91\uD835\uDC92\uD835\uDC93\uD835\uDC94\uD835\uDC95\uD835\uDC96\uD835\uDC97\uD835\uDC98\uD835\uDC99\uD835\uDC9A\uD835\uDC9B\uD835\uDCB6\uD835\uDCB7\uD835\uDCB8\uD835\uDCB9\uD835\uDCBB\uD835\uDCBD\uD835\uDCBE\uD835\uDCBF\uD835\uDCC0\uD835\uDCC1\uD835\uDCC2\uD835\uDCC3\uD835\uDCC5\uD835\uDCC6\uD835\uDCC7\uD835\uDCC8\uD835\uDCC9\uD835\uDCCA\uD835\uDCCB\uD835\uDCCC\uD835\uDCCD\uD835\uDCCE\uD835\uDCCF\uD835\uDCEA\uD835\uDCEB\uD835\uDCEC\uD835\uDCED\uD835\uDCEE\uD835\uDCEF\uD835\uDCF0\uD835\uDCF1\uD835\uDCF2\uD835\uDCF3\uD835\uDCF4\uD835\uDCF5\uD835\uDCF6\uD835\uDCF7\uD835\uDCF8\uD835\uDCF9\uD835\uDCFA\uD835\uDCFB\uD835\uDCFC\uD835\uDCFD\uD835\uDCFE\uD835\uDCFF\uD835\uDD00\uD835\uDD01\uD835\uDD02\uD835\uDD03\uD835\uDD1E\uD835\uDD1F\uD835\uDD20\uD835\uDD21\uD835\uDD22\uD835\uDD23\uD835\uDD24\uD835\uDD25\uD835\uDD26\uD835\uDD27\uD835\uDD28\uD835\uDD29\uD835\uDD2A\uD835\uDD2B\uD835\uDD2C\uD835\uDD2D\uD835\uDD2E\uD835\uDD2F\uD835\uDD30\uD835\uDD31\uD835\uDD32\uD835\uDD33\uD835\uDD34\uD835\uDD35\uD835\uDD36\uD835\uDD37\uD835\uDD52\uD835\uDD53\uD835\uDD54\uD835\uDD55\uD835\uDD56\uD835\uDD57\uD835\uDD58\uD835\uDD59\uD835\uDD5A\uD835\uDD5B\uD835\uDD5C\uD835\uDD5D\uD835\uDD5E\uD835\uDD5F\uD835\uDD60\uD835\uDD61\uD835\uDD62\uD835\uDD63\uD835\uDD64\uD835\uDD65\uD835\uDD66\uD835\uDD67\uD835\uDD68\uD835\uDD69\uD835\uDD6A\uD835\uDD6B\uD835\uDD86\uD835\uDD87\uD835\uDD88\uD835\uDD89\uD835\uDD8A\uD835\uDD8B\uD835\uDD8C\uD835\uDD8D\uD835\uDD8E\uD835\uDD8F\uD835\uDD90\uD835\uDD91\uD835\uDD92\uD835\uDD93\uD835\uDD94\uD835\uDD95\uD835\uDD96\uD835\uDD97\uD835\uDD98\uD835\uDD99\uD835\uDD9A\uD835\uDD9B\uD835\uDD9C\uD835\uDD9D\uD835\uDD9E\uD835\uDD9F\uD835\uDDBA\uD835\uDDBB\uD835\uDDBC\uD835\uDDBD\uD835\uDDBE\uD835\uDDBF\uD835\uDDC0\uD835\uDDC1\uD835\uDDC2\uD835\uDDC3\uD835\uDDC4\uD835\uDDC5\uD835\uDDC6\uD835\uDDC7\uD835\uDDC8\uD835\uDDC9\uD835\uDDCA\uD835\uDDCB\uD835\uDDCC\uD835\uDDCD\uD835\uDDCE\uD835\uDDCF\uD835\uDDD0\uD835\uDDD1\uD835\uDDD2\uD835\uDDD3\uD835\uDDEE\uD835\uDDEF\uD835\uDDF0\uD835\uDDF1\uD835\uDDF2\uD835\uDDF3\uD835\uDDF4\uD835\uDDF5\uD835\uDDF6\uD835\uDDF7\uD835\uDDF8\uD835\uDDF9\uD835\uDDFA\uD835\uDDFB\uD835\uDDFC\uD835\uDDFD\uD835\uDDFE\uD835\uDDFF\uD835\uDE00\uD835\uDE01\uD835\uDE02\uD835\uDE03\uD835\uDE04\uD835\uDE05\uD835\uDE06\uD835\uDE07\uD835\uDE22\uD835\uDE23\uD835\uDE24\uD835\uDE25\uD835\uDE26\uD835\uDE27\uD835\uDE28\uD835\uDE29\uD835\uDE2A\uD835\uDE2B\uD835\uDE2C\uD835\uDE2D\uD835\uDE2E\uD835\uDE2F\uD835\uDE30\uD835\uDE31\uD835\uDE32\uD835\uDE33\uD835\uDE34\uD835\uDE35\uD835\uDE36\uD835\uDE37\uD835\uDE38\uD835\uDE39\uD835\uDE3A\uD835\uDE3B\uD835\uDE56\uD835\uDE57\uD835\uDE58\uD835\uDE59\uD835\uDE5A\uD835\uDE5B\uD835\uDE5C\uD835\uDE5D\uD835\uDE5E\uD835\uDE5F\uD835\uDE60\uD835\uDE61\uD835\uDE62\uD835\uDE63\uD835\uDE64\uD835\uDE65\uD835\uDE66\uD835\uDE67\uD835\uDE68\uD835\uDE69\uD835\uDE6A\uD835\uDE6B\uD835\uDE6C\uD835\uDE6D\uD835\uDE6E\uD835\uDE6F\uD835\uDE8A\uD835\uDE8B\uD835\uDE8C\uD835\uDE8D\uD835\uDE8E\uD835\uDE8F\uD835\uDE90\uD835\uDE91\uD835\uDE92\uD835\uDE93\uD835\uDE94\uD835\uDE95\uD835\uDE96\uD835\uDE97\uD835\uDE98\uD835\uDE99\uD835\uDE9A\uD835\uDE9B\uD835\uDE9C\uD835\uDE9D\uD835\uDE9E\uD835\uDE9F\uD835\uDEA0\uD835\uDEA1\uD835\uDEA2\uD835\uDEA3\uD835\uDEA4\uD835\uDEA5\uD835\uDEC2\uD835\uDEC3\uD835\uDEC4\uD835\uDEC5\uD835\uDEC6\uD835\uDEC7\uD835\uDEC8\uD835\uDEC9\uD835\uDECA\uD835\uDECB\uD835\uDECC\uD835\uDECD\uD835\uDECE\uD835\uDECF\uD835\uDED0\uD835\uDED1\uD835\uDED2\uD835\uDED3\uD835\uDED4\uD835\uDED5\uD835\uDED6\uD835\uDED7\uD835\uDED8\uD835\uDED9\uD835\uDEDA\uD835\uDEDC\uD835\uDEDD\uD835\uDEDE\uD835\uDEDF\uD835\uDEE0\uD835\uDEE1\uD835\uDEFC\uD835\uDEFD\uD835\uDEFE\uD835\uDEFF\uD835\uDF00\uD835\uDF01\uD835\uDF02\uD835\uDF03\uD835\uDF04\uD835\uDF05\uD835\uDF06\uD835\uDF07\uD835\uDF08\uD835\uDF09\uD835\uDF0A\uD835\uDF0B\uD835\uDF0C\uD835\uDF0D\uD835\uDF0E\uD835\uDF0F\uD835\uDF10\uD835\uDF11\uD835\uDF12\uD835\uDF13\uD835\uDF14\uD835\uDF16\uD835\uDF17\uD835\uDF18\uD835\uDF19\uD835\uDF1A\uD835\uDF1B\uD835\uDF36\uD835\uDF37\uD835\uDF38\uD835\uDF39\uD835\uDF3A\uD835\uDF3B\uD835\uDF3C\uD835\uDF3D\uD835\uDF3E\uD835\uDF3F\uD835\uDF40\uD835\uDF41\uD835\uDF42\uD835\uDF43\uD835\uDF44\uD835\uDF45\uD835\uDF46\uD835\uDF47\uD835\uDF48\uD835\uDF49\uD835\uDF4A\uD835\uDF4B\uD835\uDF4C\uD835\uDF4D\uD835\uDF4E\uD835\uDF50\uD835\uDF51\uD835\uDF52\uD835\uDF53\uD835\uDF54\uD835\uDF55\uD835\uDF70\uD835\uDF71\uD835\uDF72\uD835\uDF73\uD835\uDF74\uD835\uDF75\uD835\uDF76\uD835\uDF77\uD835\uDF78\uD835\uDF79\uD835\uDF7A\uD835\uDF7B\uD835\uDF7C\uD835\uDF7D\uD835\uDF7E\uD835\uDF7F\uD835\uDF80\uD835\uDF81\uD835\uDF82\uD835\uDF83\uD835\uDF84\uD835\uDF85\uD835\uDF86\uD835\uDF87\uD835\uDF88\uD835\uDF8A\uD835\uDF8B\uD835\uDF8C\uD835\uDF8D\uD835\uDF8E\uD835\uDF8F\uD835\uDFAA\uD835\uDFAB\uD835\uDFAC\uD835\uDFAD\uD835\uDFAE\uD835\uDFAF\uD835\uDFB0\uD835\uDFB1\uD835\uDFB2\uD835\uDFB3\uD835\uDFB4\uD835\uDFB5\uD835\uDFB6\uD835\uDFB7\uD835\uDFB8\uD835\uDFB9\uD835\uDFBA\uD835\uDFBB\uD835\uDFBC\uD835\uDFBD\uD835\uDFBE\uD835\uDFBF\uD835\uDFC0\uD835\uDFC1\uD835\uDFC2\uD835\uDFC4\uD835\uDFC5\uD835\uDFC6\uD835\uDFC7\uD835\uDFC8\uD835\uDFC9\uD835\uDFCB'
			},
			{
				'description': 'Unicode character category [Lm]: letter, modifier',
				'decoded': [688, 689, 690, 691, 692, 693, 694, 695, 696, 697, 698, 699, 700, 701, 702, 703, 704, 705, 710, 711, 712, 713, 714, 715, 716, 717, 718, 719, 720, 721, 736, 737, 738, 739, 740, 748, 750, 884, 890, 1369, 1600, 1765, 1766, 2036, 2037, 2042, 2074, 2084, 2088, 2417, 3654, 3782, 4348, 6103, 6211, 6823, 7288, 7289, 7290, 7291, 7292, 7293, 7468, 7469, 7470, 7471, 7472, 7473, 7474, 7475, 7476, 7477, 7478, 7479, 7480, 7481, 7482, 7483, 7484, 7485, 7486, 7487, 7488, 7489, 7490, 7491, 7492, 7493, 7494, 7495, 7496, 7497, 7498, 7499, 7500, 7501, 7502, 7503, 7504, 7505, 7506, 7507, 7508, 7509, 7510, 7511, 7512, 7513, 7514, 7515, 7516, 7517, 7518, 7519, 7520, 7521, 7544, 7579, 7580, 7581, 7582, 7583, 7584, 7585, 7586, 7587, 7588, 7589, 7590, 7591, 7592, 7593, 7594, 7595, 7596, 7597, 7598, 7599, 7600, 7601, 7602, 7603, 7604, 7605, 7606, 7607, 7608, 7609, 7610, 7611, 7612, 7613, 7614, 7615, 8305, 8319, 8336, 8337, 8338, 8339, 8340, 8341, 8342, 8343, 8344, 8345, 8346, 8347, 8348, 11389, 11631, 11823, 12293, 12337, 12338, 12339, 12340, 12341, 12347, 12445, 12446, 12540, 12541, 12542, 40981, 42232, 42233, 42234, 42235, 42236, 42237, 42508, 42623, 42775, 42776, 42777, 42778, 42779, 42780, 42781, 42782, 42783, 42864, 42888, 43471, 43632, 43741, 65392, 65438, 65439],
				'encoded': '\u02B0\u02B1\u02B2\u02B3\u02B4\u02B5\u02B6\u02B7\u02B8\u02B9\u02BA\u02BB\u02BC\u02BD\u02BE\u02BF\u02C0\u02C1\u02C6\u02C7\u02C8\u02C9\u02CA\u02CB\u02CC\u02CD\u02CE\u02CF\u02D0\u02D1\u02E0\u02E1\u02E2\u02E3\u02E4\u02EC\u02EE\u0374\u037A\u0559\u0640\u06E5\u06E6\u07F4\u07F5\u07FA\u081A\u0824\u0828\u0971\u0E46\u0EC6\u10FC\u17D7\u1843\u1AA7\u1C78\u1C79\u1C7A\u1C7B\u1C7C\u1C7D\u1D2C\u1D2D\u1D2E\u1D2F\u1D30\u1D31\u1D32\u1D33\u1D34\u1D35\u1D36\u1D37\u1D38\u1D39\u1D3A\u1D3B\u1D3C\u1D3D\u1D3E\u1D3F\u1D40\u1D41\u1D42\u1D43\u1D44\u1D45\u1D46\u1D47\u1D48\u1D49\u1D4A\u1D4B\u1D4C\u1D4D\u1D4E\u1D4F\u1D50\u1D51\u1D52\u1D53\u1D54\u1D55\u1D56\u1D57\u1D58\u1D59\u1D5A\u1D5B\u1D5C\u1D5D\u1D5E\u1D5F\u1D60\u1D61\u1D78\u1D9B\u1D9C\u1D9D\u1D9E\u1D9F\u1DA0\u1DA1\u1DA2\u1DA3\u1DA4\u1DA5\u1DA6\u1DA7\u1DA8\u1DA9\u1DAA\u1DAB\u1DAC\u1DAD\u1DAE\u1DAF\u1DB0\u1DB1\u1DB2\u1DB3\u1DB4\u1DB5\u1DB6\u1DB7\u1DB8\u1DB9\u1DBA\u1DBB\u1DBC\u1DBD\u1DBE\u1DBF\u2071\u207F\u2090\u2091\u2092\u2093\u2094\u2095\u2096\u2097\u2098\u2099\u209A\u209B\u209C\u2C7D\u2D6F\u2E2F\u3005\u3031\u3032\u3033\u3034\u3035\u303B\u309D\u309E\u30FC\u30FD\u30FE\uA015\uA4F8\uA4F9\uA4FA\uA4FB\uA4FC\uA4FD\uA60C\uA67F\uA717\uA718\uA719\uA71A\uA71B\uA71C\uA71D\uA71E\uA71F\uA770\uA788\uA9CF\uAA70\uAADD\uFF70\uFF9E\uFF9F'
			},
			{
				'description': 'Unicode character category [Lo]: letter, other',
				'decoded': [443, 448, 449, 450, 451, 660, 1488, 1489, 1490, 1491, 1492, 1493, 1494, 1495, 1496, 1497, 1498, 1499, 1500, 1501, 1502, 1503, 1504, 1505, 1506, 1507, 1508, 1509, 1510, 1511, 1512, 1513, 1514, 1520, 1521, 1522, 1568, 1569, 1570, 1571, 1572, 1573, 1574, 1575, 1576, 1577, 1578, 1579, 1580, 1581, 1582, 1583, 1584, 1585, 1586, 1587, 1588, 1589, 1590, 1591, 1592, 1593, 1594, 1595, 1596, 1597, 1598, 1599, 1601, 1602, 1603, 1604, 1605, 1606, 1607, 1608, 1609, 1610, 1646, 1647, 1649, 1650, 1651, 1652, 1653, 1654, 1655, 1656, 1657, 1658, 1659, 1660, 1661, 1662, 1663, 1664, 1665, 1666, 1667, 1668, 1669, 1670, 1671, 1672, 1673, 1674, 1675, 1676, 1677, 1678, 1679, 1680, 1681, 1682, 1683, 1684, 1685, 1686, 1687, 1688, 1689, 1690, 1691, 1692, 1693, 1694, 1695, 1696, 1697, 1698, 1699, 1700, 1701, 1702, 1703, 1704, 1705, 1706, 1707, 1708, 1709, 1710, 1711, 1712, 1713, 1714, 1715, 1716, 1717, 1718, 1719, 1720, 1721, 1722, 1723, 1724, 1725, 1726, 1727, 1728, 1729, 1730, 1731, 1732, 1733, 1734, 1735, 1736, 1737, 1738, 1739, 1740, 1741, 1742, 1743, 1744, 1745, 1746, 1747, 1749, 1774, 1775, 1786, 1787, 1788, 1791, 1808, 1810, 1811, 1812, 1813, 1814, 1815, 1816, 1817, 1818, 1819, 1820, 1821, 1822, 1823, 1824, 1825, 1826, 1827, 1828, 1829, 1830, 1831, 1832, 1833, 1834, 1835, 1836, 1837, 1838, 1839, 1869, 1870, 1871, 1872, 1873, 1874, 1875, 1876, 1877, 1878, 1879, 1880, 1881, 1882, 1883, 1884, 1885, 1886, 1887, 1888, 1889, 1890, 1891, 1892, 1893, 1894, 1895, 1896, 1897, 1898, 1899, 1900, 1901, 1902, 1903, 1904, 1905, 1906, 1907, 1908, 1909, 1910, 1911, 1912, 1913, 1914, 1915, 1916, 1917, 1918, 1919, 1920, 1921, 1922, 1923, 1924, 1925, 1926, 1927, 1928, 1929, 1930, 1931, 1932, 1933, 1934, 1935, 1936, 1937, 1938, 1939, 1940, 1941, 1942, 1943, 1944, 1945, 1946, 1947, 1948, 1949, 1950, 1951, 1952, 1953, 1954, 1955, 1956, 1957, 1969, 1994, 1995, 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024, 2025, 2026, 2048, 2049, 2050, 2051, 2052, 2053, 2054, 2055, 2056, 2057, 2058, 2059, 2060, 2061, 2062, 2063, 2064, 2065, 2066, 2067, 2068, 2069, 2112, 2113, 2114, 2115, 2116, 2117, 2118, 2119, 2120, 2121, 2122, 2123, 2124, 2125, 2126, 2127, 2128, 2129, 2130, 2131, 2132, 2133, 2134, 2135, 2136, 2308, 2309, 2310, 2311, 2312, 2313, 2314, 2315, 2316, 2317, 2318, 2319, 2320, 2321, 2322, 2323, 2324, 2325, 2326, 2327, 2328, 2329, 2330, 2331, 2332, 2333, 2334, 2335, 2336, 2337, 2338, 2339, 2340, 2341, 2342, 2343, 2344, 2345, 2346, 2347, 2348, 2349, 2350, 2351, 2352, 2353, 2354, 2355, 2356, 2357, 2358, 2359, 2360, 2361, 2365, 2384, 2325, 2364, 2326, 2364, 2327, 2364, 2332, 2364, 2337, 2364, 2338, 2364, 2347, 2364, 2351, 2364, 2400, 2401, 2418, 2419, 2420, 2421, 2422, 2423, 2425, 2426, 2427, 2428, 2429, 2430, 2431, 2437, 2438, 2439, 2440, 2441, 2442, 2443, 2444, 2447, 2448, 2451, 2452, 2453, 2454, 2455, 2456, 2457, 2458, 2459, 2460, 2461, 2462, 2463, 2464, 2465, 2466, 2467, 2468, 2469, 2470, 2471, 2472, 2474, 2475, 2476, 2477, 2478, 2479, 2480, 2482, 2486, 2487, 2488, 2489, 2493, 2510, 2465, 2492, 2466, 2492, 2479, 2492, 2528, 2529, 2544, 2545, 2565, 2566, 2567, 2568, 2569, 2570, 2575, 2576, 2579, 2580, 2581, 2582, 2583, 2584, 2585, 2586, 2587, 2588, 2589, 2590, 2591, 2592, 2593, 2594, 2595, 2596, 2597, 2598, 2599, 2600, 2602, 2603, 2604, 2605, 2606, 2607, 2608, 2610, 2610, 2620, 2613, 2616, 2620, 2616, 2617, 2582, 2620, 2583, 2620, 2588, 2620, 2652, 2603, 2620, 2674, 2675, 2676, 2693, 2694, 2695, 2696, 2697, 2698, 2699, 2700, 2701, 2703, 2704, 2705, 2707, 2708, 2709, 2710, 2711, 2712, 2713, 2714, 2715, 2716, 2717, 2718, 2719, 2720, 2721, 2722, 2723, 2724, 2725, 2726, 2727, 2728, 2730, 2731, 2732, 2733, 2734, 2735, 2736, 2738, 2739, 2741, 2742, 2743, 2744, 2745, 2749, 2768, 2784, 2785, 2821, 2822, 2823, 2824, 2825, 2826, 2827, 2828, 2831, 2832, 2835, 2836, 2837, 2838, 2839, 2840, 2841, 2842, 2843, 2844, 2845, 2846, 2847, 2848, 2849, 2850, 2851, 2852, 2853, 2854, 2855, 2856, 2858, 2859, 2860, 2861, 2862, 2863, 2864, 2866, 2867, 2869, 2870, 2871, 2872, 2873, 2877, 2849, 2876, 2850, 2876, 2911, 2912, 2913, 2929, 2947, 2949, 2950, 2951, 2952, 2953, 2954, 2958, 2959, 2960, 2962, 2963, 2964, 2965, 2969, 2970, 2972, 2974, 2975, 2979, 2980, 2984, 2985, 2986, 2990, 2991, 2992, 2993, 2994, 2995, 2996, 2997, 2998, 2999, 3000, 3001, 3024, 3077, 3078, 3079, 3080, 3081, 3082, 3083, 3084, 3086, 3087, 3088, 3090, 3091, 3092, 3093, 3094, 3095, 3096, 3097, 3098, 3099, 3100, 3101, 3102, 3103, 3104, 3105, 3106, 3107, 3108, 3109, 3110, 3111, 3112, 3114, 3115, 3116, 3117, 3118, 3119, 3120, 3121, 3122, 3123, 3125, 3126, 3127, 3128, 3129, 3133, 3160, 3161, 3168, 3169, 3205, 3206, 3207, 3208, 3209, 3210, 3211, 3212, 3214, 3215, 3216, 3218, 3219, 3220, 3221, 3222, 3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230, 3231, 3232, 3233, 3234, 3235, 3236, 3237, 3238, 3239, 3240, 3242, 3243, 3244, 3245, 3246, 3247, 3248, 3249, 3250, 3251, 3253, 3254, 3255, 3256, 3257, 3261, 3294, 3296, 3297, 3313, 3314, 3333, 3334, 3335, 3336, 3337, 3338, 3339, 3340, 3342, 3343, 3344, 3346, 3347, 3348, 3349, 3350, 3351, 3352, 3353, 3354, 3355, 3356, 3357, 3358, 3359, 3360, 3361, 3362, 3363, 3364, 3365, 3366, 3367, 3368, 3369, 3370, 3371, 3372, 3373, 3374, 3375, 3376, 3377, 3378, 3379, 3380, 3381, 3382, 3383, 3384, 3385, 3386, 3389, 3406, 3424, 3425, 3450, 3451, 3452, 3453, 3454, 3455, 3461, 3462, 3463, 3464, 3465, 3466, 3467, 3468, 3469, 3470, 3471, 3472, 3473, 3474, 3475, 3476, 3477, 3478, 3482, 3483, 3484, 3485, 3486, 3487, 3488, 3489, 3490, 3491, 3492, 3493, 3494, 3495, 3496, 3497, 3498, 3499, 3500, 3501, 3502, 3503, 3504, 3505, 3507, 3508, 3509, 3510, 3511, 3512, 3513, 3514, 3515, 3517, 3520, 3521, 3522, 3523, 3524, 3525, 3526, 3585, 3586, 3587, 3588, 3589, 3590, 3591, 3592, 3593, 3594, 3595, 3596, 3597, 3598, 3599, 3600, 3601, 3602, 3603, 3604, 3605, 3606, 3607, 3608, 3609, 3610, 3611, 3612, 3613, 3614, 3615, 3616, 3617, 3618, 3619, 3620, 3621, 3622, 3623, 3624, 3625, 3626, 3627, 3628, 3629, 3630, 3631, 3632, 3634, 3635, 3648, 3649, 3650, 3651, 3652, 3653, 3713, 3714, 3716, 3719, 3720, 3722, 3725, 3732, 3733, 3734, 3735, 3737, 3738, 3739, 3740, 3741, 3742, 3743, 3745, 3746, 3747, 3749, 3751, 3754, 3755, 3757, 3758, 3759, 3760, 3762, 3763, 3773, 3776, 3777, 3778, 3779, 3780, 3804, 3805, 3840, 3904, 3905, 3906, 3906, 4023, 3908, 3909, 3910, 3911, 3913, 3914, 3915, 3916, 3916, 4023, 3918, 3919, 3920, 3921, 3921, 4023, 3923, 3924, 3925, 3926, 3926, 4023, 3928, 3929, 3930, 3931, 3931, 4023, 3933, 3934, 3935, 3936, 3937, 3938, 3939, 3940, 3941, 3942, 3943, 3944, 3904, 4021, 3946, 3947, 3948, 3976, 3977, 3978, 3979, 3980, 4096, 4097, 4098, 4099, 4100, 4101, 4102, 4103, 4104, 4105, 4106, 4107, 4108, 4109, 4110, 4111, 4112, 4113, 4114, 4115, 4116, 4117, 4118, 4119, 4120, 4121, 4122, 4123, 4124, 4125, 4126, 4127, 4128, 4129, 4130, 4131, 4132, 4133, 4134, 4135, 4136, 4137, 4138, 4159, 4176, 4177, 4178, 4179, 4180, 4181, 4186, 4187, 4188, 4189, 4193, 4197, 4198, 4206, 4207, 4208, 4213, 4214, 4215, 4216, 4217, 4218, 4219, 4220, 4221, 4222, 4223, 4224, 4225, 4238, 4304, 4305, 4306, 4307, 4308, 4309, 4310, 4311, 4312, 4313, 4314, 4315, 4316, 4317, 4318, 4319, 4320, 4321, 4322, 4323, 4324, 4325, 4326, 4327, 4328, 4329, 4330, 4331, 4332, 4333, 4334, 4335, 4336, 4337, 4338, 4339, 4340, 4341, 4342, 4343, 4344, 4345, 4346, 4352, 4353, 4354, 4355, 4356, 4357, 4358, 4359, 4360, 4361, 4362, 4363, 4364, 4365, 4366, 4367, 4368, 4369, 4370, 4371, 4372, 4373, 4374, 4375, 4376, 4377, 4378, 4379, 4380, 4381, 4382, 4383, 4384, 4385, 4386, 4387, 4388, 4389, 4390, 4391, 4392, 4393, 4394, 4395, 4396, 4397, 4398, 4399, 4400, 4401, 4402, 4403, 4404, 4405, 4406, 4407, 4408, 4409, 4410, 4411, 4412, 4413, 4414, 4415, 4416, 4417, 4418, 4419, 4420, 4421, 4422, 4423, 4424, 4425, 4426, 4427, 4428, 4429, 4430, 4431, 4432, 4433, 4434, 4435, 4436, 4437, 4438, 4439, 4440, 4441, 4442, 4443, 4444, 4445, 4446, 4447, 4448, 4449, 4450, 4451, 4452, 4453, 4454, 4455, 4456, 4457, 4458, 4459, 4460, 4461, 4462, 4463, 4464, 4465, 4466, 4467, 4468, 4469, 4470, 4471, 4472, 4473, 4474, 4475, 4476, 4477, 4478, 4479, 4480, 4481, 4482, 4483, 4484, 4485, 4486, 4487, 4488, 4489, 4490, 4491, 4492, 4493, 4494, 4495, 4496, 4497, 4498, 4499, 4500, 4501, 4502, 4503, 4504, 4505, 4506, 4507, 4508, 4509, 4510, 4511, 4512, 4513, 4514, 4515, 4516, 4517, 4518, 4519, 4520, 4521, 4522, 4523, 4524, 4525, 4526, 4527, 4528, 4529, 4530, 4531, 4532, 4533, 4534, 4535, 4536, 4537, 4538, 4539, 4540, 4541, 4542, 4543, 4544, 4545, 4546, 4547, 4548, 4549, 4550, 4551, 4552, 4553, 4554, 4555, 4556, 4557, 4558, 4559, 4560, 4561, 4562, 4563, 4564, 4565, 4566, 4567, 4568, 4569, 4570, 4571, 4572, 4573, 4574, 4575, 4576, 4577, 4578, 4579, 4580, 4581, 4582, 4583, 4584, 4585, 4586, 4587, 4588, 4589, 4590, 4591, 4592, 4593, 4594, 4595, 4596, 4597, 4598, 4599, 4600, 4601, 4602, 4603, 4604, 4605, 4606, 4607, 4608, 4609, 4610, 4611, 4612, 4613, 4614, 4615, 4616, 4617, 4618, 4619, 4620, 4621, 4622, 4623, 4624, 4625, 4626, 4627, 4628, 4629, 4630, 4631, 4632, 4633, 4634, 4635, 4636, 4637, 4638, 4639, 4640, 4641, 4642, 4643, 4644, 4645, 4646, 4647, 4648, 4649, 4650, 4651, 4652, 4653, 4654, 4655, 4656, 4657, 4658, 4659, 4660, 4661, 4662, 4663, 4664, 4665, 4666, 4667, 4668, 4669, 4670, 4671, 4672, 4673, 4674, 4675, 4676, 4677, 4678, 4679, 4680, 4682, 4683, 4684, 4685, 4688, 4689, 4690, 4691, 4692, 4693, 4694, 4696, 4698, 4699, 4700, 4701, 4704, 4705, 4706, 4707, 4708, 4709, 4710, 4711, 4712, 4713, 4714, 4715, 4716, 4717, 4718, 4719, 4720, 4721, 4722, 4723, 4724, 4725, 4726, 4727, 4728, 4729, 4730, 4731, 4732, 4733, 4734, 4735, 4736, 4737, 4738, 4739, 4740, 4741, 4742, 4743, 4744, 4746, 4747, 4748, 4749, 4752, 4753, 4754, 4755, 4756, 4757, 4758, 4759, 4760, 4761, 4762, 4763, 4764, 4765, 4766, 4767, 4768, 4769, 4770, 4771, 4772, 4773, 4774, 4775, 4776, 4777, 4778, 4779, 4780, 4781, 4782, 4783, 4784, 4786, 4787, 4788, 4789, 4792, 4793, 4794, 4795, 4796, 4797, 4798, 4800, 4802, 4803, 4804, 4805, 4808, 4809, 4810, 4811, 4812, 4813, 4814, 4815, 4816, 4817, 4818, 4819, 4820, 4821, 4822, 4824, 4825, 4826, 4827, 4828, 4829, 4830, 4831, 4832, 4833, 4834, 4835, 4836, 4837, 4838, 4839, 4840, 4841, 4842, 4843, 4844, 4845, 4846, 4847, 4848, 4849, 4850, 4851, 4852, 4853, 4854, 4855, 4856, 4857, 4858, 4859, 4860, 4861, 4862, 4863, 4864, 4865, 4866, 4867, 4868, 4869, 4870, 4871, 4872, 4873, 4874, 4875, 4876, 4877, 4878, 4879, 4880, 4882, 4883, 4884, 4885, 4888, 4889, 4890, 4891, 4892, 4893, 4894, 4895, 4896, 4897, 4898, 4899, 4900, 4901, 4902, 4903, 4904, 4905, 4906, 4907, 4908, 4909, 4910, 4911, 4912, 4913, 4914, 4915, 4916, 4917, 4918, 4919, 4920, 4921, 4922, 4923, 4924, 4925, 4926, 4927, 4928, 4929, 4930, 4931, 4932, 4933, 4934, 4935, 4936, 4937, 4938, 4939, 4940, 4941, 4942, 4943, 4944, 4945, 4946, 4947, 4948, 4949, 4950, 4951, 4952, 4953, 4954, 4992, 4993, 4994, 4995, 4996, 4997, 4998, 4999, 5000, 5001, 5002, 5003, 5004, 5005, 5006, 5007, 5024, 5025, 5026, 5027, 5028, 5029, 5030, 5031, 5032, 5033, 5034, 5035, 5036, 5037, 5038, 5039, 5040, 5041, 5042, 5043, 5044, 5045, 5046, 5047, 5048, 5049, 5050, 5051, 5052, 5053, 5054, 5055, 5056, 5057, 5058, 5059, 5060, 5061, 5062, 5063, 5064, 5065, 5066, 5067, 5068, 5069, 5070, 5071, 5072, 5073, 5074, 5075, 5076, 5077, 5078, 5079, 5080, 5081, 5082, 5083, 5084, 5085, 5086, 5087, 5088, 5089, 5090, 5091, 5092, 5093, 5094, 5095, 5096, 5097, 5098, 5099, 5100, 5101, 5102, 5103, 5104, 5105, 5106, 5107, 5108, 5121, 5122, 5123, 5124, 5125, 5126, 5127, 5128, 5129, 5130, 5131, 5132, 5133, 5134, 5135, 5136, 5137, 5138, 5139, 5140, 5141, 5142, 5143, 5144, 5145, 5146, 5147, 5148, 5149, 5150, 5151, 5152, 5153, 5154, 5155, 5156, 5157, 5158, 5159, 5160, 5161, 5162, 5163, 5164, 5165, 5166, 5167, 5168, 5169, 5170, 5171, 5172, 5173, 5174, 5175, 5176, 5177, 5178, 5179, 5180, 5181, 5182, 5183, 5184, 5185, 5186, 5187, 5188, 5189, 5190, 5191, 5192, 5193, 5194, 5195, 5196, 5197, 5198, 5199, 5200, 5201, 5202, 5203, 5204, 5205, 5206, 5207, 5208, 5209, 5210, 5211, 5212, 5213, 5214, 5215, 5216, 5217, 5218, 5219, 5220, 5221, 5222, 5223, 5224, 5225, 5226, 5227, 5228, 5229, 5230, 5231, 5232, 5233, 5234, 5235, 5236, 5237, 5238, 5239, 5240, 5241, 5242, 5243, 5244, 5245, 5246, 5247, 5248, 5249, 5250, 5251, 5252, 5253, 5254, 5255, 5256, 5257, 5258, 5259, 5260, 5261, 5262, 5263, 5264, 5265, 5266, 5267, 5268, 5269, 5270, 5271, 5272, 5273, 5274, 5275, 5276, 5277, 5278, 5279, 5280, 5281, 5282, 5283, 5284, 5285, 5286, 5287, 5288, 5289, 5290, 5291, 5292, 5293, 5294, 5295, 5296, 5297, 5298, 5299, 5300, 5301, 5302, 5303, 5304, 5305, 5306, 5307, 5308, 5309, 5310, 5311, 5312, 5313, 5314, 5315, 5316, 5317, 5318, 5319, 5320, 5321, 5322, 5323, 5324, 5325, 5326, 5327, 5328, 5329, 5330, 5331, 5332, 5333, 5334, 5335, 5336, 5337, 5338, 5339, 5340, 5341, 5342, 5343, 5344, 5345, 5346, 5347, 5348, 5349, 5350, 5351, 5352, 5353, 5354, 5355, 5356, 5357, 5358, 5359, 5360, 5361, 5362, 5363, 5364, 5365, 5366, 5367, 5368, 5369, 5370, 5371, 5372, 5373, 5374, 5375, 5376, 5377, 5378, 5379, 5380, 5381, 5382, 5383, 5384, 5385, 5386, 5387, 5388, 5389, 5390, 5391, 5392, 5393, 5394, 5395, 5396, 5397, 5398, 5399, 5400, 5401, 5402, 5403, 5404, 5405, 5406, 5407, 5408, 5409, 5410, 5411, 5412, 5413, 5414, 5415, 5416, 5417, 5418, 5419, 5420, 5421, 5422, 5423, 5424, 5425, 5426, 5427, 5428, 5429, 5430, 5431, 5432, 5433, 5434, 5435, 5436, 5437, 5438, 5439, 5440, 5441, 5442, 5443, 5444, 5445, 5446, 5447, 5448, 5449, 5450, 5451, 5452, 5453, 5454, 5455, 5456, 5457, 5458, 5459, 5460, 5461, 5462, 5463, 5464, 5465, 5466, 5467, 5468, 5469, 5470, 5471, 5472, 5473, 5474, 5475, 5476, 5477, 5478, 5479, 5480, 5481, 5482, 5483, 5484, 5485, 5486, 5487, 5488, 5489, 5490, 5491, 5492, 5493, 5494, 5495, 5496, 5497, 5498, 5499, 5500, 5501, 5502, 5503, 5504, 5505, 5506, 5507, 5508, 5509, 5510, 5511, 5512, 5513, 5514, 5515, 5516, 5517, 5518, 5519, 5520, 5521, 5522, 5523, 5524, 5525, 5526, 5527, 5528, 5529, 5530, 5531, 5532, 5533, 5534, 5535, 5536, 5537, 5538, 5539, 5540, 5541, 5542, 5543, 5544, 5545, 5546, 5547, 5548, 5549, 5550, 5551, 5552, 5553, 5554, 5555, 5556, 5557, 5558, 5559, 5560, 5561, 5562, 5563, 5564, 5565, 5566, 5567, 5568, 5569, 5570, 5571, 5572, 5573, 5574, 5575, 5576, 5577, 5578, 5579, 5580, 5581, 5582, 5583, 5584, 5585, 5586, 5587, 5588, 5589, 5590, 5591, 5592, 5593, 5594, 5595, 5596, 5597, 5598, 5599, 5600, 5601, 5602, 5603, 5604, 5605, 5606, 5607, 5608, 5609, 5610, 5611, 5612, 5613, 5614, 5615, 5616, 5617, 5618, 5619, 5620, 5621, 5622, 5623, 5624, 5625, 5626, 5627, 5628, 5629, 5630, 5631, 5632, 5633, 5634, 5635, 5636, 5637, 5638, 5639, 5640, 5641, 5642, 5643, 5644, 5645, 5646, 5647, 5648, 5649, 5650, 5651, 5652, 5653, 5654, 5655, 5656, 5657, 5658, 5659, 5660, 5661, 5662, 5663, 5664, 5665, 5666, 5667, 5668, 5669, 5670, 5671, 5672, 5673, 5674, 5675, 5676, 5677, 5678, 5679, 5680, 5681, 5682, 5683, 5684, 5685, 5686, 5687, 5688, 5689, 5690, 5691, 5692, 5693, 5694, 5695, 5696, 5697, 5698, 5699, 5700, 5701, 5702, 5703, 5704, 5705, 5706, 5707, 5708, 5709, 5710, 5711, 5712, 5713, 5714, 5715, 5716, 5717, 5718, 5719, 5720, 5721, 5722, 5723, 5724, 5725, 5726, 5727, 5728, 5729, 5730, 5731, 5732, 5733, 5734, 5735, 5736, 5737, 5738, 5739, 5740, 5743, 5744, 5745, 5746, 5747, 5748, 5749, 5750, 5751, 5752, 5753, 5754, 5755, 5756, 5757, 5758, 5759, 5761, 5762, 5763, 5764, 5765, 5766, 5767, 5768, 5769, 5770, 5771, 5772, 5773, 5774, 5775, 5776, 5777, 5778, 5779, 5780, 5781, 5782, 5783, 5784, 5785, 5786, 5792, 5793, 5794, 5795, 5796, 5797, 5798, 5799, 5800, 5801, 5802, 5803, 5804, 5805, 5806, 5807, 5808, 5809, 5810, 5811, 5812, 5813, 5814, 5815, 5816, 5817, 5818, 5819, 5820, 5821, 5822, 5823, 5824, 5825, 5826, 5827, 5828, 5829, 5830, 5831, 5832, 5833, 5834, 5835, 5836, 5837, 5838, 5839, 5840, 5841, 5842, 5843, 5844, 5845, 5846, 5847, 5848, 5849, 5850, 5851, 5852, 5853, 5854, 5855, 5856, 5857, 5858, 5859, 5860, 5861, 5862, 5863, 5864, 5865, 5866, 5888, 5889, 5890, 5891, 5892, 5893, 5894, 5895, 5896, 5897, 5898, 5899, 5900, 5902, 5903, 5904, 5905, 5920, 5921, 5922, 5923, 5924, 5925, 5926, 5927, 5928, 5929, 5930, 5931, 5932, 5933, 5934, 5935, 5936, 5937, 5952, 5953, 5954, 5955, 5956, 5957, 5958, 5959, 5960, 5961, 5962, 5963, 5964, 5965, 5966, 5967, 5968, 5969, 5984, 5985, 5986, 5987, 5988, 5989, 5990, 5991, 5992, 5993, 5994, 5995, 5996, 5998, 5999, 6000, 6016, 6017, 6018, 6019, 6020, 6021, 6022, 6023, 6024, 6025, 6026, 6027, 6028, 6029, 6030, 6031, 6032, 6033, 6034, 6035, 6036, 6037, 6038, 6039, 6040, 6041, 6042, 6043, 6044, 6045, 6046, 6047, 6048, 6049, 6050, 6051, 6052, 6053, 6054, 6055, 6056, 6057, 6058, 6059, 6060, 6061, 6062, 6063, 6064, 6065, 6066, 6067, 6108, 6176, 6177, 6178, 6179, 6180, 6181, 6182, 6183, 6184, 6185, 6186, 6187, 6188, 6189, 6190, 6191, 6192, 6193, 6194, 6195, 6196, 6197, 6198, 6199, 6200, 6201, 6202, 6203, 6204, 6205, 6206, 6207, 6208, 6209, 6210, 6212, 6213, 6214, 6215, 6216, 6217, 6218, 6219, 6220, 6221, 6222, 6223, 6224, 6225, 6226, 6227, 6228, 6229, 6230, 6231, 6232, 6233, 6234, 6235, 6236, 6237, 6238, 6239, 6240, 6241, 6242, 6243, 6244, 6245, 6246, 6247, 6248, 6249, 6250, 6251, 6252, 6253, 6254, 6255, 6256, 6257, 6258, 6259, 6260, 6261, 6262, 6263, 6272, 6273, 6274, 6275, 6276, 6277, 6278, 6279, 6280, 6281, 6282, 6283, 6284, 6285, 6286, 6287, 6288, 6289, 6290, 6291, 6292, 6293, 6294, 6295, 6296, 6297, 6298, 6299, 6300, 6301, 6302, 6303, 6304, 6305, 6306, 6307, 6308, 6309, 6310, 6311, 6312, 6314, 6320, 6321, 6322, 6323, 6324, 6325, 6326, 6327, 6328, 6329, 6330, 6331, 6332, 6333, 6334, 6335, 6336, 6337, 6338, 6339, 6340, 6341, 6342, 6343, 6344, 6345, 6346, 6347, 6348, 6349, 6350, 6351, 6352, 6353, 6354, 6355, 6356, 6357, 6358, 6359, 6360, 6361, 6362, 6363, 6364, 6365, 6366, 6367, 6368, 6369, 6370, 6371, 6372, 6373, 6374, 6375, 6376, 6377, 6378, 6379, 6380, 6381, 6382, 6383, 6384, 6385, 6386, 6387, 6388, 6389, 6400, 6401, 6402, 6403, 6404, 6405, 6406, 6407, 6408, 6409, 6410, 6411, 6412, 6413, 6414, 6415, 6416, 6417, 6418, 6419, 6420, 6421, 6422, 6423, 6424, 6425, 6426, 6427, 6428, 6480, 6481, 6482, 6483, 6484, 6485, 6486, 6487, 6488, 6489, 6490, 6491, 6492, 6493, 6494, 6495, 6496, 6497, 6498, 6499, 6500, 6501, 6502, 6503, 6504, 6505, 6506, 6507, 6508, 6509, 6512, 6513, 6514, 6515, 6516, 6528, 6529, 6530, 6531, 6532, 6533, 6534, 6535, 6536, 6537, 6538, 6539, 6540, 6541, 6542, 6543, 6544, 6545, 6546, 6547, 6548, 6549, 6550, 6551, 6552, 6553, 6554, 6555, 6556, 6557, 6558, 6559, 6560, 6561, 6562, 6563, 6564, 6565, 6566, 6567, 6568, 6569, 6570, 6571, 6593, 6594, 6595, 6596, 6597, 6598, 6599, 6656, 6657, 6658, 6659, 6660, 6661, 6662, 6663, 6664, 6665, 6666, 6667, 6668, 6669, 6670, 6671, 6672, 6673, 6674, 6675, 6676, 6677, 6678, 6688, 6689, 6690, 6691, 6692, 6693, 6694, 6695, 6696, 6697, 6698, 6699, 6700, 6701, 6702, 6703, 6704, 6705, 6706, 6707, 6708, 6709, 6710, 6711, 6712, 6713, 6714, 6715, 6716, 6717, 6718, 6719, 6720, 6721, 6722, 6723, 6724, 6725, 6726, 6727, 6728, 6729, 6730, 6731, 6732, 6733, 6734, 6735, 6736, 6737, 6738, 6739, 6740, 6917, 6918, 6919, 6920, 6921, 6922, 6923, 6924, 6925, 6926, 6927, 6928, 6929, 6930, 6931, 6932, 6933, 6934, 6935, 6936, 6937, 6938, 6939, 6940, 6941, 6942, 6943, 6944, 6945, 6946, 6947, 6948, 6949, 6950, 6951, 6952, 6953, 6954, 6955, 6956, 6957, 6958, 6959, 6960, 6961, 6962, 6963, 6981, 6982, 6983, 6984, 6985, 6986, 6987, 7043, 7044, 7045, 7046, 7047, 7048, 7049, 7050, 7051, 7052, 7053, 7054, 7055, 7056, 7057, 7058, 7059, 7060, 7061, 7062, 7063, 7064, 7065, 7066, 7067, 7068, 7069, 7070, 7071, 7072, 7086, 7087, 7104, 7105, 7106, 7107, 7108, 7109, 7110, 7111, 7112, 7113, 7114, 7115, 7116, 7117, 7118, 7119, 7120, 7121, 7122, 7123, 7124, 7125, 7126, 7127, 7128, 7129, 7130, 7131, 7132, 7133, 7134, 7135, 7136, 7137, 7138, 7139, 7140, 7141, 7168, 7169, 7170, 7171, 7172, 7173, 7174, 7175, 7176, 7177, 7178, 7179, 7180, 7181, 7182, 7183, 7184, 7185, 7186, 7187, 7188, 7189, 7190, 7191, 7192, 7193, 7194, 7195, 7196, 7197, 7198, 7199, 7200, 7201, 7202, 7203, 7245, 7246, 7247, 7258, 7259, 7260, 7261, 7262, 7263, 7264, 7265, 7266, 7267, 7268, 7269, 7270, 7271, 7272, 7273, 7274, 7275, 7276, 7277, 7278, 7279, 7280, 7281, 7282, 7283, 7284, 7285, 7286, 7287, 7401, 7402, 7403, 7404, 7406, 7407, 7408, 7409, 8501, 8502, 8503, 8504, 11568, 11569, 11570, 11571, 11572, 11573, 11574, 11575, 11576, 11577, 11578, 11579, 11580, 11581, 11582, 11583, 11584, 11585, 11586, 11587, 11588, 11589, 11590, 11591, 11592, 11593, 11594, 11595, 11596, 11597, 11598, 11599, 11600, 11601, 11602, 11603, 11604, 11605, 11606, 11607, 11608, 11609, 11610, 11611, 11612, 11613, 11614, 11615, 11616, 11617, 11618, 11619, 11620, 11621, 11648, 11649, 11650, 11651, 11652, 11653, 11654, 11655, 11656, 11657, 11658, 11659, 11660, 11661, 11662, 11663, 11664, 11665, 11666, 11667, 11668, 11669, 11670, 11680, 11681, 11682, 11683, 11684, 11685, 11686, 11688, 11689, 11690, 11691, 11692, 11693, 11694, 11696, 11697, 11698, 11699, 11700, 11701, 11702, 11704, 11705, 11706, 11707, 11708, 11709, 11710, 11712, 11713, 11714, 11715, 11716, 11717, 11718, 11720, 11721, 11722, 11723, 11724, 11725, 11726, 11728, 11729, 11730, 11731, 11732, 11733, 11734, 11736, 11737, 11738, 11739, 11740, 11741, 11742, 12294, 12348, 12353, 12354, 12355, 12356, 12357, 12358, 12359, 12360, 12361, 12362, 12363, 12364, 12365, 12366, 12367, 12368, 12369, 12370, 12371, 12372, 12373, 12374, 12375, 12376, 12377, 12378, 12379, 12380, 12381, 12382, 12383, 12384, 12385, 12386, 12387, 12388, 12389, 12390, 12391, 12392, 12393, 12394, 12395, 12396, 12397, 12398, 12399, 12400, 12401, 12402, 12403, 12404, 12405, 12406, 12407, 12408, 12409, 12410, 12411, 12412, 12413, 12414, 12415, 12416, 12417, 12418, 12419, 12420, 12421, 12422, 12423, 12424, 12425, 12426, 12427, 12428, 12429, 12430, 12431, 12432, 12433, 12434, 12435, 12436, 12437, 12438, 12447, 12449, 12450, 12451, 12452, 12453, 12454, 12455, 12456, 12457, 12458, 12459, 12460, 12461, 12462, 12463, 12464, 12465, 12466, 12467, 12468, 12469, 12470, 12471, 12472, 12473, 12474, 12475, 12476, 12477, 12478, 12479, 12480, 12481, 12482, 12483, 12484, 12485, 12486, 12487, 12488, 12489, 12490, 12491, 12492, 12493, 12494, 12495, 12496, 12497, 12498, 12499, 12500, 12501, 12502, 12503, 12504, 12505, 12506, 12507, 12508, 12509, 12510, 12511, 12512, 12513, 12514, 12515, 12516, 12517, 12518, 12519, 12520, 12521, 12522, 12523, 12524, 12525, 12526, 12527, 12528, 12529, 12530, 12531, 12532, 12533, 12534, 12535, 12536, 12537, 12538, 12543, 12549, 12550, 12551, 12552, 12553, 12554, 12555, 12556, 12557, 12558, 12559, 12560, 12561, 12562, 12563, 12564, 12565, 12566, 12567, 12568, 12569, 12570, 12571, 12572, 12573, 12574, 12575, 12576, 12577, 12578, 12579, 12580, 12581, 12582, 12583, 12584, 12585, 12586, 12587, 12588, 12589, 12593, 12594, 12595, 12596, 12597, 12598, 12599, 12600, 12601, 12602, 12603, 12604, 12605, 12606, 12607, 12608, 12609, 12610, 12611, 12612, 12613, 12614, 12615, 12616, 12617, 12618, 12619, 12620, 12621, 12622, 12623, 12624, 12625, 12626, 12627, 12628, 12629, 12630, 12631, 12632, 12633, 12634, 12635, 12636, 12637, 12638, 12639, 12640, 12641, 12642, 12643, 12644, 12645, 12646, 12647, 12648, 12649, 12650, 12651, 12652, 12653, 12654, 12655, 12656, 12657, 12658, 12659, 12660, 12661, 12662, 12663, 12664, 12665, 12666, 12667, 12668, 12669, 12670, 12671, 12672, 12673, 12674, 12675, 12676, 12677, 12678, 12679, 12680, 12681, 12682, 12683, 12684, 12685, 12686, 12704, 12705, 12706, 12707, 12708, 12709, 12710, 12711, 12712, 12713, 12714, 12715, 12716, 12717, 12718, 12719, 12720, 12721, 12722, 12723, 12724, 12725, 12726, 12727, 12728, 12729, 12730, 12784, 12785, 12786, 12787, 12788, 12789, 12790, 12791, 12792, 12793, 12794, 12795, 12796, 12797, 12798, 12799, 13312, 19893, 19968, 40907, 40960, 40961, 40962, 40963, 40964, 40965, 40966, 40967, 40968, 40969, 40970, 40971, 40972, 40973, 40974, 40975, 40976, 40977, 40978, 40979, 40980, 40982, 40983, 40984, 40985, 40986, 40987, 40988, 40989, 40990, 40991, 40992, 40993, 40994, 40995, 40996, 40997, 40998, 40999, 41000, 41001, 41002, 41003, 41004, 41005, 41006, 41007, 41008, 41009, 41010, 41011, 41012, 41013, 41014, 41015, 41016, 41017, 41018, 41019, 41020, 41021, 41022, 41023, 41024, 41025, 41026, 41027, 41028, 41029, 41030, 41031, 41032, 41033, 41034, 41035, 41036, 41037, 41038, 41039, 41040, 41041, 41042, 41043, 41044, 41045, 41046, 41047, 41048, 41049, 41050, 41051, 41052, 41053, 41054, 41055, 41056, 41057, 41058, 41059, 41060, 41061, 41062, 41063, 41064, 41065, 41066, 41067, 41068, 41069, 41070, 41071, 41072, 41073, 41074, 41075, 41076, 41077, 41078, 41079, 41080, 41081, 41082, 41083, 41084, 41085, 41086, 41087, 41088, 41089, 41090, 41091, 41092, 41093, 41094, 41095, 41096, 41097, 41098, 41099, 41100, 41101, 41102, 41103, 41104, 41105, 41106, 41107, 41108, 41109, 41110, 41111, 41112, 41113, 41114, 41115, 41116, 41117, 41118, 41119, 41120, 41121, 41122, 41123, 41124, 41125, 41126, 41127, 41128, 41129, 41130, 41131, 41132, 41133, 41134, 41135, 41136, 41137, 41138, 41139, 41140, 41141, 41142, 41143, 41144, 41145, 41146, 41147, 41148, 41149, 41150, 41151, 41152, 41153, 41154, 41155, 41156, 41157, 41158, 41159, 41160, 41161, 41162, 41163, 41164, 41165, 41166, 41167, 41168, 41169, 41170, 41171, 41172, 41173, 41174, 41175, 41176, 41177, 41178, 41179, 41180, 41181, 41182, 41183, 41184, 41185, 41186, 41187, 41188, 41189, 41190, 41191, 41192, 41193, 41194, 41195, 41196, 41197, 41198, 41199, 41200, 41201, 41202, 41203, 41204, 41205, 41206, 41207, 41208, 41209, 41210, 41211, 41212, 41213, 41214, 41215, 41216, 41217, 41218, 41219, 41220, 41221, 41222, 41223, 41224, 41225, 41226, 41227, 41228, 41229, 41230, 41231, 41232, 41233, 41234, 41235, 41236, 41237, 41238, 41239, 41240, 41241, 41242, 41243, 41244, 41245, 41246, 41247, 41248, 41249, 41250, 41251, 41252, 41253, 41254, 41255, 41256, 41257, 41258, 41259, 41260, 41261, 41262, 41263, 41264, 41265, 41266, 41267, 41268, 41269, 41270, 41271, 41272, 41273, 41274, 41275, 41276, 41277, 41278, 41279, 41280, 41281, 41282, 41283, 41284, 41285, 41286, 41287, 41288, 41289, 41290, 41291, 41292, 41293, 41294, 41295, 41296, 41297, 41298, 41299, 41300, 41301, 41302, 41303, 41304, 41305, 41306, 41307, 41308, 41309, 41310, 41311, 41312, 41313, 41314, 41315, 41316, 41317, 41318, 41319, 41320, 41321, 41322, 41323, 41324, 41325, 41326, 41327, 41328, 41329, 41330, 41331, 41332, 41333, 41334, 41335, 41336, 41337, 41338, 41339, 41340, 41341, 41342, 41343, 41344, 41345, 41346, 41347, 41348, 41349, 41350, 41351, 41352, 41353, 41354, 41355, 41356, 41357, 41358, 41359, 41360, 41361, 41362, 41363, 41364, 41365, 41366, 41367, 41368, 41369, 41370, 41371, 41372, 41373, 41374, 41375, 41376, 41377, 41378, 41379, 41380, 41381, 41382, 41383, 41384, 41385, 41386, 41387, 41388, 41389, 41390, 41391, 41392, 41393, 41394, 41395, 41396, 41397, 41398, 41399, 41400, 41401, 41402, 41403, 41404, 41405, 41406, 41407, 41408, 41409, 41410, 41411, 41412, 41413, 41414, 41415, 41416, 41417, 41418, 41419, 41420, 41421, 41422, 41423, 41424, 41425, 41426, 41427, 41428, 41429, 41430, 41431, 41432, 41433, 41434, 41435, 41436, 41437, 41438, 41439, 41440, 41441, 41442, 41443, 41444, 41445, 41446, 41447, 41448, 41449, 41450, 41451, 41452, 41453, 41454, 41455, 41456, 41457, 41458, 41459, 41460, 41461, 41462, 41463, 41464, 41465, 41466, 41467, 41468, 41469, 41470, 41471, 41472, 41473, 41474, 41475, 41476, 41477, 41478, 41479, 41480, 41481, 41482, 41483, 41484, 41485, 41486, 41487, 41488, 41489, 41490, 41491, 41492, 41493, 41494, 41495, 41496, 41497, 41498, 41499, 41500, 41501, 41502, 41503, 41504, 41505, 41506, 41507, 41508, 41509, 41510, 41511, 41512, 41513, 41514, 41515, 41516, 41517, 41518, 41519, 41520, 41521, 41522, 41523, 41524, 41525, 41526, 41527, 41528, 41529, 41530, 41531, 41532, 41533, 41534, 41535, 41536, 41537, 41538, 41539, 41540, 41541, 41542, 41543, 41544, 41545, 41546, 41547, 41548, 41549, 41550, 41551, 41552, 41553, 41554, 41555, 41556, 41557, 41558, 41559, 41560, 41561, 41562, 41563, 41564, 41565, 41566, 41567, 41568, 41569, 41570, 41571, 41572, 41573, 41574, 41575, 41576, 41577, 41578, 41579, 41580, 41581, 41582, 41583, 41584, 41585, 41586, 41587, 41588, 41589, 41590, 41591, 41592, 41593, 41594, 41595, 41596, 41597, 41598, 41599, 41600, 41601, 41602, 41603, 41604, 41605, 41606, 41607, 41608, 41609, 41610, 41611, 41612, 41613, 41614, 41615, 41616, 41617, 41618, 41619, 41620, 41621, 41622, 41623, 41624, 41625, 41626, 41627, 41628, 41629, 41630, 41631, 41632, 41633, 41634, 41635, 41636, 41637, 41638, 41639, 41640, 41641, 41642, 41643, 41644, 41645, 41646, 41647, 41648, 41649, 41650, 41651, 41652, 41653, 41654, 41655, 41656, 41657, 41658, 41659, 41660, 41661, 41662, 41663, 41664, 41665, 41666, 41667, 41668, 41669, 41670, 41671, 41672, 41673, 41674, 41675, 41676, 41677, 41678, 41679, 41680, 41681, 41682, 41683, 41684, 41685, 41686, 41687, 41688, 41689, 41690, 41691, 41692, 41693, 41694, 41695, 41696, 41697, 41698, 41699, 41700, 41701, 41702, 41703, 41704, 41705, 41706, 41707, 41708, 41709, 41710, 41711, 41712, 41713, 41714, 41715, 41716, 41717, 41718, 41719, 41720, 41721, 41722, 41723, 41724, 41725, 41726, 41727, 41728, 41729, 41730, 41731, 41732, 41733, 41734, 41735, 41736, 41737, 41738, 41739, 41740, 41741, 41742, 41743, 41744, 41745, 41746, 41747, 41748, 41749, 41750, 41751, 41752, 41753, 41754, 41755, 41756, 41757, 41758, 41759, 41760, 41761, 41762, 41763, 41764, 41765, 41766, 41767, 41768, 41769, 41770, 41771, 41772, 41773, 41774, 41775, 41776, 41777, 41778, 41779, 41780, 41781, 41782, 41783, 41784, 41785, 41786, 41787, 41788, 41789, 41790, 41791, 41792, 41793, 41794, 41795, 41796, 41797, 41798, 41799, 41800, 41801, 41802, 41803, 41804, 41805, 41806, 41807, 41808, 41809, 41810, 41811, 41812, 41813, 41814, 41815, 41816, 41817, 41818, 41819, 41820, 41821, 41822, 41823, 41824, 41825, 41826, 41827, 41828, 41829, 41830, 41831, 41832, 41833, 41834, 41835, 41836, 41837, 41838, 41839, 41840, 41841, 41842, 41843, 41844, 41845, 41846, 41847, 41848, 41849, 41850, 41851, 41852, 41853, 41854, 41855, 41856, 41857, 41858, 41859, 41860, 41861, 41862, 41863, 41864, 41865, 41866, 41867, 41868, 41869, 41870, 41871, 41872, 41873, 41874, 41875, 41876, 41877, 41878, 41879, 41880, 41881, 41882, 41883, 41884, 41885, 41886, 41887, 41888, 41889, 41890, 41891, 41892, 41893, 41894, 41895, 41896, 41897, 41898, 41899, 41900, 41901, 41902, 41903, 41904, 41905, 41906, 41907, 41908, 41909, 41910, 41911, 41912, 41913, 41914, 41915, 41916, 41917, 41918, 41919, 41920, 41921, 41922, 41923, 41924, 41925, 41926, 41927, 41928, 41929, 41930, 41931, 41932, 41933, 41934, 41935, 41936, 41937, 41938, 41939, 41940, 41941, 41942, 41943, 41944, 41945, 41946, 41947, 41948, 41949, 41950, 41951, 41952, 41953, 41954, 41955, 41956, 41957, 41958, 41959, 41960, 41961, 41962, 41963, 41964, 41965, 41966, 41967, 41968, 41969, 41970, 41971, 41972, 41973, 41974, 41975, 41976, 41977, 41978, 41979, 41980, 41981, 41982, 41983, 41984, 41985, 41986, 41987, 41988, 41989, 41990, 41991, 41992, 41993, 41994, 41995, 41996, 41997, 41998, 41999, 42000, 42001, 42002, 42003, 42004, 42005, 42006, 42007, 42008, 42009, 42010, 42011, 42012, 42013, 42014, 42015, 42016, 42017, 42018, 42019, 42020, 42021, 42022, 42023, 42024, 42025, 42026, 42027, 42028, 42029, 42030, 42031, 42032, 42033, 42034, 42035, 42036, 42037, 42038, 42039, 42040, 42041, 42042, 42043, 42044, 42045, 42046, 42047, 42048, 42049, 42050, 42051, 42052, 42053, 42054, 42055, 42056, 42057, 42058, 42059, 42060, 42061, 42062, 42063, 42064, 42065, 42066, 42067, 42068, 42069, 42070, 42071, 42072, 42073, 42074, 42075, 42076, 42077, 42078, 42079, 42080, 42081, 42082, 42083, 42084, 42085, 42086, 42087, 42088, 42089, 42090, 42091, 42092, 42093, 42094, 42095, 42096, 42097, 42098, 42099, 42100, 42101, 42102, 42103, 42104, 42105, 42106, 42107, 42108, 42109, 42110, 42111, 42112, 42113, 42114, 42115, 42116, 42117, 42118, 42119, 42120, 42121, 42122, 42123, 42124, 42192, 42193, 42194, 42195, 42196, 42197, 42198, 42199, 42200, 42201, 42202, 42203, 42204, 42205, 42206, 42207, 42208, 42209, 42210, 42211, 42212, 42213, 42214, 42215, 42216, 42217, 42218, 42219, 42220, 42221, 42222, 42223, 42224, 42225, 42226, 42227, 42228, 42229, 42230, 42231, 42240, 42241, 42242, 42243, 42244, 42245, 42246, 42247, 42248, 42249, 42250, 42251, 42252, 42253, 42254, 42255, 42256, 42257, 42258, 42259, 42260, 42261, 42262, 42263, 42264, 42265, 42266, 42267, 42268, 42269, 42270, 42271, 42272, 42273, 42274, 42275, 42276, 42277, 42278, 42279, 42280, 42281, 42282, 42283, 42284, 42285, 42286, 42287, 42288, 42289, 42290, 42291, 42292, 42293, 42294, 42295, 42296, 42297, 42298, 42299, 42300, 42301, 42302, 42303, 42304, 42305, 42306, 42307, 42308, 42309, 42310, 42311, 42312, 42313, 42314, 42315, 42316, 42317, 42318, 42319, 42320, 42321, 42322, 42323, 42324, 42325, 42326, 42327, 42328, 42329, 42330, 42331, 42332, 42333, 42334, 42335, 42336, 42337, 42338, 42339, 42340, 42341, 42342, 42343, 42344, 42345, 42346, 42347, 42348, 42349, 42350, 42351, 42352, 42353, 42354, 42355, 42356, 42357, 42358, 42359, 42360, 42361, 42362, 42363, 42364, 42365, 42366, 42367, 42368, 42369, 42370, 42371, 42372, 42373, 42374, 42375, 42376, 42377, 42378, 42379, 42380, 42381, 42382, 42383, 42384, 42385, 42386, 42387, 42388, 42389, 42390, 42391, 42392, 42393, 42394, 42395, 42396, 42397, 42398, 42399, 42400, 42401, 42402, 42403, 42404, 42405, 42406, 42407, 42408, 42409, 42410, 42411, 42412, 42413, 42414, 42415, 42416, 42417, 42418, 42419, 42420, 42421, 42422, 42423, 42424, 42425, 42426, 42427, 42428, 42429, 42430, 42431, 42432, 42433, 42434, 42435, 42436, 42437, 42438, 42439, 42440, 42441, 42442, 42443, 42444, 42445, 42446, 42447, 42448, 42449, 42450, 42451, 42452, 42453, 42454, 42455, 42456, 42457, 42458, 42459, 42460, 42461, 42462, 42463, 42464, 42465, 42466, 42467, 42468, 42469, 42470, 42471, 42472, 42473, 42474, 42475, 42476, 42477, 42478, 42479, 42480, 42481, 42482, 42483, 42484, 42485, 42486, 42487, 42488, 42489, 42490, 42491, 42492, 42493, 42494, 42495, 42496, 42497, 42498, 42499, 42500, 42501, 42502, 42503, 42504, 42505, 42506, 42507, 42512, 42513, 42514, 42515, 42516, 42517, 42518, 42519, 42520, 42521, 42522, 42523, 42524, 42525, 42526, 42527, 42538, 42539, 42606, 42656, 42657, 42658, 42659, 42660, 42661, 42662, 42663, 42664, 42665, 42666, 42667, 42668, 42669, 42670, 42671, 42672, 42673, 42674, 42675, 42676, 42677, 42678, 42679, 42680, 42681, 42682, 42683, 42684, 42685, 42686, 42687, 42688, 42689, 42690, 42691, 42692, 42693, 42694, 42695, 42696, 42697, 42698, 42699, 42700, 42701, 42702, 42703, 42704, 42705, 42706, 42707, 42708, 42709, 42710, 42711, 42712, 42713, 42714, 42715, 42716, 42717, 42718, 42719, 42720, 42721, 42722, 42723, 42724, 42725, 43003, 43004, 43005, 43006, 43007, 43008, 43009, 43011, 43012, 43013, 43015, 43016, 43017, 43018, 43020, 43021, 43022, 43023, 43024, 43025, 43026, 43027, 43028, 43029, 43030, 43031, 43032, 43033, 43034, 43035, 43036, 43037, 43038, 43039, 43040, 43041, 43042, 43072, 43073, 43074, 43075, 43076, 43077, 43078, 43079, 43080, 43081, 43082, 43083, 43084, 43085, 43086, 43087, 43088, 43089, 43090, 43091, 43092, 43093, 43094, 43095, 43096, 43097, 43098, 43099, 43100, 43101, 43102, 43103, 43104, 43105, 43106, 43107, 43108, 43109, 43110, 43111, 43112, 43113, 43114, 43115, 43116, 43117, 43118, 43119, 43120, 43121, 43122, 43123, 43138, 43139, 43140, 43141, 43142, 43143, 43144, 43145, 43146, 43147, 43148, 43149, 43150, 43151, 43152, 43153, 43154, 43155, 43156, 43157, 43158, 43159, 43160, 43161, 43162, 43163, 43164, 43165, 43166, 43167, 43168, 43169, 43170, 43171, 43172, 43173, 43174, 43175, 43176, 43177, 43178, 43179, 43180, 43181, 43182, 43183, 43184, 43185, 43186, 43187, 43250, 43251, 43252, 43253, 43254, 43255, 43259, 43274, 43275, 43276, 43277, 43278, 43279, 43280, 43281, 43282, 43283, 43284, 43285, 43286, 43287, 43288, 43289, 43290, 43291, 43292, 43293, 43294, 43295, 43296, 43297, 43298, 43299, 43300, 43301, 43312, 43313, 43314, 43315, 43316, 43317, 43318, 43319, 43320, 43321, 43322, 43323, 43324, 43325, 43326, 43327, 43328, 43329, 43330, 43331, 43332, 43333, 43334, 43360, 43361, 43362, 43363, 43364, 43365, 43366, 43367, 43368, 43369, 43370, 43371, 43372, 43373, 43374, 43375, 43376, 43377, 43378, 43379, 43380, 43381, 43382, 43383, 43384, 43385, 43386, 43387, 43388, 43396, 43397, 43398, 43399, 43400, 43401, 43402, 43403, 43404, 43405, 43406, 43407, 43408, 43409, 43410, 43411, 43412, 43413, 43414, 43415, 43416, 43417, 43418, 43419, 43420, 43421, 43422, 43423, 43424, 43425, 43426, 43427, 43428, 43429, 43430, 43431, 43432, 43433, 43434, 43435, 43436, 43437, 43438, 43439, 43440, 43441, 43442, 43520, 43521, 43522, 43523, 43524, 43525, 43526, 43527, 43528, 43529, 43530, 43531, 43532, 43533, 43534, 43535, 43536, 43537, 43538, 43539, 43540, 43541, 43542, 43543, 43544, 43545, 43546, 43547, 43548, 43549, 43550, 43551, 43552, 43553, 43554, 43555, 43556, 43557, 43558, 43559, 43560, 43584, 43585, 43586, 43588, 43589, 43590, 43591, 43592, 43593, 43594, 43595, 43616, 43617, 43618, 43619, 43620, 43621, 43622, 43623, 43624, 43625, 43626, 43627, 43628, 43629, 43630, 43631, 43633, 43634, 43635, 43636, 43637, 43638, 43642, 43648, 43649, 43650, 43651, 43652, 43653, 43654, 43655, 43656, 43657, 43658, 43659, 43660, 43661, 43662, 43663, 43664, 43665, 43666, 43667, 43668, 43669, 43670, 43671, 43672, 43673, 43674, 43675, 43676, 43677, 43678, 43679, 43680, 43681, 43682, 43683, 43684, 43685, 43686, 43687, 43688, 43689, 43690, 43691, 43692, 43693, 43694, 43695, 43697, 43701, 43702, 43705, 43706, 43707, 43708, 43709, 43712, 43714, 43739, 43740, 43777, 43778, 43779, 43780, 43781, 43782, 43785, 43786, 43787, 43788, 43789, 43790, 43793, 43794, 43795, 43796, 43797, 43798, 43808, 43809, 43810, 43811, 43812, 43813, 43814, 43816, 43817, 43818, 43819, 43820, 43821, 43822, 43968, 43969, 43970, 43971, 43972, 43973, 43974, 43975, 43976, 43977, 43978, 43979, 43980, 43981, 43982, 43983, 43984, 43985, 43986, 43987, 43988, 43989, 43990, 43991, 43992, 43993, 43994, 43995, 43996, 43997, 43998, 43999, 44000, 44001, 44002, 44032, 55203, 55216, 55217, 55218, 55219, 55220, 55221, 55222, 55223, 55224, 55225, 55226, 55227, 55228, 55229, 55230, 55231, 55232, 55233, 55234, 55235, 55236, 55237, 55238, 55243, 55244, 55245, 55246, 55247, 55248, 55249, 55250, 55251, 55252, 55253, 55254, 55255, 55256, 55257, 55258, 55259, 55260, 55261, 55262, 55263, 55264, 55265, 55266, 55267, 55268, 55269, 55270, 55271, 55272, 55273, 55274, 55275, 55276, 55277, 55278, 55279, 55280, 55281, 55282, 55283, 55284, 55285, 55286, 55287, 55288, 55289, 55290, 55291, 35912, 26356, 36554, 36040, 28369, 20018, 21477, 40860, 40860, 22865, 37329, 21895, 22856, 25078, 30313, 32645, 34367, 34746, 35064, 37007, 27138, 27931, 28889, 29662, 33853, 37226, 39409, 20098, 21365, 27396, 29211, 34349, 40478, 23888, 28651, 34253, 35172, 25289, 33240, 34847, 24266, 26391, 28010, 29436, 37070, 20358, 20919, 21214, 25796, 27347, 29200, 30439, 32769, 34310, 34396, 36335, 38706, 39791, 40442, 30860, 31103, 32160, 33737, 37636, 40575, 35542, 22751, 24324, 31840, 32894, 29282, 30922, 36034, 38647, 22744, 23650, 27155, 28122, 28431, 32047, 32311, 38475, 21202, 32907, 20956, 20940, 31260, 32190, 33777, 38517, 35712, 25295, 27138, 35582, 20025, 23527, 24594, 29575, 30064, 21271, 30971, 20415, 24489, 19981, 27852, 25976, 32034, 21443, 22622, 30465, 33865, 35498, 27578, 36784, 27784, 25342, 33509, 25504, 30053, 20142, 20841, 20937, 26753, 31975, 33391, 35538, 37327, 21237, 21570, 22899, 24300, 26053, 28670, 31018, 38317, 39530, 40599, 40654, 21147, 26310, 27511, 36706, 24180, 24976, 25088, 25754, 28451, 29001, 29833, 31178, 32244, 32879, 36646, 34030, 36899, 37706, 21015, 21155, 21693, 28872, 35010, 35498, 24265, 24565, 25467, 27566, 31806, 29557, 20196, 22265, 23527, 23994, 24604, 29618, 29801, 32666, 32838, 37428, 38646, 38728, 38936, 20363, 31150, 37300, 38584, 24801, 20102, 20698, 23534, 23615, 26009, 27138, 29134, 30274, 34044, 36988, 40845, 26248, 38446, 21129, 26491, 26611, 27969, 28316, 29705, 30041, 30827, 32016, 39006, 20845, 25134, 38520, 20523, 23833, 28138, 36650, 24459, 24900, 26647, 29575, 38534, 21033, 21519, 23653, 26131, 26446, 26792, 27877, 29702, 30178, 32633, 35023, 35041, 37324, 38626, 21311, 28346, 21533, 29136, 29848, 34298, 38563, 40023, 40607, 26519, 28107, 33256, 31435, 31520, 31890, 29376, 28825, 35672, 20160, 33590, 21050, 20999, 24230, 25299, 31958, 23429, 27934, 26292, 36667, 34892, 38477, 35211, 24275, 20800, 21952, 64014, 64015, 22618, 64017, 26228, 64019, 64020, 20958, 29482, 30410, 31036, 31070, 31077, 31119, 38742, 31934, 32701, 64031, 34322, 64033, 35576, 64035, 64036, 36920, 37117, 64039, 64040, 64041, 39151, 39164, 39208, 40372, 20398, 20711, 20813, 21193, 21220, 21329, 21917, 22022, 22120, 22592, 22696, 23652, 23662, 24724, 24936, 24974, 25074, 25935, 26082, 26257, 26757, 28023, 28186, 28450, 29038, 29227, 29730, 30865, 31038, 31049, 31048, 31056, 31062, 31069, 31117, 31118, 31296, 31361, 31680, 32244, 32265, 32321, 32626, 32773, 33261, 33401, 33401, 33879, 35088, 35222, 35585, 35641, 36051, 36104, 36790, 36920, 38627, 38911, 38971, 24693, 148206, 33304, 20006, 20917, 20840, 20352, 20805, 20864, 21191, 21242, 21917, 21845, 21913, 21986, 22618, 22707, 22852, 22868, 23138, 23336, 24274, 24281, 24425, 24493, 24792, 24910, 24840, 24974, 24928, 25074, 25140, 25540, 25628, 25682, 25942, 26228, 26391, 26395, 26454, 27513, 27578, 27969, 28379, 28363, 28450, 28702, 29038, 30631, 29237, 29359, 29482, 29809, 29958, 30011, 30237, 30239, 30410, 30427, 30452, 30538, 30528, 30924, 31409, 31680, 31867, 32091, 32244, 32574, 32773, 33618, 33775, 34681, 35137, 35206, 35222, 35519, 35576, 35531, 35585, 35582, 35565, 35641, 35722, 36104, 36664, 36978, 37273, 37494, 38524, 38627, 38742, 38875, 38911, 38923, 38971, 39698, 40860, 141386, 141380, 144341, 15261, 16408, 16441, 152137, 154832, 163539, 40771, 40846, 1497, 1460, 1522, 1463, 64288, 64289, 64290, 64291, 64292, 64293, 64294, 64295, 64296, 1513, 1473, 1513, 1474, 1513, 1468, 1473, 1513, 1468, 1474, 1488, 1463, 1488, 1464, 1488, 1468, 1489, 1468, 1490, 1468, 1491, 1468, 1492, 1468, 1493, 1468, 1494, 1468, 1496, 1468, 1497, 1468, 1498, 1468, 1499, 1468, 1500, 1468, 1502, 1468, 1504, 1468, 1505, 1468, 1507, 1468, 1508, 1468, 1510, 1468, 1511, 1468, 1512, 1468, 1513, 1468, 1514, 1468, 1493, 1465, 1489, 1471, 1499, 1471, 1508, 1471, 64335, 64336, 64337, 64338, 64339, 64340, 64341, 64342, 64343, 64344, 64345, 64346, 64347, 64348, 64349, 64350, 64351, 64352, 64353, 64354, 64355, 64356, 64357, 64358, 64359, 64360, 64361, 64362, 64363, 64364, 64365, 64366, 64367, 64368, 64369, 64370, 64371, 64372, 64373, 64374, 64375, 64376, 64377, 64378, 64379, 64380, 64381, 64382, 64383, 64384, 64385, 64386, 64387, 64388, 64389, 64390, 64391, 64392, 64393, 64394, 64395, 64396, 64397, 64398, 64399, 64400, 64401, 64402, 64403, 64404, 64405, 64406, 64407, 64408, 64409, 64410, 64411, 64412, 64413, 64414, 64415, 64416, 64417, 64418, 64419, 64420, 64421, 64422, 64423, 64424, 64425, 64426, 64427, 64428, 64429, 64430, 64431, 64432, 64433, 64467, 64468, 64469, 64470, 64471, 64472, 64473, 64474, 64475, 64476, 64477, 64478, 64479, 64480, 64481, 64482, 64483, 64484, 64485, 64486, 64487, 64488, 64489, 64490, 64491, 64492, 64493, 64494, 64495, 64496, 64497, 64498, 64499, 64500, 64501, 64502, 64503, 64504, 64505, 64506, 64507, 64508, 64509, 64510, 64511, 64512, 64513, 64514, 64515, 64516, 64517, 64518, 64519, 64520, 64521, 64522, 64523, 64524, 64525, 64526, 64527, 64528, 64529, 64530, 64531, 64532, 64533, 64534, 64535, 64536, 64537, 64538, 64539, 64540, 64541, 64542, 64543, 64544, 64545, 64546, 64547, 64548, 64549, 64550, 64551, 64552, 64553, 64554, 64555, 64556, 64557, 64558, 64559, 64560, 64561, 64562, 64563, 64564, 64565, 64566, 64567, 64568, 64569, 64570, 64571, 64572, 64573, 64574, 64575, 64576, 64577, 64578, 64579, 64580, 64581, 64582, 64583, 64584, 64585, 64586, 64587, 64588, 64589, 64590, 64591, 64592, 64593, 64594, 64595, 64596, 64597, 64598, 64599, 64600, 64601, 64602, 64603, 64604, 64605, 64606, 64607, 64608, 64609, 64610, 64611, 64612, 64613, 64614, 64615, 64616, 64617, 64618, 64619, 64620, 64621, 64622, 64623, 64624, 64625, 64626, 64627, 64628, 64629, 64630, 64631, 64632, 64633, 64634, 64635, 64636, 64637, 64638, 64639, 64640, 64641, 64642, 64643, 64644, 64645, 64646, 64647, 64648, 64649, 64650, 64651, 64652, 64653, 64654, 64655, 64656, 64657, 64658, 64659, 64660, 64661, 64662, 64663, 64664, 64665, 64666, 64667, 64668, 64669, 64670, 64671, 64672, 64673, 64674, 64675, 64676, 64677, 64678, 64679, 64680, 64681, 64682, 64683, 64684, 64685, 64686, 64687, 64688, 64689, 64690, 64691, 64692, 64693, 64694, 64695, 64696, 64697, 64698, 64699, 64700, 64701, 64702, 64703, 64704, 64705, 64706, 64707, 64708, 64709, 64710, 64711, 64712, 64713, 64714, 64715, 64716, 64717, 64718, 64719, 64720, 64721, 64722, 64723, 64724, 64725, 64726, 64727, 64728, 64729, 64730, 64731, 64732, 64733, 64734, 64735, 64736, 64737, 64738, 64739, 64740, 64741, 64742, 64743, 64744, 64745, 64746, 64747, 64748, 64749, 64750, 64751, 64752, 64753, 64754, 64755, 64756, 64757, 64758, 64759, 64760, 64761, 64762, 64763, 64764, 64765, 64766, 64767, 64768, 64769, 64770, 64771, 64772, 64773, 64774, 64775, 64776, 64777, 64778, 64779, 64780, 64781, 64782, 64783, 64784, 64785, 64786, 64787, 64788, 64789, 64790, 64791, 64792, 64793, 64794, 64795, 64796, 64797, 64798, 64799, 64800, 64801, 64802, 64803, 64804, 64805, 64806, 64807, 64808, 64809, 64810, 64811, 64812, 64813, 64814, 64815, 64816, 64817, 64818, 64819, 64820, 64821, 64822, 64823, 64824, 64825, 64826, 64827, 64828, 64829, 64848, 64849, 64850, 64851, 64852, 64853, 64854, 64855, 64856, 64857, 64858, 64859, 64860, 64861, 64862, 64863, 64864, 64865, 64866, 64867, 64868, 64869, 64870, 64871, 64872, 64873, 64874, 64875, 64876, 64877, 64878, 64879, 64880, 64881, 64882, 64883, 64884, 64885, 64886, 64887, 64888, 64889, 64890, 64891, 64892, 64893, 64894, 64895, 64896, 64897, 64898, 64899, 64900, 64901, 64902, 64903, 64904, 64905, 64906, 64907, 64908, 64909, 64910, 64911, 64914, 64915, 64916, 64917, 64918, 64919, 64920, 64921, 64922, 64923, 64924, 64925, 64926, 64927, 64928, 64929, 64930, 64931, 64932, 64933, 64934, 64935, 64936, 64937, 64938, 64939, 64940, 64941, 64942, 64943, 64944, 64945, 64946, 64947, 64948, 64949, 64950, 64951, 64952, 64953, 64954, 64955, 64956, 64957, 64958, 64959, 64960, 64961, 64962, 64963, 64964, 64965, 64966, 64967, 65008, 65009, 65010, 65011, 65012, 65013, 65014, 65015, 65016, 65017, 65018, 65019, 65136, 65137, 65138, 65139, 65140, 65142, 65143, 65144, 65145, 65146, 65147, 65148, 65149, 65150, 65151, 65152, 65153, 65154, 65155, 65156, 65157, 65158, 65159, 65160, 65161, 65162, 65163, 65164, 65165, 65166, 65167, 65168, 65169, 65170, 65171, 65172, 65173, 65174, 65175, 65176, 65177, 65178, 65179, 65180, 65181, 65182, 65183, 65184, 65185, 65186, 65187, 65188, 65189, 65190, 65191, 65192, 65193, 65194, 65195, 65196, 65197, 65198, 65199, 65200, 65201, 65202, 65203, 65204, 65205, 65206, 65207, 65208, 65209, 65210, 65211, 65212, 65213, 65214, 65215, 65216, 65217, 65218, 65219, 65220, 65221, 65222, 65223, 65224, 65225, 65226, 65227, 65228, 65229, 65230, 65231, 65232, 65233, 65234, 65235, 65236, 65237, 65238, 65239, 65240, 65241, 65242, 65243, 65244, 65245, 65246, 65247, 65248, 65249, 65250, 65251, 65252, 65253, 65254, 65255, 65256, 65257, 65258, 65259, 65260, 65261, 65262, 65263, 65264, 65265, 65266, 65267, 65268, 65269, 65270, 65271, 65272, 65273, 65274, 65275, 65276, 65382, 65383, 65384, 65385, 65386, 65387, 65388, 65389, 65390, 65391, 65393, 65394, 65395, 65396, 65397, 65398, 65399, 65400, 65401, 65402, 65403, 65404, 65405, 65406, 65407, 65408, 65409, 65410, 65411, 65412, 65413, 65414, 65415, 65416, 65417, 65418, 65419, 65420, 65421, 65422, 65423, 65424, 65425, 65426, 65427, 65428, 65429, 65430, 65431, 65432, 65433, 65434, 65435, 65436, 65437, 65440, 65441, 65442, 65443, 65444, 65445, 65446, 65447, 65448, 65449, 65450, 65451, 65452, 65453, 65454, 65455, 65456, 65457, 65458, 65459, 65460, 65461, 65462, 65463, 65464, 65465, 65466, 65467, 65468, 65469, 65470, 65474, 65475, 65476, 65477, 65478, 65479, 65482, 65483, 65484, 65485, 65486, 65487, 65490, 65491, 65492, 65493, 65494, 65495, 65498, 65499, 65500, 65536, 65537, 65538, 65539, 65540, 65541, 65542, 65543, 65544, 65545, 65546, 65547, 65549, 65550, 65551, 65552, 65553, 65554, 65555, 65556, 65557, 65558, 65559, 65560, 65561, 65562, 65563, 65564, 65565, 65566, 65567, 65568, 65569, 65570, 65571, 65572, 65573, 65574, 65576, 65577, 65578, 65579, 65580, 65581, 65582, 65583, 65584, 65585, 65586, 65587, 65588, 65589, 65590, 65591, 65592, 65593, 65594, 65596, 65597, 65599, 65600, 65601, 65602, 65603, 65604, 65605, 65606, 65607, 65608, 65609, 65610, 65611, 65612, 65613, 65616, 65617, 65618, 65619, 65620, 65621, 65622, 65623, 65624, 65625, 65626, 65627, 65628, 65629, 65664, 65665, 65666, 65667, 65668, 65669, 65670, 65671, 65672, 65673, 65674, 65675, 65676, 65677, 65678, 65679, 65680, 65681, 65682, 65683, 65684, 65685, 65686, 65687, 65688, 65689, 65690, 65691, 65692, 65693, 65694, 65695, 65696, 65697, 65698, 65699, 65700, 65701, 65702, 65703, 65704, 65705, 65706, 65707, 65708, 65709, 65710, 65711, 65712, 65713, 65714, 65715, 65716, 65717, 65718, 65719, 65720, 65721, 65722, 65723, 65724, 65725, 65726, 65727, 65728, 65729, 65730, 65731, 65732, 65733, 65734, 65735, 65736, 65737, 65738, 65739, 65740, 65741, 65742, 65743, 65744, 65745, 65746, 65747, 65748, 65749, 65750, 65751, 65752, 65753, 65754, 65755, 65756, 65757, 65758, 65759, 65760, 65761, 65762, 65763, 65764, 65765, 65766, 65767, 65768, 65769, 65770, 65771, 65772, 65773, 65774, 65775, 65776, 65777, 65778, 65779, 65780, 65781, 65782, 65783, 65784, 65785, 65786, 66176, 66177, 66178, 66179, 66180, 66181, 66182, 66183, 66184, 66185, 66186, 66187, 66188, 66189, 66190, 66191, 66192, 66193, 66194, 66195, 66196, 66197, 66198, 66199, 66200, 66201, 66202, 66203, 66204, 66208, 66209, 66210, 66211, 66212, 66213, 66214, 66215, 66216, 66217, 66218, 66219, 66220, 66221, 66222, 66223, 66224, 66225, 66226, 66227, 66228, 66229, 66230, 66231, 66232, 66233, 66234, 66235, 66236, 66237, 66238, 66239, 66240, 66241, 66242, 66243, 66244, 66245, 66246, 66247, 66248, 66249, 66250, 66251, 66252, 66253, 66254, 66255, 66256, 66304, 66305, 66306, 66307, 66308, 66309, 66310, 66311, 66312, 66313, 66314, 66315, 66316, 66317, 66318, 66319, 66320, 66321, 66322, 66323, 66324, 66325, 66326, 66327, 66328, 66329, 66330, 66331, 66332, 66333, 66334, 66352, 66353, 66354, 66355, 66356, 66357, 66358, 66359, 66360, 66361, 66362, 66363, 66364, 66365, 66366, 66367, 66368, 66370, 66371, 66372, 66373, 66374, 66375, 66376, 66377, 66432, 66433, 66434, 66435, 66436, 66437, 66438, 66439, 66440, 66441, 66442, 66443, 66444, 66445, 66446, 66447, 66448, 66449, 66450, 66451, 66452, 66453, 66454, 66455, 66456, 66457, 66458, 66459, 66460, 66461, 66464, 66465, 66466, 66467, 66468, 66469, 66470, 66471, 66472, 66473, 66474, 66475, 66476, 66477, 66478, 66479, 66480, 66481, 66482, 66483, 66484, 66485, 66486, 66487, 66488, 66489, 66490, 66491, 66492, 66493, 66494, 66495, 66496, 66497, 66498, 66499, 66504, 66505, 66506, 66507, 66508, 66509, 66510, 66511, 66640, 66641, 66642, 66643, 66644, 66645, 66646, 66647, 66648, 66649, 66650, 66651, 66652, 66653, 66654, 66655, 66656, 66657, 66658, 66659, 66660, 66661, 66662, 66663, 66664, 66665, 66666, 66667, 66668, 66669, 66670, 66671, 66672, 66673, 66674, 66675, 66676, 66677, 66678, 66679, 66680, 66681, 66682, 66683, 66684, 66685, 66686, 66687, 66688, 66689, 66690, 66691, 66692, 66693, 66694, 66695, 66696, 66697, 66698, 66699, 66700, 66701, 66702, 66703, 66704, 66705, 66706, 66707, 66708, 66709, 66710, 66711, 66712, 66713, 66714, 66715, 66716, 66717, 67584, 67585, 67586, 67587, 67588, 67589, 67592, 67594, 67595, 67596, 67597, 67598, 67599, 67600, 67601, 67602, 67603, 67604, 67605, 67606, 67607, 67608, 67609, 67610, 67611, 67612, 67613, 67614, 67615, 67616, 67617, 67618, 67619, 67620, 67621, 67622, 67623, 67624, 67625, 67626, 67627, 67628, 67629, 67630, 67631, 67632, 67633, 67634, 67635, 67636, 67637, 67639, 67640, 67644, 67647, 67648, 67649, 67650, 67651, 67652, 67653, 67654, 67655, 67656, 67657, 67658, 67659, 67660, 67661, 67662, 67663, 67664, 67665, 67666, 67667, 67668, 67669, 67840, 67841, 67842, 67843, 67844, 67845, 67846, 67847, 67848, 67849, 67850, 67851, 67852, 67853, 67854, 67855, 67856, 67857, 67858, 67859, 67860, 67861, 67872, 67873, 67874, 67875, 67876, 67877, 67878, 67879, 67880, 67881, 67882, 67883, 67884, 67885, 67886, 67887, 67888, 67889, 67890, 67891, 67892, 67893, 67894, 67895, 67896, 67897, 68096, 68112, 68113, 68114, 68115, 68117, 68118, 68119, 68121, 68122, 68123, 68124, 68125, 68126, 68127, 68128, 68129, 68130, 68131, 68132, 68133, 68134, 68135, 68136, 68137, 68138, 68139, 68140, 68141, 68142, 68143, 68144, 68145, 68146, 68147, 68192, 68193, 68194, 68195, 68196, 68197, 68198, 68199, 68200, 68201, 68202, 68203, 68204, 68205, 68206, 68207, 68208, 68209, 68210, 68211, 68212, 68213, 68214, 68215, 68216, 68217, 68218, 68219, 68220, 68352, 68353, 68354, 68355, 68356, 68357, 68358, 68359, 68360, 68361, 68362, 68363, 68364, 68365, 68366, 68367, 68368, 68369, 68370, 68371, 68372, 68373, 68374, 68375, 68376, 68377, 68378, 68379, 68380, 68381, 68382, 68383, 68384, 68385, 68386, 68387, 68388, 68389, 68390, 68391, 68392, 68393, 68394, 68395, 68396, 68397, 68398, 68399, 68400, 68401, 68402, 68403, 68404, 68405, 68416, 68417, 68418, 68419, 68420, 68421, 68422, 68423, 68424, 68425, 68426, 68427, 68428, 68429, 68430, 68431, 68432, 68433, 68434, 68435, 68436, 68437, 68448, 68449, 68450, 68451, 68452, 68453, 68454, 68455, 68456, 68457, 68458, 68459, 68460, 68461, 68462, 68463, 68464, 68465, 68466, 68608, 68609, 68610, 68611, 68612, 68613, 68614, 68615, 68616, 68617, 68618, 68619, 68620, 68621, 68622, 68623, 68624, 68625, 68626, 68627, 68628, 68629, 68630, 68631, 68632, 68633, 68634, 68635, 68636, 68637, 68638, 68639, 68640, 68641, 68642, 68643, 68644, 68645, 68646, 68647, 68648, 68649, 68650, 68651, 68652, 68653, 68654, 68655, 68656, 68657, 68658, 68659, 68660, 68661, 68662, 68663, 68664, 68665, 68666, 68667, 68668, 68669, 68670, 68671, 68672, 68673, 68674, 68675, 68676, 68677, 68678, 68679, 68680, 69635, 69636, 69637, 69638, 69639, 69640, 69641, 69642, 69643, 69644, 69645, 69646, 69647, 69648, 69649, 69650, 69651, 69652, 69653, 69654, 69655, 69656, 69657, 69658, 69659, 69660, 69661, 69662, 69663, 69664, 69665, 69666, 69667, 69668, 69669, 69670, 69671, 69672, 69673, 69674, 69675, 69676, 69677, 69678, 69679, 69680, 69681, 69682, 69683, 69684, 69685, 69686, 69687, 69763, 69764, 69765, 69766, 69767, 69768, 69769, 69770, 69771, 69772, 69773, 69774, 69775, 69776, 69777, 69778, 69779, 69780, 69781, 69782, 69783, 69784, 69785, 69786, 69787, 69788, 69789, 69790, 69791, 69792, 69793, 69794, 69795, 69796, 69797, 69798, 69799, 69800, 69801, 69802, 69803, 69804, 69805, 69806, 69807, 73728, 73729, 73730, 73731, 73732, 73733, 73734, 73735, 73736, 73737, 73738, 73739, 73740, 73741, 73742, 73743, 73744, 73745, 73746, 73747, 73748, 73749, 73750, 73751, 73752, 73753, 73754, 73755, 73756, 73757, 73758, 73759, 73760, 73761, 73762, 73763, 73764, 73765, 73766, 73767, 73768, 73769, 73770, 73771, 73772, 73773, 73774, 73775, 73776, 73777, 73778, 73779, 73780, 73781, 73782, 73783, 73784, 73785, 73786, 73787, 73788, 73789, 73790, 73791, 73792, 73793, 73794, 73795, 73796, 73797, 73798, 73799, 73800, 73801, 73802, 73803, 73804, 73805, 73806, 73807, 73808, 73809, 73810, 73811, 73812, 73813, 73814, 73815, 73816, 73817, 73818, 73819, 73820, 73821, 73822, 73823, 73824, 73825, 73826, 73827, 73828, 73829, 73830, 73831, 73832, 73833, 73834, 73835, 73836, 73837, 73838, 73839, 73840, 73841, 73842, 73843, 73844, 73845, 73846, 73847, 73848, 73849, 73850, 73851, 73852, 73853, 73854, 73855, 73856, 73857, 73858, 73859, 73860, 73861, 73862, 73863, 73864, 73865, 73866, 73867, 73868, 73869, 73870, 73871, 73872, 73873, 73874, 73875, 73876, 73877, 73878, 73879, 73880, 73881, 73882, 73883, 73884, 73885, 73886, 73887, 73888, 73889, 73890, 73891, 73892, 73893, 73894, 73895, 73896, 73897, 73898, 73899, 73900, 73901, 73902, 73903, 73904, 73905, 73906, 73907, 73908, 73909, 73910, 73911, 73912, 73913, 73914, 73915, 73916, 73917, 73918, 73919, 73920, 73921, 73922, 73923, 73924, 73925, 73926, 73927, 73928, 73929, 73930, 73931, 73932, 73933, 73934, 73935, 73936, 73937, 73938, 73939, 73940, 73941, 73942, 73943, 73944, 73945, 73946, 73947, 73948, 73949, 73950, 73951, 73952, 73953, 73954, 73955, 73956, 73957, 73958, 73959, 73960, 73961, 73962, 73963, 73964, 73965, 73966, 73967, 73968, 73969, 73970, 73971, 73972, 73973, 73974, 73975, 73976, 73977, 73978, 73979, 73980, 73981, 73982, 73983, 73984, 73985, 73986, 73987, 73988, 73989, 73990, 73991, 73992, 73993, 73994, 73995, 73996, 73997, 73998, 73999, 74000, 74001, 74002, 74003, 74004, 74005, 74006, 74007, 74008, 74009, 74010, 74011, 74012, 74013, 74014, 74015, 74016, 74017, 74018, 74019, 74020, 74021, 74022, 74023, 74024, 74025, 74026, 74027, 74028, 74029, 74030, 74031, 74032, 74033, 74034, 74035, 74036, 74037, 74038, 74039, 74040, 74041, 74042, 74043, 74044, 74045, 74046, 74047, 74048, 74049, 74050, 74051, 74052, 74053, 74054, 74055, 74056, 74057, 74058, 74059, 74060, 74061, 74062, 74063, 74064, 74065, 74066, 74067, 74068, 74069, 74070, 74071, 74072, 74073, 74074, 74075, 74076, 74077, 74078, 74079, 74080, 74081, 74082, 74083, 74084, 74085, 74086, 74087, 74088, 74089, 74090, 74091, 74092, 74093, 74094, 74095, 74096, 74097, 74098, 74099, 74100, 74101, 74102, 74103, 74104, 74105, 74106, 74107, 74108, 74109, 74110, 74111, 74112, 74113, 74114, 74115, 74116, 74117, 74118, 74119, 74120, 74121, 74122, 74123, 74124, 74125, 74126, 74127, 74128, 74129, 74130, 74131, 74132, 74133, 74134, 74135, 74136, 74137, 74138, 74139, 74140, 74141, 74142, 74143, 74144, 74145, 74146, 74147, 74148, 74149, 74150, 74151, 74152, 74153, 74154, 74155, 74156, 74157, 74158, 74159, 74160, 74161, 74162, 74163, 74164, 74165, 74166, 74167, 74168, 74169, 74170, 74171, 74172, 74173, 74174, 74175, 74176, 74177, 74178, 74179, 74180, 74181, 74182, 74183, 74184, 74185, 74186, 74187, 74188, 74189, 74190, 74191, 74192, 74193, 74194, 74195, 74196, 74197, 74198, 74199, 74200, 74201, 74202, 74203, 74204, 74205, 74206, 74207, 74208, 74209, 74210, 74211, 74212, 74213, 74214, 74215, 74216, 74217, 74218, 74219, 74220, 74221, 74222, 74223, 74224, 74225, 74226, 74227, 74228, 74229, 74230, 74231, 74232, 74233, 74234, 74235, 74236, 74237, 74238, 74239, 74240, 74241, 74242, 74243, 74244, 74245, 74246, 74247, 74248, 74249, 74250, 74251, 74252, 74253, 74254, 74255, 74256, 74257, 74258, 74259, 74260, 74261, 74262, 74263, 74264, 74265, 74266, 74267, 74268, 74269, 74270, 74271, 74272, 74273, 74274, 74275, 74276, 74277, 74278, 74279, 74280, 74281, 74282, 74283, 74284, 74285, 74286, 74287, 74288, 74289, 74290, 74291, 74292, 74293, 74294, 74295, 74296, 74297, 74298, 74299, 74300, 74301, 74302, 74303, 74304, 74305, 74306, 74307, 74308, 74309, 74310, 74311, 74312, 74313, 74314, 74315, 74316, 74317, 74318, 74319, 74320, 74321, 74322, 74323, 74324, 74325, 74326, 74327, 74328, 74329, 74330, 74331, 74332, 74333, 74334, 74335, 74336, 74337, 74338, 74339, 74340, 74341, 74342, 74343, 74344, 74345, 74346, 74347, 74348, 74349, 74350, 74351, 74352, 74353, 74354, 74355, 74356, 74357, 74358, 74359, 74360, 74361, 74362, 74363, 74364, 74365, 74366, 74367, 74368, 74369, 74370, 74371, 74372, 74373, 74374, 74375, 74376, 74377, 74378, 74379, 74380, 74381, 74382, 74383, 74384, 74385, 74386, 74387, 74388, 74389, 74390, 74391, 74392, 74393, 74394, 74395, 74396, 74397, 74398, 74399, 74400, 74401, 74402, 74403, 74404, 74405, 74406, 74407, 74408, 74409, 74410, 74411, 74412, 74413, 74414, 74415, 74416, 74417, 74418, 74419, 74420, 74421, 74422, 74423, 74424, 74425, 74426, 74427, 74428, 74429, 74430, 74431, 74432, 74433, 74434, 74435, 74436, 74437, 74438, 74439, 74440, 74441, 74442, 74443, 74444, 74445, 74446, 74447, 74448, 74449, 74450, 74451, 74452, 74453, 74454, 74455, 74456, 74457, 74458, 74459, 74460, 74461, 74462, 74463, 74464, 74465, 74466, 74467, 74468, 74469, 74470, 74471, 74472, 74473, 74474, 74475, 74476, 74477, 74478, 74479, 74480, 74481, 74482, 74483, 74484, 74485, 74486, 74487, 74488, 74489, 74490, 74491, 74492, 74493, 74494, 74495, 74496, 74497, 74498, 74499, 74500, 74501, 74502, 74503, 74504, 74505, 74506, 74507, 74508, 74509, 74510, 74511, 74512, 74513, 74514, 74515, 74516, 74517, 74518, 74519, 74520, 74521, 74522, 74523, 74524, 74525, 74526, 74527, 74528, 74529, 74530, 74531, 74532, 74533, 74534, 74535, 74536, 74537, 74538, 74539, 74540, 74541, 74542, 74543, 74544, 74545, 74546, 74547, 74548, 74549, 74550, 74551, 74552, 74553, 74554, 74555, 74556, 74557, 74558, 74559, 74560, 74561, 74562, 74563, 74564, 74565, 74566, 74567, 74568, 74569, 74570, 74571, 74572, 74573, 74574, 74575, 74576, 74577, 74578, 74579, 74580, 74581, 74582, 74583, 74584, 74585, 74586, 74587, 74588, 74589, 74590, 74591, 74592, 74593, 74594, 74595, 74596, 74597, 74598, 74599, 74600, 74601, 74602, 74603, 74604, 74605, 74606, 77824, 77825, 77826, 77827, 77828, 77829, 77830, 77831, 77832, 77833, 77834, 77835, 77836, 77837, 77838, 77839, 77840, 77841, 77842, 77843, 77844, 77845, 77846, 77847, 77848, 77849, 77850, 77851, 77852, 77853, 77854, 77855, 77856, 77857, 77858, 77859, 77860, 77861, 77862, 77863, 77864, 77865, 77866, 77867, 77868, 77869, 77870, 77871, 77872, 77873, 77874, 77875, 77876, 77877, 77878, 77879, 77880, 77881, 77882, 77883, 77884, 77885, 77886, 77887, 77888, 77889, 77890, 77891, 77892, 77893, 77894, 77895, 77896, 77897, 77898, 77899, 77900, 77901, 77902, 77903, 77904, 77905, 77906, 77907, 77908, 77909, 77910, 77911, 77912, 77913, 77914, 77915, 77916, 77917, 77918, 77919, 77920, 77921, 77922, 77923, 77924, 77925, 77926, 77927, 77928, 77929, 77930, 77931, 77932, 77933, 77934, 77935, 77936, 77937, 77938, 77939, 77940, 77941, 77942, 77943, 77944, 77945, 77946, 77947, 77948, 77949, 77950, 77951, 77952, 77953, 77954, 77955, 77956, 77957, 77958, 77959, 77960, 77961, 77962, 77963, 77964, 77965, 77966, 77967, 77968, 77969, 77970, 77971, 77972, 77973, 77974, 77975, 77976, 77977, 77978, 77979, 77980, 77981, 77982, 77983, 77984, 77985, 77986, 77987, 77988, 77989, 77990, 77991, 77992, 77993, 77994, 77995, 77996, 77997, 77998, 77999, 78000, 78001, 78002, 78003, 78004, 78005, 78006, 78007, 78008, 78009, 78010, 78011, 78012, 78013, 78014, 78015, 78016, 78017, 78018, 78019, 78020, 78021, 78022, 78023, 78024, 78025, 78026, 78027, 78028, 78029, 78030, 78031, 78032, 78033, 78034, 78035, 78036, 78037, 78038, 78039, 78040, 78041, 78042, 78043, 78044, 78045, 78046, 78047, 78048, 78049, 78050, 78051, 78052, 78053, 78054, 78055, 78056, 78057, 78058, 78059, 78060, 78061, 78062, 78063, 78064, 78065, 78066, 78067, 78068, 78069, 78070, 78071, 78072, 78073, 78074, 78075, 78076, 78077, 78078, 78079, 78080, 78081, 78082, 78083, 78084, 78085, 78086, 78087, 78088, 78089, 78090, 78091, 78092, 78093, 78094, 78095, 78096, 78097, 78098, 78099, 78100, 78101, 78102, 78103, 78104, 78105, 78106, 78107, 78108, 78109, 78110, 78111, 78112, 78113, 78114, 78115, 78116, 78117, 78118, 78119, 78120, 78121, 78122, 78123, 78124, 78125, 78126, 78127, 78128, 78129, 78130, 78131, 78132, 78133, 78134, 78135, 78136, 78137, 78138, 78139, 78140, 78141, 78142, 78143, 78144, 78145, 78146, 78147, 78148, 78149, 78150, 78151, 78152, 78153, 78154, 78155, 78156, 78157, 78158, 78159, 78160, 78161, 78162, 78163, 78164, 78165, 78166, 78167, 78168, 78169, 78170, 78171, 78172, 78173, 78174, 78175, 78176, 78177, 78178, 78179, 78180, 78181, 78182, 78183, 78184, 78185, 78186, 78187, 78188, 78189, 78190, 78191, 78192, 78193, 78194, 78195, 78196, 78197, 78198, 78199, 78200, 78201, 78202, 78203, 78204, 78205, 78206, 78207, 78208, 78209, 78210, 78211, 78212, 78213, 78214, 78215, 78216, 78217, 78218, 78219, 78220, 78221, 78222, 78223, 78224, 78225, 78226, 78227, 78228, 78229, 78230, 78231, 78232, 78233, 78234, 78235, 78236, 78237, 78238, 78239, 78240, 78241, 78242, 78243, 78244, 78245, 78246, 78247, 78248, 78249, 78250, 78251, 78252, 78253, 78254, 78255, 78256, 78257, 78258, 78259, 78260, 78261, 78262, 78263, 78264, 78265, 78266, 78267, 78268, 78269, 78270, 78271, 78272, 78273, 78274, 78275, 78276, 78277, 78278, 78279, 78280, 78281, 78282, 78283, 78284, 78285, 78286, 78287, 78288, 78289, 78290, 78291, 78292, 78293, 78294, 78295, 78296, 78297, 78298, 78299, 78300, 78301, 78302, 78303, 78304, 78305, 78306, 78307, 78308, 78309, 78310, 78311, 78312, 78313, 78314, 78315, 78316, 78317, 78318, 78319, 78320, 78321, 78322, 78323, 78324, 78325, 78326, 78327, 78328, 78329, 78330, 78331, 78332, 78333, 78334, 78335, 78336, 78337, 78338, 78339, 78340, 78341, 78342, 78343, 78344, 78345, 78346, 78347, 78348, 78349, 78350, 78351, 78352, 78353, 78354, 78355, 78356, 78357, 78358, 78359, 78360, 78361, 78362, 78363, 78364, 78365, 78366, 78367, 78368, 78369, 78370, 78371, 78372, 78373, 78374, 78375, 78376, 78377, 78378, 78379, 78380, 78381, 78382, 78383, 78384, 78385, 78386, 78387, 78388, 78389, 78390, 78391, 78392, 78393, 78394, 78395, 78396, 78397, 78398, 78399, 78400, 78401, 78402, 78403, 78404, 78405, 78406, 78407, 78408, 78409, 78410, 78411, 78412, 78413, 78414, 78415, 78416, 78417, 78418, 78419, 78420, 78421, 78422, 78423, 78424, 78425, 78426, 78427, 78428, 78429, 78430, 78431, 78432, 78433, 78434, 78435, 78436, 78437, 78438, 78439, 78440, 78441, 78442, 78443, 78444, 78445, 78446, 78447, 78448, 78449, 78450, 78451, 78452, 78453, 78454, 78455, 78456, 78457, 78458, 78459, 78460, 78461, 78462, 78463, 78464, 78465, 78466, 78467, 78468, 78469, 78470, 78471, 78472, 78473, 78474, 78475, 78476, 78477, 78478, 78479, 78480, 78481, 78482, 78483, 78484, 78485, 78486, 78487, 78488, 78489, 78490, 78491, 78492, 78493, 78494, 78495, 78496, 78497, 78498, 78499, 78500, 78501, 78502, 78503, 78504, 78505, 78506, 78507, 78508, 78509, 78510, 78511, 78512, 78513, 78514, 78515, 78516, 78517, 78518, 78519, 78520, 78521, 78522, 78523, 78524, 78525, 78526, 78527, 78528, 78529, 78530, 78531, 78532, 78533, 78534, 78535, 78536, 78537, 78538, 78539, 78540, 78541, 78542, 78543, 78544, 78545, 78546, 78547, 78548, 78549, 78550, 78551, 78552, 78553, 78554, 78555, 78556, 78557, 78558, 78559, 78560, 78561, 78562, 78563, 78564, 78565, 78566, 78567, 78568, 78569, 78570, 78571, 78572, 78573, 78574, 78575, 78576, 78577, 78578, 78579, 78580, 78581, 78582, 78583, 78584, 78585, 78586, 78587, 78588, 78589, 78590, 78591, 78592, 78593, 78594, 78595, 78596, 78597, 78598, 78599, 78600, 78601, 78602, 78603, 78604, 78605, 78606, 78607, 78608, 78609, 78610, 78611, 78612, 78613, 78614, 78615, 78616, 78617, 78618, 78619, 78620, 78621, 78622, 78623, 78624, 78625, 78626, 78627, 78628, 78629, 78630, 78631, 78632, 78633, 78634, 78635, 78636, 78637, 78638, 78639, 78640, 78641, 78642, 78643, 78644, 78645, 78646, 78647, 78648, 78649, 78650, 78651, 78652, 78653, 78654, 78655, 78656, 78657, 78658, 78659, 78660, 78661, 78662, 78663, 78664, 78665, 78666, 78667, 78668, 78669, 78670, 78671, 78672, 78673, 78674, 78675, 78676, 78677, 78678, 78679, 78680, 78681, 78682, 78683, 78684, 78685, 78686, 78687, 78688, 78689, 78690, 78691, 78692, 78693, 78694, 78695, 78696, 78697, 78698, 78699, 78700, 78701, 78702, 78703, 78704, 78705, 78706, 78707, 78708, 78709, 78710, 78711, 78712, 78713, 78714, 78715, 78716, 78717, 78718, 78719, 78720, 78721, 78722, 78723, 78724, 78725, 78726, 78727, 78728, 78729, 78730, 78731, 78732, 78733, 78734, 78735, 78736, 78737, 78738, 78739, 78740, 78741, 78742, 78743, 78744, 78745, 78746, 78747, 78748, 78749, 78750, 78751, 78752, 78753, 78754, 78755, 78756, 78757, 78758, 78759, 78760, 78761, 78762, 78763, 78764, 78765, 78766, 78767, 78768, 78769, 78770, 78771, 78772, 78773, 78774, 78775, 78776, 78777, 78778, 78779, 78780, 78781, 78782, 78783, 78784, 78785, 78786, 78787, 78788, 78789, 78790, 78791, 78792, 78793, 78794, 78795, 78796, 78797, 78798, 78799, 78800, 78801, 78802, 78803, 78804, 78805, 78806, 78807, 78808, 78809, 78810, 78811, 78812, 78813, 78814, 78815, 78816, 78817, 78818, 78819, 78820, 78821, 78822, 78823, 78824, 78825, 78826, 78827, 78828, 78829, 78830, 78831, 78832, 78833, 78834, 78835, 78836, 78837, 78838, 78839, 78840, 78841, 78842, 78843, 78844, 78845, 78846, 78847, 78848, 78849, 78850, 78851, 78852, 78853, 78854, 78855, 78856, 78857, 78858, 78859, 78860, 78861, 78862, 78863, 78864, 78865, 78866, 78867, 78868, 78869, 78870, 78871, 78872, 78873, 78874, 78875, 78876, 78877, 78878, 78879, 78880, 78881, 78882, 78883, 78884, 78885, 78886, 78887, 78888, 78889, 78890, 78891, 78892, 78893, 78894, 92160, 92161, 92162, 92163, 92164, 92165, 92166, 92167, 92168, 92169, 92170, 92171, 92172, 92173, 92174, 92175, 92176, 92177, 92178, 92179, 92180, 92181, 92182, 92183, 92184, 92185, 92186, 92187, 92188, 92189, 92190, 92191, 92192, 92193, 92194, 92195, 92196, 92197, 92198, 92199, 92200, 92201, 92202, 92203, 92204, 92205, 92206, 92207, 92208, 92209, 92210, 92211, 92212, 92213, 92214, 92215, 92216, 92217, 92218, 92219, 92220, 92221, 92222, 92223, 92224, 92225, 92226, 92227, 92228, 92229, 92230, 92231, 92232, 92233, 92234, 92235, 92236, 92237, 92238, 92239, 92240, 92241, 92242, 92243, 92244, 92245, 92246, 92247, 92248, 92249, 92250, 92251, 92252, 92253, 92254, 92255, 92256, 92257, 92258, 92259, 92260, 92261, 92262, 92263, 92264, 92265, 92266, 92267, 92268, 92269, 92270, 92271, 92272, 92273, 92274, 92275, 92276, 92277, 92278, 92279, 92280, 92281, 92282, 92283, 92284, 92285, 92286, 92287, 92288, 92289, 92290, 92291, 92292, 92293, 92294, 92295, 92296, 92297, 92298, 92299, 92300, 92301, 92302, 92303, 92304, 92305, 92306, 92307, 92308, 92309, 92310, 92311, 92312, 92313, 92314, 92315, 92316, 92317, 92318, 92319, 92320, 92321, 92322, 92323, 92324, 92325, 92326, 92327, 92328, 92329, 92330, 92331, 92332, 92333, 92334, 92335, 92336, 92337, 92338, 92339, 92340, 92341, 92342, 92343, 92344, 92345, 92346, 92347, 92348, 92349, 92350, 92351, 92352, 92353, 92354, 92355, 92356, 92357, 92358, 92359, 92360, 92361, 92362, 92363, 92364, 92365, 92366, 92367, 92368, 92369, 92370, 92371, 92372, 92373, 92374, 92375, 92376, 92377, 92378, 92379, 92380, 92381, 92382, 92383, 92384, 92385, 92386, 92387, 92388, 92389, 92390, 92391, 92392, 92393, 92394, 92395, 92396, 92397, 92398, 92399, 92400, 92401, 92402, 92403, 92404, 92405, 92406, 92407, 92408, 92409, 92410, 92411, 92412, 92413, 92414, 92415, 92416, 92417, 92418, 92419, 92420, 92421, 92422, 92423, 92424, 92425, 92426, 92427, 92428, 92429, 92430, 92431, 92432, 92433, 92434, 92435, 92436, 92437, 92438, 92439, 92440, 92441, 92442, 92443, 92444, 92445, 92446, 92447, 92448, 92449, 92450, 92451, 92452, 92453, 92454, 92455, 92456, 92457, 92458, 92459, 92460, 92461, 92462, 92463, 92464, 92465, 92466, 92467, 92468, 92469, 92470, 92471, 92472, 92473, 92474, 92475, 92476, 92477, 92478, 92479, 92480, 92481, 92482, 92483, 92484, 92485, 92486, 92487, 92488, 92489, 92490, 92491, 92492, 92493, 92494, 92495, 92496, 92497, 92498, 92499, 92500, 92501, 92502, 92503, 92504, 92505, 92506, 92507, 92508, 92509, 92510, 92511, 92512, 92513, 92514, 92515, 92516, 92517, 92518, 92519, 92520, 92521, 92522, 92523, 92524, 92525, 92526, 92527, 92528, 92529, 92530, 92531, 92532, 92533, 92534, 92535, 92536, 92537, 92538, 92539, 92540, 92541, 92542, 92543, 92544, 92545, 92546, 92547, 92548, 92549, 92550, 92551, 92552, 92553, 92554, 92555, 92556, 92557, 92558, 92559, 92560, 92561, 92562, 92563, 92564, 92565, 92566, 92567, 92568, 92569, 92570, 92571, 92572, 92573, 92574, 92575, 92576, 92577, 92578, 92579, 92580, 92581, 92582, 92583, 92584, 92585, 92586, 92587, 92588, 92589, 92590, 92591, 92592, 92593, 92594, 92595, 92596, 92597, 92598, 92599, 92600, 92601, 92602, 92603, 92604, 92605, 92606, 92607, 92608, 92609, 92610, 92611, 92612, 92613, 92614, 92615, 92616, 92617, 92618, 92619, 92620, 92621, 92622, 92623, 92624, 92625, 92626, 92627, 92628, 92629, 92630, 92631, 92632, 92633, 92634, 92635, 92636, 92637, 92638, 92639, 92640, 92641, 92642, 92643, 92644, 92645, 92646, 92647, 92648, 92649, 92650, 92651, 92652, 92653, 92654, 92655, 92656, 92657, 92658, 92659, 92660, 92661, 92662, 92663, 92664, 92665, 92666, 92667, 92668, 92669, 92670, 92671, 92672, 92673, 92674, 92675, 92676, 92677, 92678, 92679, 92680, 92681, 92682, 92683, 92684, 92685, 92686, 92687, 92688, 92689, 92690, 92691, 92692, 92693, 92694, 92695, 92696, 92697, 92698, 92699, 92700, 92701, 92702, 92703, 92704, 92705, 92706, 92707, 92708, 92709, 92710, 92711, 92712, 92713, 92714, 92715, 92716, 92717, 92718, 92719, 92720, 92721, 92722, 92723, 92724, 92725, 92726, 92727, 92728, 110592, 110593, 131072, 173782, 173824, 177972, 177984, 178205, 20029, 20024, 20033, 131362, 20320, 20398, 20411, 20482, 20602, 20633, 20711, 20687, 13470, 132666, 20813, 20820, 20836, 20855, 132380, 13497, 20839, 20877, 132427, 20887, 20900, 20172, 20908, 20917, 168415, 20981, 20995, 13535, 21051, 21062, 21106, 21111, 13589, 21191, 21193, 21220, 21242, 21253, 21254, 21271, 21321, 21329, 21338, 21363, 21373, 21375, 21375, 21375, 133676, 28784, 21450, 21471, 133987, 21483, 21489, 21510, 21662, 21560, 21576, 21608, 21666, 21750, 21776, 21843, 21859, 21892, 21892, 21913, 21931, 21939, 21954, 22294, 22022, 22295, 22097, 22132, 20999, 22766, 22478, 22516, 22541, 22411, 22578, 22577, 22700, 136420, 22770, 22775, 22790, 22810, 22818, 22882, 136872, 136938, 23020, 23067, 23079, 23000, 23142, 14062, 14076, 23304, 23358, 23358, 137672, 23491, 23512, 23527, 23539, 138008, 23551, 23558, 24403, 23586, 14209, 23648, 23662, 23744, 23693, 138724, 23875, 138726, 23918, 23915, 23932, 24033, 24034, 14383, 24061, 24104, 24125, 24169, 14434, 139651, 14460, 24240, 24243, 24246, 24266, 172946, 24318, 140081, 140081, 33281, 24354, 24354, 14535, 144056, 156122, 24418, 24427, 14563, 24474, 24525, 24535, 24569, 24705, 14650, 14620, 24724, 141012, 24775, 24904, 24908, 24910, 24908, 24954, 24974, 25010, 24996, 25007, 25054, 25074, 25078, 25104, 25115, 25181, 25265, 25300, 25424, 142092, 25405, 25340, 25448, 25475, 25572, 142321, 25634, 25541, 25513, 14894, 25705, 25726, 25757, 25719, 14956, 25935, 25964, 143370, 26083, 26360, 26185, 15129, 26257, 15112, 15076, 20882, 20885, 26368, 26268, 32941, 17369, 26391, 26395, 26401, 26462, 26451, 144323, 15177, 26618, 26501, 26706, 26757, 144493, 26766, 26655, 26900, 15261, 26946, 27043, 27114, 27304, 145059, 27355, 15384, 27425, 145575, 27476, 15438, 27506, 27551, 27578, 27579, 146061, 138507, 146170, 27726, 146620, 27839, 27853, 27751, 27926, 27966, 28023, 27969, 28009, 28024, 28037, 146718, 27956, 28207, 28270, 15667, 28363, 28359, 147153, 28153, 28526, 147294, 147342, 28614, 28729, 28702, 28699, 15766, 28746, 28797, 28791, 28845, 132389, 28997, 148067, 29084, 148395, 29224, 29237, 29264, 149000, 29312, 29333, 149301, 149524, 29562, 29579, 16044, 29605, 16056, 16056, 29767, 29788, 29809, 29829, 29898, 16155, 29988, 150582, 30014, 150674, 30064, 139679, 30224, 151457, 151480, 151620, 16380, 16392, 30452, 151795, 151794, 151833, 151859, 30494, 30495, 30495, 30538, 16441, 30603, 16454, 16534, 152605, 30798, 30860, 30924, 16611, 153126, 31062, 153242, 153285, 31119, 31211, 16687, 31296, 31306, 31311, 153980, 154279, 154279, 31470, 16898, 154539, 31686, 31689, 16935, 154752, 31954, 17056, 31976, 31971, 32000, 155526, 32099, 17153, 32199, 32258, 32325, 17204, 156200, 156231, 17241, 156377, 32634, 156478, 32661, 32762, 32773, 156890, 156963, 32864, 157096, 32880, 144223, 17365, 32946, 33027, 17419, 33086, 23221, 157607, 157621, 144275, 144284, 33281, 33284, 36766, 17515, 33425, 33419, 33437, 21171, 33457, 33459, 33469, 33510, 158524, 33509, 33565, 33635, 33709, 33571, 33725, 33767, 33879, 33619, 33738, 33740, 33756, 158774, 159083, 158933, 17707, 34033, 34035, 34070, 160714, 34148, 159532, 17757, 17761, 159665, 159954, 17771, 34384, 34396, 34407, 34409, 34473, 34440, 34574, 34530, 34681, 34600, 34667, 34694, 17879, 34785, 34817, 17913, 34912, 34915, 161383, 35031, 35038, 17973, 35066, 13499, 161966, 162150, 18110, 18119, 35488, 35565, 35722, 35925, 162984, 36011, 36033, 36123, 36215, 163631, 133124, 36299, 36284, 36336, 133342, 36564, 36664, 165330, 165357, 37012, 37105, 37137, 165678, 37147, 37432, 37591, 37592, 37500, 37881, 37909, 166906, 38283, 18837, 38327, 167287, 18918, 38595, 23986, 38691, 168261, 168474, 19054, 19062, 38880, 168970, 19122, 169110, 38923, 38923, 38953, 169398, 39138, 19251, 39209, 39335, 39362, 39422, 19406, 170800, 39698, 40000, 40189, 19662, 19693, 40295, 172238, 19704, 172293, 172558, 172689, 40635, 19798, 40697, 40702, 40709, 40719, 40726, 40763, 173568],
				'encoded': '\u01BB\u01C0\u01C1\u01C2\u01C3\u0294\u05D0\u05D1\u05D2\u05D3\u05D4\u05D5\u05D6\u05D7\u05D8\u05D9\u05DA\u05DB\u05DC\u05DD\u05DE\u05DF\u05E0\u05E1\u05E2\u05E3\u05E4\u05E5\u05E6\u05E7\u05E8\u05E9\u05EA\u05F0\u05F1\u05F2\u0620\u0621\u0622\u0623\u0624\u0625\u0626\u0627\u0628\u0629\u062A\u062B\u062C\u062D\u062E\u062F\u0630\u0631\u0632\u0633\u0634\u0635\u0636\u0637\u0638\u0639\u063A\u063B\u063C\u063D\u063E\u063F\u0641\u0642\u0643\u0644\u0645\u0646\u0647\u0648\u0649\u064A\u066E\u066F\u0671\u0672\u0673\u0674\u0675\u0676\u0677\u0678\u0679\u067A\u067B\u067C\u067D\u067E\u067F\u0680\u0681\u0682\u0683\u0684\u0685\u0686\u0687\u0688\u0689\u068A\u068B\u068C\u068D\u068E\u068F\u0690\u0691\u0692\u0693\u0694\u0695\u0696\u0697\u0698\u0699\u069A\u069B\u069C\u069D\u069E\u069F\u06A0\u06A1\u06A2\u06A3\u06A4\u06A5\u06A6\u06A7\u06A8\u06A9\u06AA\u06AB\u06AC\u06AD\u06AE\u06AF\u06B0\u06B1\u06B2\u06B3\u06B4\u06B5\u06B6\u06B7\u06B8\u06B9\u06BA\u06BB\u06BC\u06BD\u06BE\u06BF\u06C0\u06C1\u06C2\u06C3\u06C4\u06C5\u06C6\u06C7\u06C8\u06C9\u06CA\u06CB\u06CC\u06CD\u06CE\u06CF\u06D0\u06D1\u06D2\u06D3\u06D5\u06EE\u06EF\u06FA\u06FB\u06FC\u06FF\u0710\u0712\u0713\u0714\u0715\u0716\u0717\u0718\u0719\u071A\u071B\u071C\u071D\u071E\u071F\u0720\u0721\u0722\u0723\u0724\u0725\u0726\u0727\u0728\u0729\u072A\u072B\u072C\u072D\u072E\u072F\u074D\u074E\u074F\u0750\u0751\u0752\u0753\u0754\u0755\u0756\u0757\u0758\u0759\u075A\u075B\u075C\u075D\u075E\u075F\u0760\u0761\u0762\u0763\u0764\u0765\u0766\u0767\u0768\u0769\u076A\u076B\u076C\u076D\u076E\u076F\u0770\u0771\u0772\u0773\u0774\u0775\u0776\u0777\u0778\u0779\u077A\u077B\u077C\u077D\u077E\u077F\u0780\u0781\u0782\u0783\u0784\u0785\u0786\u0787\u0788\u0789\u078A\u078B\u078C\u078D\u078E\u078F\u0790\u0791\u0792\u0793\u0794\u0795\u0796\u0797\u0798\u0799\u079A\u079B\u079C\u079D\u079E\u079F\u07A0\u07A1\u07A2\u07A3\u07A4\u07A5\u07B1\u07CA\u07CB\u07CC\u07CD\u07CE\u07CF\u07D0\u07D1\u07D2\u07D3\u07D4\u07D5\u07D6\u07D7\u07D8\u07D9\u07DA\u07DB\u07DC\u07DD\u07DE\u07DF\u07E0\u07E1\u07E2\u07E3\u07E4\u07E5\u07E6\u07E7\u07E8\u07E9\u07EA\u0800\u0801\u0802\u0803\u0804\u0805\u0806\u0807\u0808\u0809\u080A\u080B\u080C\u080D\u080E\u080F\u0810\u0811\u0812\u0813\u0814\u0815\u0840\u0841\u0842\u0843\u0844\u0845\u0846\u0847\u0848\u0849\u084A\u084B\u084C\u084D\u084E\u084F\u0850\u0851\u0852\u0853\u0854\u0855\u0856\u0857\u0858\u0904\u0905\u0906\u0907\u0908\u0909\u090A\u090B\u090C\u090D\u090E\u090F\u0910\u0911\u0912\u0913\u0914\u0915\u0916\u0917\u0918\u0919\u091A\u091B\u091C\u091D\u091E\u091F\u0920\u0921\u0922\u0923\u0924\u0925\u0926\u0927\u0928\u0929\u092A\u092B\u092C\u092D\u092E\u092F\u0930\u0931\u0932\u0933\u0934\u0935\u0936\u0937\u0938\u0939\u093D\u0950\u0915\u093C\u0916\u093C\u0917\u093C\u091C\u093C\u0921\u093C\u0922\u093C\u092B\u093C\u092F\u093C\u0960\u0961\u0972\u0973\u0974\u0975\u0976\u0977\u0979\u097A\u097B\u097C\u097D\u097E\u097F\u0985\u0986\u0987\u0988\u0989\u098A\u098B\u098C\u098F\u0990\u0993\u0994\u0995\u0996\u0997\u0998\u0999\u099A\u099B\u099C\u099D\u099E\u099F\u09A0\u09A1\u09A2\u09A3\u09A4\u09A5\u09A6\u09A7\u09A8\u09AA\u09AB\u09AC\u09AD\u09AE\u09AF\u09B0\u09B2\u09B6\u09B7\u09B8\u09B9\u09BD\u09CE\u09A1\u09BC\u09A2\u09BC\u09AF\u09BC\u09E0\u09E1\u09F0\u09F1\u0A05\u0A06\u0A07\u0A08\u0A09\u0A0A\u0A0F\u0A10\u0A13\u0A14\u0A15\u0A16\u0A17\u0A18\u0A19\u0A1A\u0A1B\u0A1C\u0A1D\u0A1E\u0A1F\u0A20\u0A21\u0A22\u0A23\u0A24\u0A25\u0A26\u0A27\u0A28\u0A2A\u0A2B\u0A2C\u0A2D\u0A2E\u0A2F\u0A30\u0A32\u0A32\u0A3C\u0A35\u0A38\u0A3C\u0A38\u0A39\u0A16\u0A3C\u0A17\u0A3C\u0A1C\u0A3C\u0A5C\u0A2B\u0A3C\u0A72\u0A73\u0A74\u0A85\u0A86\u0A87\u0A88\u0A89\u0A8A\u0A8B\u0A8C\u0A8D\u0A8F\u0A90\u0A91\u0A93\u0A94\u0A95\u0A96\u0A97\u0A98\u0A99\u0A9A\u0A9B\u0A9C\u0A9D\u0A9E\u0A9F\u0AA0\u0AA1\u0AA2\u0AA3\u0AA4\u0AA5\u0AA6\u0AA7\u0AA8\u0AAA\u0AAB\u0AAC\u0AAD\u0AAE\u0AAF\u0AB0\u0AB2\u0AB3\u0AB5\u0AB6\u0AB7\u0AB8\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0B05\u0B06\u0B07\u0B08\u0B09\u0B0A\u0B0B\u0B0C\u0B0F\u0B10\u0B13\u0B14\u0B15\u0B16\u0B17\u0B18\u0B19\u0B1A\u0B1B\u0B1C\u0B1D\u0B1E\u0B1F\u0B20\u0B21\u0B22\u0B23\u0B24\u0B25\u0B26\u0B27\u0B28\u0B2A\u0B2B\u0B2C\u0B2D\u0B2E\u0B2F\u0B30\u0B32\u0B33\u0B35\u0B36\u0B37\u0B38\u0B39\u0B3D\u0B21\u0B3C\u0B22\u0B3C\u0B5F\u0B60\u0B61\u0B71\u0B83\u0B85\u0B86\u0B87\u0B88\u0B89\u0B8A\u0B8E\u0B8F\u0B90\u0B92\u0B93\u0B94\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8\u0BA9\u0BAA\u0BAE\u0BAF\u0BB0\u0BB1\u0BB2\u0BB3\u0BB4\u0BB5\u0BB6\u0BB7\u0BB8\u0BB9\u0BD0\u0C05\u0C06\u0C07\u0C08\u0C09\u0C0A\u0C0B\u0C0C\u0C0E\u0C0F\u0C10\u0C12\u0C13\u0C14\u0C15\u0C16\u0C17\u0C18\u0C19\u0C1A\u0C1B\u0C1C\u0C1D\u0C1E\u0C1F\u0C20\u0C21\u0C22\u0C23\u0C24\u0C25\u0C26\u0C27\u0C28\u0C2A\u0C2B\u0C2C\u0C2D\u0C2E\u0C2F\u0C30\u0C31\u0C32\u0C33\u0C35\u0C36\u0C37\u0C38\u0C39\u0C3D\u0C58\u0C59\u0C60\u0C61\u0C85\u0C86\u0C87\u0C88\u0C89\u0C8A\u0C8B\u0C8C\u0C8E\u0C8F\u0C90\u0C92\u0C93\u0C94\u0C95\u0C96\u0C97\u0C98\u0C99\u0C9A\u0C9B\u0C9C\u0C9D\u0C9E\u0C9F\u0CA0\u0CA1\u0CA2\u0CA3\u0CA4\u0CA5\u0CA6\u0CA7\u0CA8\u0CAA\u0CAB\u0CAC\u0CAD\u0CAE\u0CAF\u0CB0\u0CB1\u0CB2\u0CB3\u0CB5\u0CB6\u0CB7\u0CB8\u0CB9\u0CBD\u0CDE\u0CE0\u0CE1\u0CF1\u0CF2\u0D05\u0D06\u0D07\u0D08\u0D09\u0D0A\u0D0B\u0D0C\u0D0E\u0D0F\u0D10\u0D12\u0D13\u0D14\u0D15\u0D16\u0D17\u0D18\u0D19\u0D1A\u0D1B\u0D1C\u0D1D\u0D1E\u0D1F\u0D20\u0D21\u0D22\u0D23\u0D24\u0D25\u0D26\u0D27\u0D28\u0D29\u0D2A\u0D2B\u0D2C\u0D2D\u0D2E\u0D2F\u0D30\u0D31\u0D32\u0D33\u0D34\u0D35\u0D36\u0D37\u0D38\u0D39\u0D3A\u0D3D\u0D4E\u0D60\u0D61\u0D7A\u0D7B\u0D7C\u0D7D\u0D7E\u0D7F\u0D85\u0D86\u0D87\u0D88\u0D89\u0D8A\u0D8B\u0D8C\u0D8D\u0D8E\u0D8F\u0D90\u0D91\u0D92\u0D93\u0D94\u0D95\u0D96\u0D9A\u0D9B\u0D9C\u0D9D\u0D9E\u0D9F\u0DA0\u0DA1\u0DA2\u0DA3\u0DA4\u0DA5\u0DA6\u0DA7\u0DA8\u0DA9\u0DAA\u0DAB\u0DAC\u0DAD\u0DAE\u0DAF\u0DB0\u0DB1\u0DB3\u0DB4\u0DB5\u0DB6\u0DB7\u0DB8\u0DB9\u0DBA\u0DBB\u0DBD\u0DC0\u0DC1\u0DC2\u0DC3\u0DC4\u0DC5\u0DC6\u0E01\u0E02\u0E03\u0E04\u0E05\u0E06\u0E07\u0E08\u0E09\u0E0A\u0E0B\u0E0C\u0E0D\u0E0E\u0E0F\u0E10\u0E11\u0E12\u0E13\u0E14\u0E15\u0E16\u0E17\u0E18\u0E19\u0E1A\u0E1B\u0E1C\u0E1D\u0E1E\u0E1F\u0E20\u0E21\u0E22\u0E23\u0E24\u0E25\u0E26\u0E27\u0E28\u0E29\u0E2A\u0E2B\u0E2C\u0E2D\u0E2E\u0E2F\u0E30\u0E32\u0E33\u0E40\u0E41\u0E42\u0E43\u0E44\u0E45\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94\u0E95\u0E96\u0E97\u0E99\u0E9A\u0E9B\u0E9C\u0E9D\u0E9E\u0E9F\u0EA1\u0EA2\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD\u0EAE\u0EAF\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0\u0EC1\u0EC2\u0EC3\u0EC4\u0EDC\u0EDD\u0F00\u0F40\u0F41\u0F42\u0F42\u0FB7\u0F44\u0F45\u0F46\u0F47\u0F49\u0F4A\u0F4B\u0F4C\u0F4C\u0FB7\u0F4E\u0F4F\u0F50\u0F51\u0F51\u0FB7\u0F53\u0F54\u0F55\u0F56\u0F56\u0FB7\u0F58\u0F59\u0F5A\u0F5B\u0F5B\u0FB7\u0F5D\u0F5E\u0F5F\u0F60\u0F61\u0F62\u0F63\u0F64\u0F65\u0F66\u0F67\u0F68\u0F40\u0FB5\u0F6A\u0F6B\u0F6C\u0F88\u0F89\u0F8A\u0F8B\u0F8C\u1000\u1001\u1002\u1003\u1004\u1005\u1006\u1007\u1008\u1009\u100A\u100B\u100C\u100D\u100E\u100F\u1010\u1011\u1012\u1013\u1014\u1015\u1016\u1017\u1018\u1019\u101A\u101B\u101C\u101D\u101E\u101F\u1020\u1021\u1022\u1023\u1024\u1025\u1026\u1027\u1028\u1029\u102A\u103F\u1050\u1051\u1052\u1053\u1054\u1055\u105A\u105B\u105C\u105D\u1061\u1065\u1066\u106E\u106F\u1070\u1075\u1076\u1077\u1078\u1079\u107A\u107B\u107C\u107D\u107E\u107F\u1080\u1081\u108E\u10D0\u10D1\u10D2\u10D3\u10D4\u10D5\u10D6\u10D7\u10D8\u10D9\u10DA\u10DB\u10DC\u10DD\u10DE\u10DF\u10E0\u10E1\u10E2\u10E3\u10E4\u10E5\u10E6\u10E7\u10E8\u10E9\u10EA\u10EB\u10EC\u10ED\u10EE\u10EF\u10F0\u10F1\u10F2\u10F3\u10F4\u10F5\u10F6\u10F7\u10F8\u10F9\u10FA\u1100\u1101\u1102\u1103\u1104\u1105\u1106\u1107\u1108\u1109\u110A\u110B\u110C\u110D\u110E\u110F\u1110\u1111\u1112\u1113\u1114\u1115\u1116\u1117\u1118\u1119\u111A\u111B\u111C\u111D\u111E\u111F\u1120\u1121\u1122\u1123\u1124\u1125\u1126\u1127\u1128\u1129\u112A\u112B\u112C\u112D\u112E\u112F\u1130\u1131\u1132\u1133\u1134\u1135\u1136\u1137\u1138\u1139\u113A\u113B\u113C\u113D\u113E\u113F\u1140\u1141\u1142\u1143\u1144\u1145\u1146\u1147\u1148\u1149\u114A\u114B\u114C\u114D\u114E\u114F\u1150\u1151\u1152\u1153\u1154\u1155\u1156\u1157\u1158\u1159\u115A\u115B\u115C\u115D\u115E\u115F\u1160\u1161\u1162\u1163\u1164\u1165\u1166\u1167\u1168\u1169\u116A\u116B\u116C\u116D\u116E\u116F\u1170\u1171\u1172\u1173\u1174\u1175\u1176\u1177\u1178\u1179\u117A\u117B\u117C\u117D\u117E\u117F\u1180\u1181\u1182\u1183\u1184\u1185\u1186\u1187\u1188\u1189\u118A\u118B\u118C\u118D\u118E\u118F\u1190\u1191\u1192\u1193\u1194\u1195\u1196\u1197\u1198\u1199\u119A\u119B\u119C\u119D\u119E\u119F\u11A0\u11A1\u11A2\u11A3\u11A4\u11A5\u11A6\u11A7\u11A8\u11A9\u11AA\u11AB\u11AC\u11AD\u11AE\u11AF\u11B0\u11B1\u11B2\u11B3\u11B4\u11B5\u11B6\u11B7\u11B8\u11B9\u11BA\u11BB\u11BC\u11BD\u11BE\u11BF\u11C0\u11C1\u11C2\u11C3\u11C4\u11C5\u11C6\u11C7\u11C8\u11C9\u11CA\u11CB\u11CC\u11CD\u11CE\u11CF\u11D0\u11D1\u11D2\u11D3\u11D4\u11D5\u11D6\u11D7\u11D8\u11D9\u11DA\u11DB\u11DC\u11DD\u11DE\u11DF\u11E0\u11E1\u11E2\u11E3\u11E4\u11E5\u11E6\u11E7\u11E8\u11E9\u11EA\u11EB\u11EC\u11ED\u11EE\u11EF\u11F0\u11F1\u11F2\u11F3\u11F4\u11F5\u11F6\u11F7\u11F8\u11F9\u11FA\u11FB\u11FC\u11FD\u11FE\u11FF\u1200\u1201\u1202\u1203\u1204\u1205\u1206\u1207\u1208\u1209\u120A\u120B\u120C\u120D\u120E\u120F\u1210\u1211\u1212\u1213\u1214\u1215\u1216\u1217\u1218\u1219\u121A\u121B\u121C\u121D\u121E\u121F\u1220\u1221\u1222\u1223\u1224\u1225\u1226\u1227\u1228\u1229\u122A\u122B\u122C\u122D\u122E\u122F\u1230\u1231\u1232\u1233\u1234\u1235\u1236\u1237\u1238\u1239\u123A\u123B\u123C\u123D\u123E\u123F\u1240\u1241\u1242\u1243\u1244\u1245\u1246\u1247\u1248\u124A\u124B\u124C\u124D\u1250\u1251\u1252\u1253\u1254\u1255\u1256\u1258\u125A\u125B\u125C\u125D\u1260\u1261\u1262\u1263\u1264\u1265\u1266\u1267\u1268\u1269\u126A\u126B\u126C\u126D\u126E\u126F\u1270\u1271\u1272\u1273\u1274\u1275\u1276\u1277\u1278\u1279\u127A\u127B\u127C\u127D\u127E\u127F\u1280\u1281\u1282\u1283\u1284\u1285\u1286\u1287\u1288\u128A\u128B\u128C\u128D\u1290\u1291\u1292\u1293\u1294\u1295\u1296\u1297\u1298\u1299\u129A\u129B\u129C\u129D\u129E\u129F\u12A0\u12A1\u12A2\u12A3\u12A4\u12A5\u12A6\u12A7\u12A8\u12A9\u12AA\u12AB\u12AC\u12AD\u12AE\u12AF\u12B0\u12B2\u12B3\u12B4\u12B5\u12B8\u12B9\u12BA\u12BB\u12BC\u12BD\u12BE\u12C0\u12C2\u12C3\u12C4\u12C5\u12C8\u12C9\u12CA\u12CB\u12CC\u12CD\u12CE\u12CF\u12D0\u12D1\u12D2\u12D3\u12D4\u12D5\u12D6\u12D8\u12D9\u12DA\u12DB\u12DC\u12DD\u12DE\u12DF\u12E0\u12E1\u12E2\u12E3\u12E4\u12E5\u12E6\u12E7\u12E8\u12E9\u12EA\u12EB\u12EC\u12ED\u12EE\u12EF\u12F0\u12F1\u12F2\u12F3\u12F4\u12F5\u12F6\u12F7\u12F8\u12F9\u12FA\u12FB\u12FC\u12FD\u12FE\u12FF\u1300\u1301\u1302\u1303\u1304\u1305\u1306\u1307\u1308\u1309\u130A\u130B\u130C\u130D\u130E\u130F\u1310\u1312\u1313\u1314\u1315\u1318\u1319\u131A\u131B\u131C\u131D\u131E\u131F\u1320\u1321\u1322\u1323\u1324\u1325\u1326\u1327\u1328\u1329\u132A\u132B\u132C\u132D\u132E\u132F\u1330\u1331\u1332\u1333\u1334\u1335\u1336\u1337\u1338\u1339\u133A\u133B\u133C\u133D\u133E\u133F\u1340\u1341\u1342\u1343\u1344\u1345\u1346\u1347\u1348\u1349\u134A\u134B\u134C\u134D\u134E\u134F\u1350\u1351\u1352\u1353\u1354\u1355\u1356\u1357\u1358\u1359\u135A\u1380\u1381\u1382\u1383\u1384\u1385\u1386\u1387\u1388\u1389\u138A\u138B\u138C\u138D\u138E\u138F\u13A0\u13A1\u13A2\u13A3\u13A4\u13A5\u13A6\u13A7\u13A8\u13A9\u13AA\u13AB\u13AC\u13AD\u13AE\u13AF\u13B0\u13B1\u13B2\u13B3\u13B4\u13B5\u13B6\u13B7\u13B8\u13B9\u13BA\u13BB\u13BC\u13BD\u13BE\u13BF\u13C0\u13C1\u13C2\u13C3\u13C4\u13C5\u13C6\u13C7\u13C8\u13C9\u13CA\u13CB\u13CC\u13CD\u13CE\u13CF\u13D0\u13D1\u13D2\u13D3\u13D4\u13D5\u13D6\u13D7\u13D8\u13D9\u13DA\u13DB\u13DC\u13DD\u13DE\u13DF\u13E0\u13E1\u13E2\u13E3\u13E4\u13E5\u13E6\u13E7\u13E8\u13E9\u13EA\u13EB\u13EC\u13ED\u13EE\u13EF\u13F0\u13F1\u13F2\u13F3\u13F4\u1401\u1402\u1403\u1404\u1405\u1406\u1407\u1408\u1409\u140A\u140B\u140C\u140D\u140E\u140F\u1410\u1411\u1412\u1413\u1414\u1415\u1416\u1417\u1418\u1419\u141A\u141B\u141C\u141D\u141E\u141F\u1420\u1421\u1422\u1423\u1424\u1425\u1426\u1427\u1428\u1429\u142A\u142B\u142C\u142D\u142E\u142F\u1430\u1431\u1432\u1433\u1434\u1435\u1436\u1437\u1438\u1439\u143A\u143B\u143C\u143D\u143E\u143F\u1440\u1441\u1442\u1443\u1444\u1445\u1446\u1447\u1448\u1449\u144A\u144B\u144C\u144D\u144E\u144F\u1450\u1451\u1452\u1453\u1454\u1455\u1456\u1457\u1458\u1459\u145A\u145B\u145C\u145D\u145E\u145F\u1460\u1461\u1462\u1463\u1464\u1465\u1466\u1467\u1468\u1469\u146A\u146B\u146C\u146D\u146E\u146F\u1470\u1471\u1472\u1473\u1474\u1475\u1476\u1477\u1478\u1479\u147A\u147B\u147C\u147D\u147E\u147F\u1480\u1481\u1482\u1483\u1484\u1485\u1486\u1487\u1488\u1489\u148A\u148B\u148C\u148D\u148E\u148F\u1490\u1491\u1492\u1493\u1494\u1495\u1496\u1497\u1498\u1499\u149A\u149B\u149C\u149D\u149E\u149F\u14A0\u14A1\u14A2\u14A3\u14A4\u14A5\u14A6\u14A7\u14A8\u14A9\u14AA\u14AB\u14AC\u14AD\u14AE\u14AF\u14B0\u14B1\u14B2\u14B3\u14B4\u14B5\u14B6\u14B7\u14B8\u14B9\u14BA\u14BB\u14BC\u14BD\u14BE\u14BF\u14C0\u14C1\u14C2\u14C3\u14C4\u14C5\u14C6\u14C7\u14C8\u14C9\u14CA\u14CB\u14CC\u14CD\u14CE\u14CF\u14D0\u14D1\u14D2\u14D3\u14D4\u14D5\u14D6\u14D7\u14D8\u14D9\u14DA\u14DB\u14DC\u14DD\u14DE\u14DF\u14E0\u14E1\u14E2\u14E3\u14E4\u14E5\u14E6\u14E7\u14E8\u14E9\u14EA\u14EB\u14EC\u14ED\u14EE\u14EF\u14F0\u14F1\u14F2\u14F3\u14F4\u14F5\u14F6\u14F7\u14F8\u14F9\u14FA\u14FB\u14FC\u14FD\u14FE\u14FF\u1500\u1501\u1502\u1503\u1504\u1505\u1506\u1507\u1508\u1509\u150A\u150B\u150C\u150D\u150E\u150F\u1510\u1511\u1512\u1513\u1514\u1515\u1516\u1517\u1518\u1519\u151A\u151B\u151C\u151D\u151E\u151F\u1520\u1521\u1522\u1523\u1524\u1525\u1526\u1527\u1528\u1529\u152A\u152B\u152C\u152D\u152E\u152F\u1530\u1531\u1532\u1533\u1534\u1535\u1536\u1537\u1538\u1539\u153A\u153B\u153C\u153D\u153E\u153F\u1540\u1541\u1542\u1543\u1544\u1545\u1546\u1547\u1548\u1549\u154A\u154B\u154C\u154D\u154E\u154F\u1550\u1551\u1552\u1553\u1554\u1555\u1556\u1557\u1558\u1559\u155A\u155B\u155C\u155D\u155E\u155F\u1560\u1561\u1562\u1563\u1564\u1565\u1566\u1567\u1568\u1569\u156A\u156B\u156C\u156D\u156E\u156F\u1570\u1571\u1572\u1573\u1574\u1575\u1576\u1577\u1578\u1579\u157A\u157B\u157C\u157D\u157E\u157F\u1580\u1581\u1582\u1583\u1584\u1585\u1586\u1587\u1588\u1589\u158A\u158B\u158C\u158D\u158E\u158F\u1590\u1591\u1592\u1593\u1594\u1595\u1596\u1597\u1598\u1599\u159A\u159B\u159C\u159D\u159E\u159F\u15A0\u15A1\u15A2\u15A3\u15A4\u15A5\u15A6\u15A7\u15A8\u15A9\u15AA\u15AB\u15AC\u15AD\u15AE\u15AF\u15B0\u15B1\u15B2\u15B3\u15B4\u15B5\u15B6\u15B7\u15B8\u15B9\u15BA\u15BB\u15BC\u15BD\u15BE\u15BF\u15C0\u15C1\u15C2\u15C3\u15C4\u15C5\u15C6\u15C7\u15C8\u15C9\u15CA\u15CB\u15CC\u15CD\u15CE\u15CF\u15D0\u15D1\u15D2\u15D3\u15D4\u15D5\u15D6\u15D7\u15D8\u15D9\u15DA\u15DB\u15DC\u15DD\u15DE\u15DF\u15E0\u15E1\u15E2\u15E3\u15E4\u15E5\u15E6\u15E7\u15E8\u15E9\u15EA\u15EB\u15EC\u15ED\u15EE\u15EF\u15F0\u15F1\u15F2\u15F3\u15F4\u15F5\u15F6\u15F7\u15F8\u15F9\u15FA\u15FB\u15FC\u15FD\u15FE\u15FF\u1600\u1601\u1602\u1603\u1604\u1605\u1606\u1607\u1608\u1609\u160A\u160B\u160C\u160D\u160E\u160F\u1610\u1611\u1612\u1613\u1614\u1615\u1616\u1617\u1618\u1619\u161A\u161B\u161C\u161D\u161E\u161F\u1620\u1621\u1622\u1623\u1624\u1625\u1626\u1627\u1628\u1629\u162A\u162B\u162C\u162D\u162E\u162F\u1630\u1631\u1632\u1633\u1634\u1635\u1636\u1637\u1638\u1639\u163A\u163B\u163C\u163D\u163E\u163F\u1640\u1641\u1642\u1643\u1644\u1645\u1646\u1647\u1648\u1649\u164A\u164B\u164C\u164D\u164E\u164F\u1650\u1651\u1652\u1653\u1654\u1655\u1656\u1657\u1658\u1659\u165A\u165B\u165C\u165D\u165E\u165F\u1660\u1661\u1662\u1663\u1664\u1665\u1666\u1667\u1668\u1669\u166A\u166B\u166C\u166F\u1670\u1671\u1672\u1673\u1674\u1675\u1676\u1677\u1678\u1679\u167A\u167B\u167C\u167D\u167E\u167F\u1681\u1682\u1683\u1684\u1685\u1686\u1687\u1688\u1689\u168A\u168B\u168C\u168D\u168E\u168F\u1690\u1691\u1692\u1693\u1694\u1695\u1696\u1697\u1698\u1699\u169A\u16A0\u16A1\u16A2\u16A3\u16A4\u16A5\u16A6\u16A7\u16A8\u16A9\u16AA\u16AB\u16AC\u16AD\u16AE\u16AF\u16B0\u16B1\u16B2\u16B3\u16B4\u16B5\u16B6\u16B7\u16B8\u16B9\u16BA\u16BB\u16BC\u16BD\u16BE\u16BF\u16C0\u16C1\u16C2\u16C3\u16C4\u16C5\u16C6\u16C7\u16C8\u16C9\u16CA\u16CB\u16CC\u16CD\u16CE\u16CF\u16D0\u16D1\u16D2\u16D3\u16D4\u16D5\u16D6\u16D7\u16D8\u16D9\u16DA\u16DB\u16DC\u16DD\u16DE\u16DF\u16E0\u16E1\u16E2\u16E3\u16E4\u16E5\u16E6\u16E7\u16E8\u16E9\u16EA\u1700\u1701\u1702\u1703\u1704\u1705\u1706\u1707\u1708\u1709\u170A\u170B\u170C\u170E\u170F\u1710\u1711\u1720\u1721\u1722\u1723\u1724\u1725\u1726\u1727\u1728\u1729\u172A\u172B\u172C\u172D\u172E\u172F\u1730\u1731\u1740\u1741\u1742\u1743\u1744\u1745\u1746\u1747\u1748\u1749\u174A\u174B\u174C\u174D\u174E\u174F\u1750\u1751\u1760\u1761\u1762\u1763\u1764\u1765\u1766\u1767\u1768\u1769\u176A\u176B\u176C\u176E\u176F\u1770\u1780\u1781\u1782\u1783\u1784\u1785\u1786\u1787\u1788\u1789\u178A\u178B\u178C\u178D\u178E\u178F\u1790\u1791\u1792\u1793\u1794\u1795\u1796\u1797\u1798\u1799\u179A\u179B\u179C\u179D\u179E\u179F\u17A0\u17A1\u17A2\u17A3\u17A4\u17A5\u17A6\u17A7\u17A8\u17A9\u17AA\u17AB\u17AC\u17AD\u17AE\u17AF\u17B0\u17B1\u17B2\u17B3\u17DC\u1820\u1821\u1822\u1823\u1824\u1825\u1826\u1827\u1828\u1829\u182A\u182B\u182C\u182D\u182E\u182F\u1830\u1831\u1832\u1833\u1834\u1835\u1836\u1837\u1838\u1839\u183A\u183B\u183C\u183D\u183E\u183F\u1840\u1841\u1842\u1844\u1845\u1846\u1847\u1848\u1849\u184A\u184B\u184C\u184D\u184E\u184F\u1850\u1851\u1852\u1853\u1854\u1855\u1856\u1857\u1858\u1859\u185A\u185B\u185C\u185D\u185E\u185F\u1860\u1861\u1862\u1863\u1864\u1865\u1866\u1867\u1868\u1869\u186A\u186B\u186C\u186D\u186E\u186F\u1870\u1871\u1872\u1873\u1874\u1875\u1876\u1877\u1880\u1881\u1882\u1883\u1884\u1885\u1886\u1887\u1888\u1889\u188A\u188B\u188C\u188D\u188E\u188F\u1890\u1891\u1892\u1893\u1894\u1895\u1896\u1897\u1898\u1899\u189A\u189B\u189C\u189D\u189E\u189F\u18A0\u18A1\u18A2\u18A3\u18A4\u18A5\u18A6\u18A7\u18A8\u18AA\u18B0\u18B1\u18B2\u18B3\u18B4\u18B5\u18B6\u18B7\u18B8\u18B9\u18BA\u18BB\u18BC\u18BD\u18BE\u18BF\u18C0\u18C1\u18C2\u18C3\u18C4\u18C5\u18C6\u18C7\u18C8\u18C9\u18CA\u18CB\u18CC\u18CD\u18CE\u18CF\u18D0\u18D1\u18D2\u18D3\u18D4\u18D5\u18D6\u18D7\u18D8\u18D9\u18DA\u18DB\u18DC\u18DD\u18DE\u18DF\u18E0\u18E1\u18E2\u18E3\u18E4\u18E5\u18E6\u18E7\u18E8\u18E9\u18EA\u18EB\u18EC\u18ED\u18EE\u18EF\u18F0\u18F1\u18F2\u18F3\u18F4\u18F5\u1900\u1901\u1902\u1903\u1904\u1905\u1906\u1907\u1908\u1909\u190A\u190B\u190C\u190D\u190E\u190F\u1910\u1911\u1912\u1913\u1914\u1915\u1916\u1917\u1918\u1919\u191A\u191B\u191C\u1950\u1951\u1952\u1953\u1954\u1955\u1956\u1957\u1958\u1959\u195A\u195B\u195C\u195D\u195E\u195F\u1960\u1961\u1962\u1963\u1964\u1965\u1966\u1967\u1968\u1969\u196A\u196B\u196C\u196D\u1970\u1971\u1972\u1973\u1974\u1980\u1981\u1982\u1983\u1984\u1985\u1986\u1987\u1988\u1989\u198A\u198B\u198C\u198D\u198E\u198F\u1990\u1991\u1992\u1993\u1994\u1995\u1996\u1997\u1998\u1999\u199A\u199B\u199C\u199D\u199E\u199F\u19A0\u19A1\u19A2\u19A3\u19A4\u19A5\u19A6\u19A7\u19A8\u19A9\u19AA\u19AB\u19C1\u19C2\u19C3\u19C4\u19C5\u19C6\u19C7\u1A00\u1A01\u1A02\u1A03\u1A04\u1A05\u1A06\u1A07\u1A08\u1A09\u1A0A\u1A0B\u1A0C\u1A0D\u1A0E\u1A0F\u1A10\u1A11\u1A12\u1A13\u1A14\u1A15\u1A16\u1A20\u1A21\u1A22\u1A23\u1A24\u1A25\u1A26\u1A27\u1A28\u1A29\u1A2A\u1A2B\u1A2C\u1A2D\u1A2E\u1A2F\u1A30\u1A31\u1A32\u1A33\u1A34\u1A35\u1A36\u1A37\u1A38\u1A39\u1A3A\u1A3B\u1A3C\u1A3D\u1A3E\u1A3F\u1A40\u1A41\u1A42\u1A43\u1A44\u1A45\u1A46\u1A47\u1A48\u1A49\u1A4A\u1A4B\u1A4C\u1A4D\u1A4E\u1A4F\u1A50\u1A51\u1A52\u1A53\u1A54\u1B05\u1B06\u1B07\u1B08\u1B09\u1B0A\u1B0B\u1B0C\u1B0D\u1B0E\u1B0F\u1B10\u1B11\u1B12\u1B13\u1B14\u1B15\u1B16\u1B17\u1B18\u1B19\u1B1A\u1B1B\u1B1C\u1B1D\u1B1E\u1B1F\u1B20\u1B21\u1B22\u1B23\u1B24\u1B25\u1B26\u1B27\u1B28\u1B29\u1B2A\u1B2B\u1B2C\u1B2D\u1B2E\u1B2F\u1B30\u1B31\u1B32\u1B33\u1B45\u1B46\u1B47\u1B48\u1B49\u1B4A\u1B4B\u1B83\u1B84\u1B85\u1B86\u1B87\u1B88\u1B89\u1B8A\u1B8B\u1B8C\u1B8D\u1B8E\u1B8F\u1B90\u1B91\u1B92\u1B93\u1B94\u1B95\u1B96\u1B97\u1B98\u1B99\u1B9A\u1B9B\u1B9C\u1B9D\u1B9E\u1B9F\u1BA0\u1BAE\u1BAF\u1BC0\u1BC1\u1BC2\u1BC3\u1BC4\u1BC5\u1BC6\u1BC7\u1BC8\u1BC9\u1BCA\u1BCB\u1BCC\u1BCD\u1BCE\u1BCF\u1BD0\u1BD1\u1BD2\u1BD3\u1BD4\u1BD5\u1BD6\u1BD7\u1BD8\u1BD9\u1BDA\u1BDB\u1BDC\u1BDD\u1BDE\u1BDF\u1BE0\u1BE1\u1BE2\u1BE3\u1BE4\u1BE5\u1C00\u1C01\u1C02\u1C03\u1C04\u1C05\u1C06\u1C07\u1C08\u1C09\u1C0A\u1C0B\u1C0C\u1C0D\u1C0E\u1C0F\u1C10\u1C11\u1C12\u1C13\u1C14\u1C15\u1C16\u1C17\u1C18\u1C19\u1C1A\u1C1B\u1C1C\u1C1D\u1C1E\u1C1F\u1C20\u1C21\u1C22\u1C23\u1C4D\u1C4E\u1C4F\u1C5A\u1C5B\u1C5C\u1C5D\u1C5E\u1C5F\u1C60\u1C61\u1C62\u1C63\u1C64\u1C65\u1C66\u1C67\u1C68\u1C69\u1C6A\u1C6B\u1C6C\u1C6D\u1C6E\u1C6F\u1C70\u1C71\u1C72\u1C73\u1C74\u1C75\u1C76\u1C77\u1CE9\u1CEA\u1CEB\u1CEC\u1CEE\u1CEF\u1CF0\u1CF1\u2135\u2136\u2137\u2138\u2D30\u2D31\u2D32\u2D33\u2D34\u2D35\u2D36\u2D37\u2D38\u2D39\u2D3A\u2D3B\u2D3C\u2D3D\u2D3E\u2D3F\u2D40\u2D41\u2D42\u2D43\u2D44\u2D45\u2D46\u2D47\u2D48\u2D49\u2D4A\u2D4B\u2D4C\u2D4D\u2D4E\u2D4F\u2D50\u2D51\u2D52\u2D53\u2D54\u2D55\u2D56\u2D57\u2D58\u2D59\u2D5A\u2D5B\u2D5C\u2D5D\u2D5E\u2D5F\u2D60\u2D61\u2D62\u2D63\u2D64\u2D65\u2D80\u2D81\u2D82\u2D83\u2D84\u2D85\u2D86\u2D87\u2D88\u2D89\u2D8A\u2D8B\u2D8C\u2D8D\u2D8E\u2D8F\u2D90\u2D91\u2D92\u2D93\u2D94\u2D95\u2D96\u2DA0\u2DA1\u2DA2\u2DA3\u2DA4\u2DA5\u2DA6\u2DA8\u2DA9\u2DAA\u2DAB\u2DAC\u2DAD\u2DAE\u2DB0\u2DB1\u2DB2\u2DB3\u2DB4\u2DB5\u2DB6\u2DB8\u2DB9\u2DBA\u2DBB\u2DBC\u2DBD\u2DBE\u2DC0\u2DC1\u2DC2\u2DC3\u2DC4\u2DC5\u2DC6\u2DC8\u2DC9\u2DCA\u2DCB\u2DCC\u2DCD\u2DCE\u2DD0\u2DD1\u2DD2\u2DD3\u2DD4\u2DD5\u2DD6\u2DD8\u2DD9\u2DDA\u2DDB\u2DDC\u2DDD\u2DDE\u3006\u303C\u3041\u3042\u3043\u3044\u3045\u3046\u3047\u3048\u3049\u304A\u304B\u304C\u304D\u304E\u304F\u3050\u3051\u3052\u3053\u3054\u3055\u3056\u3057\u3058\u3059\u305A\u305B\u305C\u305D\u305E\u305F\u3060\u3061\u3062\u3063\u3064\u3065\u3066\u3067\u3068\u3069\u306A\u306B\u306C\u306D\u306E\u306F\u3070\u3071\u3072\u3073\u3074\u3075\u3076\u3077\u3078\u3079\u307A\u307B\u307C\u307D\u307E\u307F\u3080\u3081\u3082\u3083\u3084\u3085\u3086\u3087\u3088\u3089\u308A\u308B\u308C\u308D\u308E\u308F\u3090\u3091\u3092\u3093\u3094\u3095\u3096\u309F\u30A1\u30A2\u30A3\u30A4\u30A5\u30A6\u30A7\u30A8\u30A9\u30AA\u30AB\u30AC\u30AD\u30AE\u30AF\u30B0\u30B1\u30B2\u30B3\u30B4\u30B5\u30B6\u30B7\u30B8\u30B9\u30BA\u30BB\u30BC\u30BD\u30BE\u30BF\u30C0\u30C1\u30C2\u30C3\u30C4\u30C5\u30C6\u30C7\u30C8\u30C9\u30CA\u30CB\u30CC\u30CD\u30CE\u30CF\u30D0\u30D1\u30D2\u30D3\u30D4\u30D5\u30D6\u30D7\u30D8\u30D9\u30DA\u30DB\u30DC\u30DD\u30DE\u30DF\u30E0\u30E1\u30E2\u30E3\u30E4\u30E5\u30E6\u30E7\u30E8\u30E9\u30EA\u30EB\u30EC\u30ED\u30EE\u30EF\u30F0\u30F1\u30F2\u30F3\u30F4\u30F5\u30F6\u30F7\u30F8\u30F9\u30FA\u30FF\u3105\u3106\u3107\u3108\u3109\u310A\u310B\u310C\u310D\u310E\u310F\u3110\u3111\u3112\u3113\u3114\u3115\u3116\u3117\u3118\u3119\u311A\u311B\u311C\u311D\u311E\u311F\u3120\u3121\u3122\u3123\u3124\u3125\u3126\u3127\u3128\u3129\u312A\u312B\u312C\u312D\u3131\u3132\u3133\u3134\u3135\u3136\u3137\u3138\u3139\u313A\u313B\u313C\u313D\u313E\u313F\u3140\u3141\u3142\u3143\u3144\u3145\u3146\u3147\u3148\u3149\u314A\u314B\u314C\u314D\u314E\u314F\u3150\u3151\u3152\u3153\u3154\u3155\u3156\u3157\u3158\u3159\u315A\u315B\u315C\u315D\u315E\u315F\u3160\u3161\u3162\u3163\u3164\u3165\u3166\u3167\u3168\u3169\u316A\u316B\u316C\u316D\u316E\u316F\u3170\u3171\u3172\u3173\u3174\u3175\u3176\u3177\u3178\u3179\u317A\u317B\u317C\u317D\u317E\u317F\u3180\u3181\u3182\u3183\u3184\u3185\u3186\u3187\u3188\u3189\u318A\u318B\u318C\u318D\u318E\u31A0\u31A1\u31A2\u31A3\u31A4\u31A5\u31A6\u31A7\u31A8\u31A9\u31AA\u31AB\u31AC\u31AD\u31AE\u31AF\u31B0\u31B1\u31B2\u31B3\u31B4\u31B5\u31B6\u31B7\u31B8\u31B9\u31BA\u31F0\u31F1\u31F2\u31F3\u31F4\u31F5\u31F6\u31F7\u31F8\u31F9\u31FA\u31FB\u31FC\u31FD\u31FE\u31FF\u3400\u4DB5\u4E00\u9FCB\uA000\uA001\uA002\uA003\uA004\uA005\uA006\uA007\uA008\uA009\uA00A\uA00B\uA00C\uA00D\uA00E\uA00F\uA010\uA011\uA012\uA013\uA014\uA016\uA017\uA018\uA019\uA01A\uA01B\uA01C\uA01D\uA01E\uA01F\uA020\uA021\uA022\uA023\uA024\uA025\uA026\uA027\uA028\uA029\uA02A\uA02B\uA02C\uA02D\uA02E\uA02F\uA030\uA031\uA032\uA033\uA034\uA035\uA036\uA037\uA038\uA039\uA03A\uA03B\uA03C\uA03D\uA03E\uA03F\uA040\uA041\uA042\uA043\uA044\uA045\uA046\uA047\uA048\uA049\uA04A\uA04B\uA04C\uA04D\uA04E\uA04F\uA050\uA051\uA052\uA053\uA054\uA055\uA056\uA057\uA058\uA059\uA05A\uA05B\uA05C\uA05D\uA05E\uA05F\uA060\uA061\uA062\uA063\uA064\uA065\uA066\uA067\uA068\uA069\uA06A\uA06B\uA06C\uA06D\uA06E\uA06F\uA070\uA071\uA072\uA073\uA074\uA075\uA076\uA077\uA078\uA079\uA07A\uA07B\uA07C\uA07D\uA07E\uA07F\uA080\uA081\uA082\uA083\uA084\uA085\uA086\uA087\uA088\uA089\uA08A\uA08B\uA08C\uA08D\uA08E\uA08F\uA090\uA091\uA092\uA093\uA094\uA095\uA096\uA097\uA098\uA099\uA09A\uA09B\uA09C\uA09D\uA09E\uA09F\uA0A0\uA0A1\uA0A2\uA0A3\uA0A4\uA0A5\uA0A6\uA0A7\uA0A8\uA0A9\uA0AA\uA0AB\uA0AC\uA0AD\uA0AE\uA0AF\uA0B0\uA0B1\uA0B2\uA0B3\uA0B4\uA0B5\uA0B6\uA0B7\uA0B8\uA0B9\uA0BA\uA0BB\uA0BC\uA0BD\uA0BE\uA0BF\uA0C0\uA0C1\uA0C2\uA0C3\uA0C4\uA0C5\uA0C6\uA0C7\uA0C8\uA0C9\uA0CA\uA0CB\uA0CC\uA0CD\uA0CE\uA0CF\uA0D0\uA0D1\uA0D2\uA0D3\uA0D4\uA0D5\uA0D6\uA0D7\uA0D8\uA0D9\uA0DA\uA0DB\uA0DC\uA0DD\uA0DE\uA0DF\uA0E0\uA0E1\uA0E2\uA0E3\uA0E4\uA0E5\uA0E6\uA0E7\uA0E8\uA0E9\uA0EA\uA0EB\uA0EC\uA0ED\uA0EE\uA0EF\uA0F0\uA0F1\uA0F2\uA0F3\uA0F4\uA0F5\uA0F6\uA0F7\uA0F8\uA0F9\uA0FA\uA0FB\uA0FC\uA0FD\uA0FE\uA0FF\uA100\uA101\uA102\uA103\uA104\uA105\uA106\uA107\uA108\uA109\uA10A\uA10B\uA10C\uA10D\uA10E\uA10F\uA110\uA111\uA112\uA113\uA114\uA115\uA116\uA117\uA118\uA119\uA11A\uA11B\uA11C\uA11D\uA11E\uA11F\uA120\uA121\uA122\uA123\uA124\uA125\uA126\uA127\uA128\uA129\uA12A\uA12B\uA12C\uA12D\uA12E\uA12F\uA130\uA131\uA132\uA133\uA134\uA135\uA136\uA137\uA138\uA139\uA13A\uA13B\uA13C\uA13D\uA13E\uA13F\uA140\uA141\uA142\uA143\uA144\uA145\uA146\uA147\uA148\uA149\uA14A\uA14B\uA14C\uA14D\uA14E\uA14F\uA150\uA151\uA152\uA153\uA154\uA155\uA156\uA157\uA158\uA159\uA15A\uA15B\uA15C\uA15D\uA15E\uA15F\uA160\uA161\uA162\uA163\uA164\uA165\uA166\uA167\uA168\uA169\uA16A\uA16B\uA16C\uA16D\uA16E\uA16F\uA170\uA171\uA172\uA173\uA174\uA175\uA176\uA177\uA178\uA179\uA17A\uA17B\uA17C\uA17D\uA17E\uA17F\uA180\uA181\uA182\uA183\uA184\uA185\uA186\uA187\uA188\uA189\uA18A\uA18B\uA18C\uA18D\uA18E\uA18F\uA190\uA191\uA192\uA193\uA194\uA195\uA196\uA197\uA198\uA199\uA19A\uA19B\uA19C\uA19D\uA19E\uA19F\uA1A0\uA1A1\uA1A2\uA1A3\uA1A4\uA1A5\uA1A6\uA1A7\uA1A8\uA1A9\uA1AA\uA1AB\uA1AC\uA1AD\uA1AE\uA1AF\uA1B0\uA1B1\uA1B2\uA1B3\uA1B4\uA1B5\uA1B6\uA1B7\uA1B8\uA1B9\uA1BA\uA1BB\uA1BC\uA1BD\uA1BE\uA1BF\uA1C0\uA1C1\uA1C2\uA1C3\uA1C4\uA1C5\uA1C6\uA1C7\uA1C8\uA1C9\uA1CA\uA1CB\uA1CC\uA1CD\uA1CE\uA1CF\uA1D0\uA1D1\uA1D2\uA1D3\uA1D4\uA1D5\uA1D6\uA1D7\uA1D8\uA1D9\uA1DA\uA1DB\uA1DC\uA1DD\uA1DE\uA1DF\uA1E0\uA1E1\uA1E2\uA1E3\uA1E4\uA1E5\uA1E6\uA1E7\uA1E8\uA1E9\uA1EA\uA1EB\uA1EC\uA1ED\uA1EE\uA1EF\uA1F0\uA1F1\uA1F2\uA1F3\uA1F4\uA1F5\uA1F6\uA1F7\uA1F8\uA1F9\uA1FA\uA1FB\uA1FC\uA1FD\uA1FE\uA1FF\uA200\uA201\uA202\uA203\uA204\uA205\uA206\uA207\uA208\uA209\uA20A\uA20B\uA20C\uA20D\uA20E\uA20F\uA210\uA211\uA212\uA213\uA214\uA215\uA216\uA217\uA218\uA219\uA21A\uA21B\uA21C\uA21D\uA21E\uA21F\uA220\uA221\uA222\uA223\uA224\uA225\uA226\uA227\uA228\uA229\uA22A\uA22B\uA22C\uA22D\uA22E\uA22F\uA230\uA231\uA232\uA233\uA234\uA235\uA236\uA237\uA238\uA239\uA23A\uA23B\uA23C\uA23D\uA23E\uA23F\uA240\uA241\uA242\uA243\uA244\uA245\uA246\uA247\uA248\uA249\uA24A\uA24B\uA24C\uA24D\uA24E\uA24F\uA250\uA251\uA252\uA253\uA254\uA255\uA256\uA257\uA258\uA259\uA25A\uA25B\uA25C\uA25D\uA25E\uA25F\uA260\uA261\uA262\uA263\uA264\uA265\uA266\uA267\uA268\uA269\uA26A\uA26B\uA26C\uA26D\uA26E\uA26F\uA270\uA271\uA272\uA273\uA274\uA275\uA276\uA277\uA278\uA279\uA27A\uA27B\uA27C\uA27D\uA27E\uA27F\uA280\uA281\uA282\uA283\uA284\uA285\uA286\uA287\uA288\uA289\uA28A\uA28B\uA28C\uA28D\uA28E\uA28F\uA290\uA291\uA292\uA293\uA294\uA295\uA296\uA297\uA298\uA299\uA29A\uA29B\uA29C\uA29D\uA29E\uA29F\uA2A0\uA2A1\uA2A2\uA2A3\uA2A4\uA2A5\uA2A6\uA2A7\uA2A8\uA2A9\uA2AA\uA2AB\uA2AC\uA2AD\uA2AE\uA2AF\uA2B0\uA2B1\uA2B2\uA2B3\uA2B4\uA2B5\uA2B6\uA2B7\uA2B8\uA2B9\uA2BA\uA2BB\uA2BC\uA2BD\uA2BE\uA2BF\uA2C0\uA2C1\uA2C2\uA2C3\uA2C4\uA2C5\uA2C6\uA2C7\uA2C8\uA2C9\uA2CA\uA2CB\uA2CC\uA2CD\uA2CE\uA2CF\uA2D0\uA2D1\uA2D2\uA2D3\uA2D4\uA2D5\uA2D6\uA2D7\uA2D8\uA2D9\uA2DA\uA2DB\uA2DC\uA2DD\uA2DE\uA2DF\uA2E0\uA2E1\uA2E2\uA2E3\uA2E4\uA2E5\uA2E6\uA2E7\uA2E8\uA2E9\uA2EA\uA2EB\uA2EC\uA2ED\uA2EE\uA2EF\uA2F0\uA2F1\uA2F2\uA2F3\uA2F4\uA2F5\uA2F6\uA2F7\uA2F8\uA2F9\uA2FA\uA2FB\uA2FC\uA2FD\uA2FE\uA2FF\uA300\uA301\uA302\uA303\uA304\uA305\uA306\uA307\uA308\uA309\uA30A\uA30B\uA30C\uA30D\uA30E\uA30F\uA310\uA311\uA312\uA313\uA314\uA315\uA316\uA317\uA318\uA319\uA31A\uA31B\uA31C\uA31D\uA31E\uA31F\uA320\uA321\uA322\uA323\uA324\uA325\uA326\uA327\uA328\uA329\uA32A\uA32B\uA32C\uA32D\uA32E\uA32F\uA330\uA331\uA332\uA333\uA334\uA335\uA336\uA337\uA338\uA339\uA33A\uA33B\uA33C\uA33D\uA33E\uA33F\uA340\uA341\uA342\uA343\uA344\uA345\uA346\uA347\uA348\uA349\uA34A\uA34B\uA34C\uA34D\uA34E\uA34F\uA350\uA351\uA352\uA353\uA354\uA355\uA356\uA357\uA358\uA359\uA35A\uA35B\uA35C\uA35D\uA35E\uA35F\uA360\uA361\uA362\uA363\uA364\uA365\uA366\uA367\uA368\uA369\uA36A\uA36B\uA36C\uA36D\uA36E\uA36F\uA370\uA371\uA372\uA373\uA374\uA375\uA376\uA377\uA378\uA379\uA37A\uA37B\uA37C\uA37D\uA37E\uA37F\uA380\uA381\uA382\uA383\uA384\uA385\uA386\uA387\uA388\uA389\uA38A\uA38B\uA38C\uA38D\uA38E\uA38F\uA390\uA391\uA392\uA393\uA394\uA395\uA396\uA397\uA398\uA399\uA39A\uA39B\uA39C\uA39D\uA39E\uA39F\uA3A0\uA3A1\uA3A2\uA3A3\uA3A4\uA3A5\uA3A6\uA3A7\uA3A8\uA3A9\uA3AA\uA3AB\uA3AC\uA3AD\uA3AE\uA3AF\uA3B0\uA3B1\uA3B2\uA3B3\uA3B4\uA3B5\uA3B6\uA3B7\uA3B8\uA3B9\uA3BA\uA3BB\uA3BC\uA3BD\uA3BE\uA3BF\uA3C0\uA3C1\uA3C2\uA3C3\uA3C4\uA3C5\uA3C6\uA3C7\uA3C8\uA3C9\uA3CA\uA3CB\uA3CC\uA3CD\uA3CE\uA3CF\uA3D0\uA3D1\uA3D2\uA3D3\uA3D4\uA3D5\uA3D6\uA3D7\uA3D8\uA3D9\uA3DA\uA3DB\uA3DC\uA3DD\uA3DE\uA3DF\uA3E0\uA3E1\uA3E2\uA3E3\uA3E4\uA3E5\uA3E6\uA3E7\uA3E8\uA3E9\uA3EA\uA3EB\uA3EC\uA3ED\uA3EE\uA3EF\uA3F0\uA3F1\uA3F2\uA3F3\uA3F4\uA3F5\uA3F6\uA3F7\uA3F8\uA3F9\uA3FA\uA3FB\uA3FC\uA3FD\uA3FE\uA3FF\uA400\uA401\uA402\uA403\uA404\uA405\uA406\uA407\uA408\uA409\uA40A\uA40B\uA40C\uA40D\uA40E\uA40F\uA410\uA411\uA412\uA413\uA414\uA415\uA416\uA417\uA418\uA419\uA41A\uA41B\uA41C\uA41D\uA41E\uA41F\uA420\uA421\uA422\uA423\uA424\uA425\uA426\uA427\uA428\uA429\uA42A\uA42B\uA42C\uA42D\uA42E\uA42F\uA430\uA431\uA432\uA433\uA434\uA435\uA436\uA437\uA438\uA439\uA43A\uA43B\uA43C\uA43D\uA43E\uA43F\uA440\uA441\uA442\uA443\uA444\uA445\uA446\uA447\uA448\uA449\uA44A\uA44B\uA44C\uA44D\uA44E\uA44F\uA450\uA451\uA452\uA453\uA454\uA455\uA456\uA457\uA458\uA459\uA45A\uA45B\uA45C\uA45D\uA45E\uA45F\uA460\uA461\uA462\uA463\uA464\uA465\uA466\uA467\uA468\uA469\uA46A\uA46B\uA46C\uA46D\uA46E\uA46F\uA470\uA471\uA472\uA473\uA474\uA475\uA476\uA477\uA478\uA479\uA47A\uA47B\uA47C\uA47D\uA47E\uA47F\uA480\uA481\uA482\uA483\uA484\uA485\uA486\uA487\uA488\uA489\uA48A\uA48B\uA48C\uA4D0\uA4D1\uA4D2\uA4D3\uA4D4\uA4D5\uA4D6\uA4D7\uA4D8\uA4D9\uA4DA\uA4DB\uA4DC\uA4DD\uA4DE\uA4DF\uA4E0\uA4E1\uA4E2\uA4E3\uA4E4\uA4E5\uA4E6\uA4E7\uA4E8\uA4E9\uA4EA\uA4EB\uA4EC\uA4ED\uA4EE\uA4EF\uA4F0\uA4F1\uA4F2\uA4F3\uA4F4\uA4F5\uA4F6\uA4F7\uA500\uA501\uA502\uA503\uA504\uA505\uA506\uA507\uA508\uA509\uA50A\uA50B\uA50C\uA50D\uA50E\uA50F\uA510\uA511\uA512\uA513\uA514\uA515\uA516\uA517\uA518\uA519\uA51A\uA51B\uA51C\uA51D\uA51E\uA51F\uA520\uA521\uA522\uA523\uA524\uA525\uA526\uA527\uA528\uA529\uA52A\uA52B\uA52C\uA52D\uA52E\uA52F\uA530\uA531\uA532\uA533\uA534\uA535\uA536\uA537\uA538\uA539\uA53A\uA53B\uA53C\uA53D\uA53E\uA53F\uA540\uA541\uA542\uA543\uA544\uA545\uA546\uA547\uA548\uA549\uA54A\uA54B\uA54C\uA54D\uA54E\uA54F\uA550\uA551\uA552\uA553\uA554\uA555\uA556\uA557\uA558\uA559\uA55A\uA55B\uA55C\uA55D\uA55E\uA55F\uA560\uA561\uA562\uA563\uA564\uA565\uA566\uA567\uA568\uA569\uA56A\uA56B\uA56C\uA56D\uA56E\uA56F\uA570\uA571\uA572\uA573\uA574\uA575\uA576\uA577\uA578\uA579\uA57A\uA57B\uA57C\uA57D\uA57E\uA57F\uA580\uA581\uA582\uA583\uA584\uA585\uA586\uA587\uA588\uA589\uA58A\uA58B\uA58C\uA58D\uA58E\uA58F\uA590\uA591\uA592\uA593\uA594\uA595\uA596\uA597\uA598\uA599\uA59A\uA59B\uA59C\uA59D\uA59E\uA59F\uA5A0\uA5A1\uA5A2\uA5A3\uA5A4\uA5A5\uA5A6\uA5A7\uA5A8\uA5A9\uA5AA\uA5AB\uA5AC\uA5AD\uA5AE\uA5AF\uA5B0\uA5B1\uA5B2\uA5B3\uA5B4\uA5B5\uA5B6\uA5B7\uA5B8\uA5B9\uA5BA\uA5BB\uA5BC\uA5BD\uA5BE\uA5BF\uA5C0\uA5C1\uA5C2\uA5C3\uA5C4\uA5C5\uA5C6\uA5C7\uA5C8\uA5C9\uA5CA\uA5CB\uA5CC\uA5CD\uA5CE\uA5CF\uA5D0\uA5D1\uA5D2\uA5D3\uA5D4\uA5D5\uA5D6\uA5D7\uA5D8\uA5D9\uA5DA\uA5DB\uA5DC\uA5DD\uA5DE\uA5DF\uA5E0\uA5E1\uA5E2\uA5E3\uA5E4\uA5E5\uA5E6\uA5E7\uA5E8\uA5E9\uA5EA\uA5EB\uA5EC\uA5ED\uA5EE\uA5EF\uA5F0\uA5F1\uA5F2\uA5F3\uA5F4\uA5F5\uA5F6\uA5F7\uA5F8\uA5F9\uA5FA\uA5FB\uA5FC\uA5FD\uA5FE\uA5FF\uA600\uA601\uA602\uA603\uA604\uA605\uA606\uA607\uA608\uA609\uA60A\uA60B\uA610\uA611\uA612\uA613\uA614\uA615\uA616\uA617\uA618\uA619\uA61A\uA61B\uA61C\uA61D\uA61E\uA61F\uA62A\uA62B\uA66E\uA6A0\uA6A1\uA6A2\uA6A3\uA6A4\uA6A5\uA6A6\uA6A7\uA6A8\uA6A9\uA6AA\uA6AB\uA6AC\uA6AD\uA6AE\uA6AF\uA6B0\uA6B1\uA6B2\uA6B3\uA6B4\uA6B5\uA6B6\uA6B7\uA6B8\uA6B9\uA6BA\uA6BB\uA6BC\uA6BD\uA6BE\uA6BF\uA6C0\uA6C1\uA6C2\uA6C3\uA6C4\uA6C5\uA6C6\uA6C7\uA6C8\uA6C9\uA6CA\uA6CB\uA6CC\uA6CD\uA6CE\uA6CF\uA6D0\uA6D1\uA6D2\uA6D3\uA6D4\uA6D5\uA6D6\uA6D7\uA6D8\uA6D9\uA6DA\uA6DB\uA6DC\uA6DD\uA6DE\uA6DF\uA6E0\uA6E1\uA6E2\uA6E3\uA6E4\uA6E5\uA7FB\uA7FC\uA7FD\uA7FE\uA7FF\uA800\uA801\uA803\uA804\uA805\uA807\uA808\uA809\uA80A\uA80C\uA80D\uA80E\uA80F\uA810\uA811\uA812\uA813\uA814\uA815\uA816\uA817\uA818\uA819\uA81A\uA81B\uA81C\uA81D\uA81E\uA81F\uA820\uA821\uA822\uA840\uA841\uA842\uA843\uA844\uA845\uA846\uA847\uA848\uA849\uA84A\uA84B\uA84C\uA84D\uA84E\uA84F\uA850\uA851\uA852\uA853\uA854\uA855\uA856\uA857\uA858\uA859\uA85A\uA85B\uA85C\uA85D\uA85E\uA85F\uA860\uA861\uA862\uA863\uA864\uA865\uA866\uA867\uA868\uA869\uA86A\uA86B\uA86C\uA86D\uA86E\uA86F\uA870\uA871\uA872\uA873\uA882\uA883\uA884\uA885\uA886\uA887\uA888\uA889\uA88A\uA88B\uA88C\uA88D\uA88E\uA88F\uA890\uA891\uA892\uA893\uA894\uA895\uA896\uA897\uA898\uA899\uA89A\uA89B\uA89C\uA89D\uA89E\uA89F\uA8A0\uA8A1\uA8A2\uA8A3\uA8A4\uA8A5\uA8A6\uA8A7\uA8A8\uA8A9\uA8AA\uA8AB\uA8AC\uA8AD\uA8AE\uA8AF\uA8B0\uA8B1\uA8B2\uA8B3\uA8F2\uA8F3\uA8F4\uA8F5\uA8F6\uA8F7\uA8FB\uA90A\uA90B\uA90C\uA90D\uA90E\uA90F\uA910\uA911\uA912\uA913\uA914\uA915\uA916\uA917\uA918\uA919\uA91A\uA91B\uA91C\uA91D\uA91E\uA91F\uA920\uA921\uA922\uA923\uA924\uA925\uA930\uA931\uA932\uA933\uA934\uA935\uA936\uA937\uA938\uA939\uA93A\uA93B\uA93C\uA93D\uA93E\uA93F\uA940\uA941\uA942\uA943\uA944\uA945\uA946\uA960\uA961\uA962\uA963\uA964\uA965\uA966\uA967\uA968\uA969\uA96A\uA96B\uA96C\uA96D\uA96E\uA96F\uA970\uA971\uA972\uA973\uA974\uA975\uA976\uA977\uA978\uA979\uA97A\uA97B\uA97C\uA984\uA985\uA986\uA987\uA988\uA989\uA98A\uA98B\uA98C\uA98D\uA98E\uA98F\uA990\uA991\uA992\uA993\uA994\uA995\uA996\uA997\uA998\uA999\uA99A\uA99B\uA99C\uA99D\uA99E\uA99F\uA9A0\uA9A1\uA9A2\uA9A3\uA9A4\uA9A5\uA9A6\uA9A7\uA9A8\uA9A9\uA9AA\uA9AB\uA9AC\uA9AD\uA9AE\uA9AF\uA9B0\uA9B1\uA9B2\uAA00\uAA01\uAA02\uAA03\uAA04\uAA05\uAA06\uAA07\uAA08\uAA09\uAA0A\uAA0B\uAA0C\uAA0D\uAA0E\uAA0F\uAA10\uAA11\uAA12\uAA13\uAA14\uAA15\uAA16\uAA17\uAA18\uAA19\uAA1A\uAA1B\uAA1C\uAA1D\uAA1E\uAA1F\uAA20\uAA21\uAA22\uAA23\uAA24\uAA25\uAA26\uAA27\uAA28\uAA40\uAA41\uAA42\uAA44\uAA45\uAA46\uAA47\uAA48\uAA49\uAA4A\uAA4B\uAA60\uAA61\uAA62\uAA63\uAA64\uAA65\uAA66\uAA67\uAA68\uAA69\uAA6A\uAA6B\uAA6C\uAA6D\uAA6E\uAA6F\uAA71\uAA72\uAA73\uAA74\uAA75\uAA76\uAA7A\uAA80\uAA81\uAA82\uAA83\uAA84\uAA85\uAA86\uAA87\uAA88\uAA89\uAA8A\uAA8B\uAA8C\uAA8D\uAA8E\uAA8F\uAA90\uAA91\uAA92\uAA93\uAA94\uAA95\uAA96\uAA97\uAA98\uAA99\uAA9A\uAA9B\uAA9C\uAA9D\uAA9E\uAA9F\uAAA0\uAAA1\uAAA2\uAAA3\uAAA4\uAAA5\uAAA6\uAAA7\uAAA8\uAAA9\uAAAA\uAAAB\uAAAC\uAAAD\uAAAE\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9\uAABA\uAABB\uAABC\uAABD\uAAC0\uAAC2\uAADB\uAADC\uAB01\uAB02\uAB03\uAB04\uAB05\uAB06\uAB09\uAB0A\uAB0B\uAB0C\uAB0D\uAB0E\uAB11\uAB12\uAB13\uAB14\uAB15\uAB16\uAB20\uAB21\uAB22\uAB23\uAB24\uAB25\uAB26\uAB28\uAB29\uAB2A\uAB2B\uAB2C\uAB2D\uAB2E\uABC0\uABC1\uABC2\uABC3\uABC4\uABC5\uABC6\uABC7\uABC8\uABC9\uABCA\uABCB\uABCC\uABCD\uABCE\uABCF\uABD0\uABD1\uABD2\uABD3\uABD4\uABD5\uABD6\uABD7\uABD8\uABD9\uABDA\uABDB\uABDC\uABDD\uABDE\uABDF\uABE0\uABE1\uABE2\uAC00\uD7A3\uD7B0\uD7B1\uD7B2\uD7B3\uD7B4\uD7B5\uD7B6\uD7B7\uD7B8\uD7B9\uD7BA\uD7BB\uD7BC\uD7BD\uD7BE\uD7BF\uD7C0\uD7C1\uD7C2\uD7C3\uD7C4\uD7C5\uD7C6\uD7CB\uD7CC\uD7CD\uD7CE\uD7CF\uD7D0\uD7D1\uD7D2\uD7D3\uD7D4\uD7D5\uD7D6\uD7D7\uD7D8\uD7D9\uD7DA\uD7DB\uD7DC\uD7DD\uD7DE\uD7DF\uD7E0\uD7E1\uD7E2\uD7E3\uD7E4\uD7E5\uD7E6\uD7E7\uD7E8\uD7E9\uD7EA\uD7EB\uD7EC\uD7ED\uD7EE\uD7EF\uD7F0\uD7F1\uD7F2\uD7F3\uD7F4\uD7F5\uD7F6\uD7F7\uD7F8\uD7F9\uD7FA\uD7FB\u8C48\u66F4\u8ECA\u8CC8\u6ED1\u4E32\u53E5\u9F9C\u9F9C\u5951\u91D1\u5587\u5948\u61F6\u7669\u7F85\u863F\u87BA\u88F8\u908F\u6A02\u6D1B\u70D9\u73DE\u843D\u916A\u99F1\u4E82\u5375\u6B04\u721B\u862D\u9E1E\u5D50\u6FEB\u85CD\u8964\u62C9\u81D8\u881F\u5ECA\u6717\u6D6A\u72FC\u90CE\u4F86\u51B7\u52DE\u64C4\u6AD3\u7210\u76E7\u8001\u8606\u865C\u8DEF\u9732\u9B6F\u9DFA\u788C\u797F\u7DA0\u83C9\u9304\u9E7F\u8AD6\u58DF\u5F04\u7C60\u807E\u7262\u78CA\u8CC2\u96F7\u58D8\u5C62\u6A13\u6DDA\u6F0F\u7D2F\u7E37\u964B\u52D2\u808B\u51DC\u51CC\u7A1C\u7DBE\u83F1\u9675\u8B80\u62CF\u6A02\u8AFE\u4E39\u5BE7\u6012\u7387\u7570\u5317\u78FB\u4FBF\u5FA9\u4E0D\u6CCC\u6578\u7D22\u53C3\u585E\u7701\u8449\u8AAA\u6BBA\u8FB0\u6C88\u62FE\u82E5\u63A0\u7565\u4EAE\u5169\u51C9\u6881\u7CE7\u826F\u8AD2\u91CF\u52F5\u5442\u5973\u5EEC\u65C5\u6FFE\u792A\u95AD\u9A6A\u9E97\u9ECE\u529B\u66C6\u6B77\u8F62\u5E74\u6190\u6200\u649A\u6F23\u7149\u7489\u79CA\u7DF4\u806F\u8F26\u84EE\u9023\u934A\u5217\u52A3\u54BD\u70C8\u88C2\u8AAA\u5EC9\u5FF5\u637B\u6BAE\u7C3E\u7375\u4EE4\u56F9\u5BE7\u5DBA\u601C\u73B2\u7469\u7F9A\u8046\u9234\u96F6\u9748\u9818\u4F8B\u79AE\u91B4\u96B8\u60E1\u4E86\u50DA\u5BEE\u5C3F\u6599\u6A02\u71CE\u7642\u84FC\u907C\u9F8D\u6688\u962E\u5289\u677B\u67F3\u6D41\u6E9C\u7409\u7559\u786B\u7D10\u985E\u516D\u622E\u9678\u502B\u5D19\u6DEA\u8F2A\u5F8B\u6144\u6817\u7387\u9686\u5229\u540F\u5C65\u6613\u674E\u68A8\u6CE5\u7406\u75E2\u7F79\u88CF\u88E1\u91CC\u96E2\u533F\u6EBA\u541D\u71D0\u7498\u85FA\u96A3\u9C57\u9E9F\u6797\u6DCB\u81E8\u7ACB\u7B20\u7C92\u72C0\u7099\u8B58\u4EC0\u8336\u523A\u5207\u5EA6\u62D3\u7CD6\u5B85\u6D1E\u66B4\u8F3B\u884C\u964D\u898B\u5ED3\u5140\u55C0\uFA0E\uFA0F\u585A\uFA11\u6674\uFA13\uFA14\u51DE\u732A\u76CA\u793C\u795E\u7965\u798F\u9756\u7CBE\u7FBD\uFA1F\u8612\uFA21\u8AF8\uFA23\uFA24\u9038\u90FD\uFA27\uFA28\uFA29\u98EF\u98FC\u9928\u9DB4\u4FAE\u50E7\u514D\u52C9\u52E4\u5351\u559D\u5606\u5668\u5840\u58A8\u5C64\u5C6E\u6094\u6168\u618E\u61F2\u654F\u65E2\u6691\u6885\u6D77\u6E1A\u6F22\u716E\u722B\u7422\u7891\u793E\u7949\u7948\u7950\u7956\u795D\u798D\u798E\u7A40\u7A81\u7BC0\u7DF4\u7E09\u7E41\u7F72\u8005\u81ED\u8279\u8279\u8457\u8910\u8996\u8B01\u8B39\u8CD3\u8D08\u8FB6\u9038\u96E3\u97FF\u983B\u6075\uD850\uDEEE\u8218\u4E26\u51B5\u5168\u4F80\u5145\u5180\u52C7\u52FA\u559D\u5555\u5599\u55E2\u585A\u58B3\u5944\u5954\u5A62\u5B28\u5ED2\u5ED9\u5F69\u5FAD\u60D8\u614E\u6108\u618E\u6160\u61F2\u6234\u63C4\u641C\u6452\u6556\u6674\u6717\u671B\u6756\u6B79\u6BBA\u6D41\u6EDB\u6ECB\u6F22\u701E\u716E\u77A7\u7235\u72AF\u732A\u7471\u7506\u753B\u761D\u761F\u76CA\u76DB\u76F4\u774A\u7740\u78CC\u7AB1\u7BC0\u7C7B\u7D5B\u7DF4\u7F3E\u8005\u8352\u83EF\u8779\u8941\u8986\u8996\u8ABF\u8AF8\u8ACB\u8B01\u8AFE\u8AED\u8B39\u8B8A\u8D08\u8F38\u9072\u9199\u9276\u967C\u96E3\u9756\u97DB\u97FF\u980B\u983B\u9B12\u9F9C\uD84A\uDC4A\uD84A\uDC44\uD84C\uDFD5\u3B9D\u4018\u4039\uD854\uDE49\uD857\uDCD0\uD85F\uDED3\u9F43\u9F8E\u05D9\u05B4\u05F2\u05B7\uFB20\uFB21\uFB22\uFB23\uFB24\uFB25\uFB26\uFB27\uFB28\u05E9\u05C1\u05E9\u05C2\u05E9\u05BC\u05C1\u05E9\u05BC\u05C2\u05D0\u05B7\u05D0\u05B8\u05D0\u05BC\u05D1\u05BC\u05D2\u05BC\u05D3\u05BC\u05D4\u05BC\u05D5\u05BC\u05D6\u05BC\u05D8\u05BC\u05D9\u05BC\u05DA\u05BC\u05DB\u05BC\u05DC\u05BC\u05DE\u05BC\u05E0\u05BC\u05E1\u05BC\u05E3\u05BC\u05E4\u05BC\u05E6\u05BC\u05E7\u05BC\u05E8\u05BC\u05E9\u05BC\u05EA\u05BC\u05D5\u05B9\u05D1\u05BF\u05DB\u05BF\u05E4\u05BF\uFB4F\uFB50\uFB51\uFB52\uFB53\uFB54\uFB55\uFB56\uFB57\uFB58\uFB59\uFB5A\uFB5B\uFB5C\uFB5D\uFB5E\uFB5F\uFB60\uFB61\uFB62\uFB63\uFB64\uFB65\uFB66\uFB67\uFB68\uFB69\uFB6A\uFB6B\uFB6C\uFB6D\uFB6E\uFB6F\uFB70\uFB71\uFB72\uFB73\uFB74\uFB75\uFB76\uFB77\uFB78\uFB79\uFB7A\uFB7B\uFB7C\uFB7D\uFB7E\uFB7F\uFB80\uFB81\uFB82\uFB83\uFB84\uFB85\uFB86\uFB87\uFB88\uFB89\uFB8A\uFB8B\uFB8C\uFB8D\uFB8E\uFB8F\uFB90\uFB91\uFB92\uFB93\uFB94\uFB95\uFB96\uFB97\uFB98\uFB99\uFB9A\uFB9B\uFB9C\uFB9D\uFB9E\uFB9F\uFBA0\uFBA1\uFBA2\uFBA3\uFBA4\uFBA5\uFBA6\uFBA7\uFBA8\uFBA9\uFBAA\uFBAB\uFBAC\uFBAD\uFBAE\uFBAF\uFBB0\uFBB1\uFBD3\uFBD4\uFBD5\uFBD6\uFBD7\uFBD8\uFBD9\uFBDA\uFBDB\uFBDC\uFBDD\uFBDE\uFBDF\uFBE0\uFBE1\uFBE2\uFBE3\uFBE4\uFBE5\uFBE6\uFBE7\uFBE8\uFBE9\uFBEA\uFBEB\uFBEC\uFBED\uFBEE\uFBEF\uFBF0\uFBF1\uFBF2\uFBF3\uFBF4\uFBF5\uFBF6\uFBF7\uFBF8\uFBF9\uFBFA\uFBFB\uFBFC\uFBFD\uFBFE\uFBFF\uFC00\uFC01\uFC02\uFC03\uFC04\uFC05\uFC06\uFC07\uFC08\uFC09\uFC0A\uFC0B\uFC0C\uFC0D\uFC0E\uFC0F\uFC10\uFC11\uFC12\uFC13\uFC14\uFC15\uFC16\uFC17\uFC18\uFC19\uFC1A\uFC1B\uFC1C\uFC1D\uFC1E\uFC1F\uFC20\uFC21\uFC22\uFC23\uFC24\uFC25\uFC26\uFC27\uFC28\uFC29\uFC2A\uFC2B\uFC2C\uFC2D\uFC2E\uFC2F\uFC30\uFC31\uFC32\uFC33\uFC34\uFC35\uFC36\uFC37\uFC38\uFC39\uFC3A\uFC3B\uFC3C\uFC3D\uFC3E\uFC3F\uFC40\uFC41\uFC42\uFC43\uFC44\uFC45\uFC46\uFC47\uFC48\uFC49\uFC4A\uFC4B\uFC4C\uFC4D\uFC4E\uFC4F\uFC50\uFC51\uFC52\uFC53\uFC54\uFC55\uFC56\uFC57\uFC58\uFC59\uFC5A\uFC5B\uFC5C\uFC5D\uFC5E\uFC5F\uFC60\uFC61\uFC62\uFC63\uFC64\uFC65\uFC66\uFC67\uFC68\uFC69\uFC6A\uFC6B\uFC6C\uFC6D\uFC6E\uFC6F\uFC70\uFC71\uFC72\uFC73\uFC74\uFC75\uFC76\uFC77\uFC78\uFC79\uFC7A\uFC7B\uFC7C\uFC7D\uFC7E\uFC7F\uFC80\uFC81\uFC82\uFC83\uFC84\uFC85\uFC86\uFC87\uFC88\uFC89\uFC8A\uFC8B\uFC8C\uFC8D\uFC8E\uFC8F\uFC90\uFC91\uFC92\uFC93\uFC94\uFC95\uFC96\uFC97\uFC98\uFC99\uFC9A\uFC9B\uFC9C\uFC9D\uFC9E\uFC9F\uFCA0\uFCA1\uFCA2\uFCA3\uFCA4\uFCA5\uFCA6\uFCA7\uFCA8\uFCA9\uFCAA\uFCAB\uFCAC\uFCAD\uFCAE\uFCAF\uFCB0\uFCB1\uFCB2\uFCB3\uFCB4\uFCB5\uFCB6\uFCB7\uFCB8\uFCB9\uFCBA\uFCBB\uFCBC\uFCBD\uFCBE\uFCBF\uFCC0\uFCC1\uFCC2\uFCC3\uFCC4\uFCC5\uFCC6\uFCC7\uFCC8\uFCC9\uFCCA\uFCCB\uFCCC\uFCCD\uFCCE\uFCCF\uFCD0\uFCD1\uFCD2\uFCD3\uFCD4\uFCD5\uFCD6\uFCD7\uFCD8\uFCD9\uFCDA\uFCDB\uFCDC\uFCDD\uFCDE\uFCDF\uFCE0\uFCE1\uFCE2\uFCE3\uFCE4\uFCE5\uFCE6\uFCE7\uFCE8\uFCE9\uFCEA\uFCEB\uFCEC\uFCED\uFCEE\uFCEF\uFCF0\uFCF1\uFCF2\uFCF3\uFCF4\uFCF5\uFCF6\uFCF7\uFCF8\uFCF9\uFCFA\uFCFB\uFCFC\uFCFD\uFCFE\uFCFF\uFD00\uFD01\uFD02\uFD03\uFD04\uFD05\uFD06\uFD07\uFD08\uFD09\uFD0A\uFD0B\uFD0C\uFD0D\uFD0E\uFD0F\uFD10\uFD11\uFD12\uFD13\uFD14\uFD15\uFD16\uFD17\uFD18\uFD19\uFD1A\uFD1B\uFD1C\uFD1D\uFD1E\uFD1F\uFD20\uFD21\uFD22\uFD23\uFD24\uFD25\uFD26\uFD27\uFD28\uFD29\uFD2A\uFD2B\uFD2C\uFD2D\uFD2E\uFD2F\uFD30\uFD31\uFD32\uFD33\uFD34\uFD35\uFD36\uFD37\uFD38\uFD39\uFD3A\uFD3B\uFD3C\uFD3D\uFD50\uFD51\uFD52\uFD53\uFD54\uFD55\uFD56\uFD57\uFD58\uFD59\uFD5A\uFD5B\uFD5C\uFD5D\uFD5E\uFD5F\uFD60\uFD61\uFD62\uFD63\uFD64\uFD65\uFD66\uFD67\uFD68\uFD69\uFD6A\uFD6B\uFD6C\uFD6D\uFD6E\uFD6F\uFD70\uFD71\uFD72\uFD73\uFD74\uFD75\uFD76\uFD77\uFD78\uFD79\uFD7A\uFD7B\uFD7C\uFD7D\uFD7E\uFD7F\uFD80\uFD81\uFD82\uFD83\uFD84\uFD85\uFD86\uFD87\uFD88\uFD89\uFD8A\uFD8B\uFD8C\uFD8D\uFD8E\uFD8F\uFD92\uFD93\uFD94\uFD95\uFD96\uFD97\uFD98\uFD99\uFD9A\uFD9B\uFD9C\uFD9D\uFD9E\uFD9F\uFDA0\uFDA1\uFDA2\uFDA3\uFDA4\uFDA5\uFDA6\uFDA7\uFDA8\uFDA9\uFDAA\uFDAB\uFDAC\uFDAD\uFDAE\uFDAF\uFDB0\uFDB1\uFDB2\uFDB3\uFDB4\uFDB5\uFDB6\uFDB7\uFDB8\uFDB9\uFDBA\uFDBB\uFDBC\uFDBD\uFDBE\uFDBF\uFDC0\uFDC1\uFDC2\uFDC3\uFDC4\uFDC5\uFDC6\uFDC7\uFDF0\uFDF1\uFDF2\uFDF3\uFDF4\uFDF5\uFDF6\uFDF7\uFDF8\uFDF9\uFDFA\uFDFB\uFE70\uFE71\uFE72\uFE73\uFE74\uFE76\uFE77\uFE78\uFE79\uFE7A\uFE7B\uFE7C\uFE7D\uFE7E\uFE7F\uFE80\uFE81\uFE82\uFE83\uFE84\uFE85\uFE86\uFE87\uFE88\uFE89\uFE8A\uFE8B\uFE8C\uFE8D\uFE8E\uFE8F\uFE90\uFE91\uFE92\uFE93\uFE94\uFE95\uFE96\uFE97\uFE98\uFE99\uFE9A\uFE9B\uFE9C\uFE9D\uFE9E\uFE9F\uFEA0\uFEA1\uFEA2\uFEA3\uFEA4\uFEA5\uFEA6\uFEA7\uFEA8\uFEA9\uFEAA\uFEAB\uFEAC\uFEAD\uFEAE\uFEAF\uFEB0\uFEB1\uFEB2\uFEB3\uFEB4\uFEB5\uFEB6\uFEB7\uFEB8\uFEB9\uFEBA\uFEBB\uFEBC\uFEBD\uFEBE\uFEBF\uFEC0\uFEC1\uFEC2\uFEC3\uFEC4\uFEC5\uFEC6\uFEC7\uFEC8\uFEC9\uFECA\uFECB\uFECC\uFECD\uFECE\uFECF\uFED0\uFED1\uFED2\uFED3\uFED4\uFED5\uFED6\uFED7\uFED8\uFED9\uFEDA\uFEDB\uFEDC\uFEDD\uFEDE\uFEDF\uFEE0\uFEE1\uFEE2\uFEE3\uFEE4\uFEE5\uFEE6\uFEE7\uFEE8\uFEE9\uFEEA\uFEEB\uFEEC\uFEED\uFEEE\uFEEF\uFEF0\uFEF1\uFEF2\uFEF3\uFEF4\uFEF5\uFEF6\uFEF7\uFEF8\uFEF9\uFEFA\uFEFB\uFEFC\uFF66\uFF67\uFF68\uFF69\uFF6A\uFF6B\uFF6C\uFF6D\uFF6E\uFF6F\uFF71\uFF72\uFF73\uFF74\uFF75\uFF76\uFF77\uFF78\uFF79\uFF7A\uFF7B\uFF7C\uFF7D\uFF7E\uFF7F\uFF80\uFF81\uFF82\uFF83\uFF84\uFF85\uFF86\uFF87\uFF88\uFF89\uFF8A\uFF8B\uFF8C\uFF8D\uFF8E\uFF8F\uFF90\uFF91\uFF92\uFF93\uFF94\uFF95\uFF96\uFF97\uFF98\uFF99\uFF9A\uFF9B\uFF9C\uFF9D\uFFA0\uFFA1\uFFA2\uFFA3\uFFA4\uFFA5\uFFA6\uFFA7\uFFA8\uFFA9\uFFAA\uFFAB\uFFAC\uFFAD\uFFAE\uFFAF\uFFB0\uFFB1\uFFB2\uFFB3\uFFB4\uFFB5\uFFB6\uFFB7\uFFB8\uFFB9\uFFBA\uFFBB\uFFBC\uFFBD\uFFBE\uFFC2\uFFC3\uFFC4\uFFC5\uFFC6\uFFC7\uFFCA\uFFCB\uFFCC\uFFCD\uFFCE\uFFCF\uFFD2\uFFD3\uFFD4\uFFD5\uFFD6\uFFD7\uFFDA\uFFDB\uFFDC\uD800\uDC00\uD800\uDC01\uD800\uDC02\uD800\uDC03\uD800\uDC04\uD800\uDC05\uD800\uDC06\uD800\uDC07\uD800\uDC08\uD800\uDC09\uD800\uDC0A\uD800\uDC0B\uD800\uDC0D\uD800\uDC0E\uD800\uDC0F\uD800\uDC10\uD800\uDC11\uD800\uDC12\uD800\uDC13\uD800\uDC14\uD800\uDC15\uD800\uDC16\uD800\uDC17\uD800\uDC18\uD800\uDC19\uD800\uDC1A\uD800\uDC1B\uD800\uDC1C\uD800\uDC1D\uD800\uDC1E\uD800\uDC1F\uD800\uDC20\uD800\uDC21\uD800\uDC22\uD800\uDC23\uD800\uDC24\uD800\uDC25\uD800\uDC26\uD800\uDC28\uD800\uDC29\uD800\uDC2A\uD800\uDC2B\uD800\uDC2C\uD800\uDC2D\uD800\uDC2E\uD800\uDC2F\uD800\uDC30\uD800\uDC31\uD800\uDC32\uD800\uDC33\uD800\uDC34\uD800\uDC35\uD800\uDC36\uD800\uDC37\uD800\uDC38\uD800\uDC39\uD800\uDC3A\uD800\uDC3C\uD800\uDC3D\uD800\uDC3F\uD800\uDC40\uD800\uDC41\uD800\uDC42\uD800\uDC43\uD800\uDC44\uD800\uDC45\uD800\uDC46\uD800\uDC47\uD800\uDC48\uD800\uDC49\uD800\uDC4A\uD800\uDC4B\uD800\uDC4C\uD800\uDC4D\uD800\uDC50\uD800\uDC51\uD800\uDC52\uD800\uDC53\uD800\uDC54\uD800\uDC55\uD800\uDC56\uD800\uDC57\uD800\uDC58\uD800\uDC59\uD800\uDC5A\uD800\uDC5B\uD800\uDC5C\uD800\uDC5D\uD800\uDC80\uD800\uDC81\uD800\uDC82\uD800\uDC83\uD800\uDC84\uD800\uDC85\uD800\uDC86\uD800\uDC87\uD800\uDC88\uD800\uDC89\uD800\uDC8A\uD800\uDC8B\uD800\uDC8C\uD800\uDC8D\uD800\uDC8E\uD800\uDC8F\uD800\uDC90\uD800\uDC91\uD800\uDC92\uD800\uDC93\uD800\uDC94\uD800\uDC95\uD800\uDC96\uD800\uDC97\uD800\uDC98\uD800\uDC99\uD800\uDC9A\uD800\uDC9B\uD800\uDC9C\uD800\uDC9D\uD800\uDC9E\uD800\uDC9F\uD800\uDCA0\uD800\uDCA1\uD800\uDCA2\uD800\uDCA3\uD800\uDCA4\uD800\uDCA5\uD800\uDCA6\uD800\uDCA7\uD800\uDCA8\uD800\uDCA9\uD800\uDCAA\uD800\uDCAB\uD800\uDCAC\uD800\uDCAD\uD800\uDCAE\uD800\uDCAF\uD800\uDCB0\uD800\uDCB1\uD800\uDCB2\uD800\uDCB3\uD800\uDCB4\uD800\uDCB5\uD800\uDCB6\uD800\uDCB7\uD800\uDCB8\uD800\uDCB9\uD800\uDCBA\uD800\uDCBB\uD800\uDCBC\uD800\uDCBD\uD800\uDCBE\uD800\uDCBF\uD800\uDCC0\uD800\uDCC1\uD800\uDCC2\uD800\uDCC3\uD800\uDCC4\uD800\uDCC5\uD800\uDCC6\uD800\uDCC7\uD800\uDCC8\uD800\uDCC9\uD800\uDCCA\uD800\uDCCB\uD800\uDCCC\uD800\uDCCD\uD800\uDCCE\uD800\uDCCF\uD800\uDCD0\uD800\uDCD1\uD800\uDCD2\uD800\uDCD3\uD800\uDCD4\uD800\uDCD5\uD800\uDCD6\uD800\uDCD7\uD800\uDCD8\uD800\uDCD9\uD800\uDCDA\uD800\uDCDB\uD800\uDCDC\uD800\uDCDD\uD800\uDCDE\uD800\uDCDF\uD800\uDCE0\uD800\uDCE1\uD800\uDCE2\uD800\uDCE3\uD800\uDCE4\uD800\uDCE5\uD800\uDCE6\uD800\uDCE7\uD800\uDCE8\uD800\uDCE9\uD800\uDCEA\uD800\uDCEB\uD800\uDCEC\uD800\uDCED\uD800\uDCEE\uD800\uDCEF\uD800\uDCF0\uD800\uDCF1\uD800\uDCF2\uD800\uDCF3\uD800\uDCF4\uD800\uDCF5\uD800\uDCF6\uD800\uDCF7\uD800\uDCF8\uD800\uDCF9\uD800\uDCFA\uD800\uDE80\uD800\uDE81\uD800\uDE82\uD800\uDE83\uD800\uDE84\uD800\uDE85\uD800\uDE86\uD800\uDE87\uD800\uDE88\uD800\uDE89\uD800\uDE8A\uD800\uDE8B\uD800\uDE8C\uD800\uDE8D\uD800\uDE8E\uD800\uDE8F\uD800\uDE90\uD800\uDE91\uD800\uDE92\uD800\uDE93\uD800\uDE94\uD800\uDE95\uD800\uDE96\uD800\uDE97\uD800\uDE98\uD800\uDE99\uD800\uDE9A\uD800\uDE9B\uD800\uDE9C\uD800\uDEA0\uD800\uDEA1\uD800\uDEA2\uD800\uDEA3\uD800\uDEA4\uD800\uDEA5\uD800\uDEA6\uD800\uDEA7\uD800\uDEA8\uD800\uDEA9\uD800\uDEAA\uD800\uDEAB\uD800\uDEAC\uD800\uDEAD\uD800\uDEAE\uD800\uDEAF\uD800\uDEB0\uD800\uDEB1\uD800\uDEB2\uD800\uDEB3\uD800\uDEB4\uD800\uDEB5\uD800\uDEB6\uD800\uDEB7\uD800\uDEB8\uD800\uDEB9\uD800\uDEBA\uD800\uDEBB\uD800\uDEBC\uD800\uDEBD\uD800\uDEBE\uD800\uDEBF\uD800\uDEC0\uD800\uDEC1\uD800\uDEC2\uD800\uDEC3\uD800\uDEC4\uD800\uDEC5\uD800\uDEC6\uD800\uDEC7\uD800\uDEC8\uD800\uDEC9\uD800\uDECA\uD800\uDECB\uD800\uDECC\uD800\uDECD\uD800\uDECE\uD800\uDECF\uD800\uDED0\uD800\uDF00\uD800\uDF01\uD800\uDF02\uD800\uDF03\uD800\uDF04\uD800\uDF05\uD800\uDF06\uD800\uDF07\uD800\uDF08\uD800\uDF09\uD800\uDF0A\uD800\uDF0B\uD800\uDF0C\uD800\uDF0D\uD800\uDF0E\uD800\uDF0F\uD800\uDF10\uD800\uDF11\uD800\uDF12\uD800\uDF13\uD800\uDF14\uD800\uDF15\uD800\uDF16\uD800\uDF17\uD800\uDF18\uD800\uDF19\uD800\uDF1A\uD800\uDF1B\uD800\uDF1C\uD800\uDF1D\uD800\uDF1E\uD800\uDF30\uD800\uDF31\uD800\uDF32\uD800\uDF33\uD800\uDF34\uD800\uDF35\uD800\uDF36\uD800\uDF37\uD800\uDF38\uD800\uDF39\uD800\uDF3A\uD800\uDF3B\uD800\uDF3C\uD800\uDF3D\uD800\uDF3E\uD800\uDF3F\uD800\uDF40\uD800\uDF42\uD800\uDF43\uD800\uDF44\uD800\uDF45\uD800\uDF46\uD800\uDF47\uD800\uDF48\uD800\uDF49\uD800\uDF80\uD800\uDF81\uD800\uDF82\uD800\uDF83\uD800\uDF84\uD800\uDF85\uD800\uDF86\uD800\uDF87\uD800\uDF88\uD800\uDF89\uD800\uDF8A\uD800\uDF8B\uD800\uDF8C\uD800\uDF8D\uD800\uDF8E\uD800\uDF8F\uD800\uDF90\uD800\uDF91\uD800\uDF92\uD800\uDF93\uD800\uDF94\uD800\uDF95\uD800\uDF96\uD800\uDF97\uD800\uDF98\uD800\uDF99\uD800\uDF9A\uD800\uDF9B\uD800\uDF9C\uD800\uDF9D\uD800\uDFA0\uD800\uDFA1\uD800\uDFA2\uD800\uDFA3\uD800\uDFA4\uD800\uDFA5\uD800\uDFA6\uD800\uDFA7\uD800\uDFA8\uD800\uDFA9\uD800\uDFAA\uD800\uDFAB\uD800\uDFAC\uD800\uDFAD\uD800\uDFAE\uD800\uDFAF\uD800\uDFB0\uD800\uDFB1\uD800\uDFB2\uD800\uDFB3\uD800\uDFB4\uD800\uDFB5\uD800\uDFB6\uD800\uDFB7\uD800\uDFB8\uD800\uDFB9\uD800\uDFBA\uD800\uDFBB\uD800\uDFBC\uD800\uDFBD\uD800\uDFBE\uD800\uDFBF\uD800\uDFC0\uD800\uDFC1\uD800\uDFC2\uD800\uDFC3\uD800\uDFC8\uD800\uDFC9\uD800\uDFCA\uD800\uDFCB\uD800\uDFCC\uD800\uDFCD\uD800\uDFCE\uD800\uDFCF\uD801\uDC50\uD801\uDC51\uD801\uDC52\uD801\uDC53\uD801\uDC54\uD801\uDC55\uD801\uDC56\uD801\uDC57\uD801\uDC58\uD801\uDC59\uD801\uDC5A\uD801\uDC5B\uD801\uDC5C\uD801\uDC5D\uD801\uDC5E\uD801\uDC5F\uD801\uDC60\uD801\uDC61\uD801\uDC62\uD801\uDC63\uD801\uDC64\uD801\uDC65\uD801\uDC66\uD801\uDC67\uD801\uDC68\uD801\uDC69\uD801\uDC6A\uD801\uDC6B\uD801\uDC6C\uD801\uDC6D\uD801\uDC6E\uD801\uDC6F\uD801\uDC70\uD801\uDC71\uD801\uDC72\uD801\uDC73\uD801\uDC74\uD801\uDC75\uD801\uDC76\uD801\uDC77\uD801\uDC78\uD801\uDC79\uD801\uDC7A\uD801\uDC7B\uD801\uDC7C\uD801\uDC7D\uD801\uDC7E\uD801\uDC7F\uD801\uDC80\uD801\uDC81\uD801\uDC82\uD801\uDC83\uD801\uDC84\uD801\uDC85\uD801\uDC86\uD801\uDC87\uD801\uDC88\uD801\uDC89\uD801\uDC8A\uD801\uDC8B\uD801\uDC8C\uD801\uDC8D\uD801\uDC8E\uD801\uDC8F\uD801\uDC90\uD801\uDC91\uD801\uDC92\uD801\uDC93\uD801\uDC94\uD801\uDC95\uD801\uDC96\uD801\uDC97\uD801\uDC98\uD801\uDC99\uD801\uDC9A\uD801\uDC9B\uD801\uDC9C\uD801\uDC9D\uD802\uDC00\uD802\uDC01\uD802\uDC02\uD802\uDC03\uD802\uDC04\uD802\uDC05\uD802\uDC08\uD802\uDC0A\uD802\uDC0B\uD802\uDC0C\uD802\uDC0D\uD802\uDC0E\uD802\uDC0F\uD802\uDC10\uD802\uDC11\uD802\uDC12\uD802\uDC13\uD802\uDC14\uD802\uDC15\uD802\uDC16\uD802\uDC17\uD802\uDC18\uD802\uDC19\uD802\uDC1A\uD802\uDC1B\uD802\uDC1C\uD802\uDC1D\uD802\uDC1E\uD802\uDC1F\uD802\uDC20\uD802\uDC21\uD802\uDC22\uD802\uDC23\uD802\uDC24\uD802\uDC25\uD802\uDC26\uD802\uDC27\uD802\uDC28\uD802\uDC29\uD802\uDC2A\uD802\uDC2B\uD802\uDC2C\uD802\uDC2D\uD802\uDC2E\uD802\uDC2F\uD802\uDC30\uD802\uDC31\uD802\uDC32\uD802\uDC33\uD802\uDC34\uD802\uDC35\uD802\uDC37\uD802\uDC38\uD802\uDC3C\uD802\uDC3F\uD802\uDC40\uD802\uDC41\uD802\uDC42\uD802\uDC43\uD802\uDC44\uD802\uDC45\uD802\uDC46\uD802\uDC47\uD802\uDC48\uD802\uDC49\uD802\uDC4A\uD802\uDC4B\uD802\uDC4C\uD802\uDC4D\uD802\uDC4E\uD802\uDC4F\uD802\uDC50\uD802\uDC51\uD802\uDC52\uD802\uDC53\uD802\uDC54\uD802\uDC55\uD802\uDD00\uD802\uDD01\uD802\uDD02\uD802\uDD03\uD802\uDD04\uD802\uDD05\uD802\uDD06\uD802\uDD07\uD802\uDD08\uD802\uDD09\uD802\uDD0A\uD802\uDD0B\uD802\uDD0C\uD802\uDD0D\uD802\uDD0E\uD802\uDD0F\uD802\uDD10\uD802\uDD11\uD802\uDD12\uD802\uDD13\uD802\uDD14\uD802\uDD15\uD802\uDD20\uD802\uDD21\uD802\uDD22\uD802\uDD23\uD802\uDD24\uD802\uDD25\uD802\uDD26\uD802\uDD27\uD802\uDD28\uD802\uDD29\uD802\uDD2A\uD802\uDD2B\uD802\uDD2C\uD802\uDD2D\uD802\uDD2E\uD802\uDD2F\uD802\uDD30\uD802\uDD31\uD802\uDD32\uD802\uDD33\uD802\uDD34\uD802\uDD35\uD802\uDD36\uD802\uDD37\uD802\uDD38\uD802\uDD39\uD802\uDE00\uD802\uDE10\uD802\uDE11\uD802\uDE12\uD802\uDE13\uD802\uDE15\uD802\uDE16\uD802\uDE17\uD802\uDE19\uD802\uDE1A\uD802\uDE1B\uD802\uDE1C\uD802\uDE1D\uD802\uDE1E\uD802\uDE1F\uD802\uDE20\uD802\uDE21\uD802\uDE22\uD802\uDE23\uD802\uDE24\uD802\uDE25\uD802\uDE26\uD802\uDE27\uD802\uDE28\uD802\uDE29\uD802\uDE2A\uD802\uDE2B\uD802\uDE2C\uD802\uDE2D\uD802\uDE2E\uD802\uDE2F\uD802\uDE30\uD802\uDE31\uD802\uDE32\uD802\uDE33\uD802\uDE60\uD802\uDE61\uD802\uDE62\uD802\uDE63\uD802\uDE64\uD802\uDE65\uD802\uDE66\uD802\uDE67\uD802\uDE68\uD802\uDE69\uD802\uDE6A\uD802\uDE6B\uD802\uDE6C\uD802\uDE6D\uD802\uDE6E\uD802\uDE6F\uD802\uDE70\uD802\uDE71\uD802\uDE72\uD802\uDE73\uD802\uDE74\uD802\uDE75\uD802\uDE76\uD802\uDE77\uD802\uDE78\uD802\uDE79\uD802\uDE7A\uD802\uDE7B\uD802\uDE7C\uD802\uDF00\uD802\uDF01\uD802\uDF02\uD802\uDF03\uD802\uDF04\uD802\uDF05\uD802\uDF06\uD802\uDF07\uD802\uDF08\uD802\uDF09\uD802\uDF0A\uD802\uDF0B\uD802\uDF0C\uD802\uDF0D\uD802\uDF0E\uD802\uDF0F\uD802\uDF10\uD802\uDF11\uD802\uDF12\uD802\uDF13\uD802\uDF14\uD802\uDF15\uD802\uDF16\uD802\uDF17\uD802\uDF18\uD802\uDF19\uD802\uDF1A\uD802\uDF1B\uD802\uDF1C\uD802\uDF1D\uD802\uDF1E\uD802\uDF1F\uD802\uDF20\uD802\uDF21\uD802\uDF22\uD802\uDF23\uD802\uDF24\uD802\uDF25\uD802\uDF26\uD802\uDF27\uD802\uDF28\uD802\uDF29\uD802\uDF2A\uD802\uDF2B\uD802\uDF2C\uD802\uDF2D\uD802\uDF2E\uD802\uDF2F\uD802\uDF30\uD802\uDF31\uD802\uDF32\uD802\uDF33\uD802\uDF34\uD802\uDF35\uD802\uDF40\uD802\uDF41\uD802\uDF42\uD802\uDF43\uD802\uDF44\uD802\uDF45\uD802\uDF46\uD802\uDF47\uD802\uDF48\uD802\uDF49\uD802\uDF4A\uD802\uDF4B\uD802\uDF4C\uD802\uDF4D\uD802\uDF4E\uD802\uDF4F\uD802\uDF50\uD802\uDF51\uD802\uDF52\uD802\uDF53\uD802\uDF54\uD802\uDF55\uD802\uDF60\uD802\uDF61\uD802\uDF62\uD802\uDF63\uD802\uDF64\uD802\uDF65\uD802\uDF66\uD802\uDF67\uD802\uDF68\uD802\uDF69\uD802\uDF6A\uD802\uDF6B\uD802\uDF6C\uD802\uDF6D\uD802\uDF6E\uD802\uDF6F\uD802\uDF70\uD802\uDF71\uD802\uDF72\uD803\uDC00\uD803\uDC01\uD803\uDC02\uD803\uDC03\uD803\uDC04\uD803\uDC05\uD803\uDC06\uD803\uDC07\uD803\uDC08\uD803\uDC09\uD803\uDC0A\uD803\uDC0B\uD803\uDC0C\uD803\uDC0D\uD803\uDC0E\uD803\uDC0F\uD803\uDC10\uD803\uDC11\uD803\uDC12\uD803\uDC13\uD803\uDC14\uD803\uDC15\uD803\uDC16\uD803\uDC17\uD803\uDC18\uD803\uDC19\uD803\uDC1A\uD803\uDC1B\uD803\uDC1C\uD803\uDC1D\uD803\uDC1E\uD803\uDC1F\uD803\uDC20\uD803\uDC21\uD803\uDC22\uD803\uDC23\uD803\uDC24\uD803\uDC25\uD803\uDC26\uD803\uDC27\uD803\uDC28\uD803\uDC29\uD803\uDC2A\uD803\uDC2B\uD803\uDC2C\uD803\uDC2D\uD803\uDC2E\uD803\uDC2F\uD803\uDC30\uD803\uDC31\uD803\uDC32\uD803\uDC33\uD803\uDC34\uD803\uDC35\uD803\uDC36\uD803\uDC37\uD803\uDC38\uD803\uDC39\uD803\uDC3A\uD803\uDC3B\uD803\uDC3C\uD803\uDC3D\uD803\uDC3E\uD803\uDC3F\uD803\uDC40\uD803\uDC41\uD803\uDC42\uD803\uDC43\uD803\uDC44\uD803\uDC45\uD803\uDC46\uD803\uDC47\uD803\uDC48\uD804\uDC03\uD804\uDC04\uD804\uDC05\uD804\uDC06\uD804\uDC07\uD804\uDC08\uD804\uDC09\uD804\uDC0A\uD804\uDC0B\uD804\uDC0C\uD804\uDC0D\uD804\uDC0E\uD804\uDC0F\uD804\uDC10\uD804\uDC11\uD804\uDC12\uD804\uDC13\uD804\uDC14\uD804\uDC15\uD804\uDC16\uD804\uDC17\uD804\uDC18\uD804\uDC19\uD804\uDC1A\uD804\uDC1B\uD804\uDC1C\uD804\uDC1D\uD804\uDC1E\uD804\uDC1F\uD804\uDC20\uD804\uDC21\uD804\uDC22\uD804\uDC23\uD804\uDC24\uD804\uDC25\uD804\uDC26\uD804\uDC27\uD804\uDC28\uD804\uDC29\uD804\uDC2A\uD804\uDC2B\uD804\uDC2C\uD804\uDC2D\uD804\uDC2E\uD804\uDC2F\uD804\uDC30\uD804\uDC31\uD804\uDC32\uD804\uDC33\uD804\uDC34\uD804\uDC35\uD804\uDC36\uD804\uDC37\uD804\uDC83\uD804\uDC84\uD804\uDC85\uD804\uDC86\uD804\uDC87\uD804\uDC88\uD804\uDC89\uD804\uDC8A\uD804\uDC8B\uD804\uDC8C\uD804\uDC8D\uD804\uDC8E\uD804\uDC8F\uD804\uDC90\uD804\uDC91\uD804\uDC92\uD804\uDC93\uD804\uDC94\uD804\uDC95\uD804\uDC96\uD804\uDC97\uD804\uDC98\uD804\uDC99\uD804\uDC9A\uD804\uDC9B\uD804\uDC9C\uD804\uDC9D\uD804\uDC9E\uD804\uDC9F\uD804\uDCA0\uD804\uDCA1\uD804\uDCA2\uD804\uDCA3\uD804\uDCA4\uD804\uDCA5\uD804\uDCA6\uD804\uDCA7\uD804\uDCA8\uD804\uDCA9\uD804\uDCAA\uD804\uDCAB\uD804\uDCAC\uD804\uDCAD\uD804\uDCAE\uD804\uDCAF\uD808\uDC00\uD808\uDC01\uD808\uDC02\uD808\uDC03\uD808\uDC04\uD808\uDC05\uD808\uDC06\uD808\uDC07\uD808\uDC08\uD808\uDC09\uD808\uDC0A\uD808\uDC0B\uD808\uDC0C\uD808\uDC0D\uD808\uDC0E\uD808\uDC0F\uD808\uDC10\uD808\uDC11\uD808\uDC12\uD808\uDC13\uD808\uDC14\uD808\uDC15\uD808\uDC16\uD808\uDC17\uD808\uDC18\uD808\uDC19\uD808\uDC1A\uD808\uDC1B\uD808\uDC1C\uD808\uDC1D\uD808\uDC1E\uD808\uDC1F\uD808\uDC20\uD808\uDC21\uD808\uDC22\uD808\uDC23\uD808\uDC24\uD808\uDC25\uD808\uDC26\uD808\uDC27\uD808\uDC28\uD808\uDC29\uD808\uDC2A\uD808\uDC2B\uD808\uDC2C\uD808\uDC2D\uD808\uDC2E\uD808\uDC2F\uD808\uDC30\uD808\uDC31\uD808\uDC32\uD808\uDC33\uD808\uDC34\uD808\uDC35\uD808\uDC36\uD808\uDC37\uD808\uDC38\uD808\uDC39\uD808\uDC3A\uD808\uDC3B\uD808\uDC3C\uD808\uDC3D\uD808\uDC3E\uD808\uDC3F\uD808\uDC40\uD808\uDC41\uD808\uDC42\uD808\uDC43\uD808\uDC44\uD808\uDC45\uD808\uDC46\uD808\uDC47\uD808\uDC48\uD808\uDC49\uD808\uDC4A\uD808\uDC4B\uD808\uDC4C\uD808\uDC4D\uD808\uDC4E\uD808\uDC4F\uD808\uDC50\uD808\uDC51\uD808\uDC52\uD808\uDC53\uD808\uDC54\uD808\uDC55\uD808\uDC56\uD808\uDC57\uD808\uDC58\uD808\uDC59\uD808\uDC5A\uD808\uDC5B\uD808\uDC5C\uD808\uDC5D\uD808\uDC5E\uD808\uDC5F\uD808\uDC60\uD808\uDC61\uD808\uDC62\uD808\uDC63\uD808\uDC64\uD808\uDC65\uD808\uDC66\uD808\uDC67\uD808\uDC68\uD808\uDC69\uD808\uDC6A\uD808\uDC6B\uD808\uDC6C\uD808\uDC6D\uD808\uDC6E\uD808\uDC6F\uD808\uDC70\uD808\uDC71\uD808\uDC72\uD808\uDC73\uD808\uDC74\uD808\uDC75\uD808\uDC76\uD808\uDC77\uD808\uDC78\uD808\uDC79\uD808\uDC7A\uD808\uDC7B\uD808\uDC7C\uD808\uDC7D\uD808\uDC7E\uD808\uDC7F\uD808\uDC80\uD808\uDC81\uD808\uDC82\uD808\uDC83\uD808\uDC84\uD808\uDC85\uD808\uDC86\uD808\uDC87\uD808\uDC88\uD808\uDC89\uD808\uDC8A\uD808\uDC8B\uD808\uDC8C\uD808\uDC8D\uD808\uDC8E\uD808\uDC8F\uD808\uDC90\uD808\uDC91\uD808\uDC92\uD808\uDC93\uD808\uDC94\uD808\uDC95\uD808\uDC96\uD808\uDC97\uD808\uDC98\uD808\uDC99\uD808\uDC9A\uD808\uDC9B\uD808\uDC9C\uD808\uDC9D\uD808\uDC9E\uD808\uDC9F\uD808\uDCA0\uD808\uDCA1\uD808\uDCA2\uD808\uDCA3\uD808\uDCA4\uD808\uDCA5\uD808\uDCA6\uD808\uDCA7\uD808\uDCA8\uD808\uDCA9\uD808\uDCAA\uD808\uDCAB\uD808\uDCAC\uD808\uDCAD\uD808\uDCAE\uD808\uDCAF\uD808\uDCB0\uD808\uDCB1\uD808\uDCB2\uD808\uDCB3\uD808\uDCB4\uD808\uDCB5\uD808\uDCB6\uD808\uDCB7\uD808\uDCB8\uD808\uDCB9\uD808\uDCBA\uD808\uDCBB\uD808\uDCBC\uD808\uDCBD\uD808\uDCBE\uD808\uDCBF\uD808\uDCC0\uD808\uDCC1\uD808\uDCC2\uD808\uDCC3\uD808\uDCC4\uD808\uDCC5\uD808\uDCC6\uD808\uDCC7\uD808\uDCC8\uD808\uDCC9\uD808\uDCCA\uD808\uDCCB\uD808\uDCCC\uD808\uDCCD\uD808\uDCCE\uD808\uDCCF\uD808\uDCD0\uD808\uDCD1\uD808\uDCD2\uD808\uDCD3\uD808\uDCD4\uD808\uDCD5\uD808\uDCD6\uD808\uDCD7\uD808\uDCD8\uD808\uDCD9\uD808\uDCDA\uD808\uDCDB\uD808\uDCDC\uD808\uDCDD\uD808\uDCDE\uD808\uDCDF\uD808\uDCE0\uD808\uDCE1\uD808\uDCE2\uD808\uDCE3\uD808\uDCE4\uD808\uDCE5\uD808\uDCE6\uD808\uDCE7\uD808\uDCE8\uD808\uDCE9\uD808\uDCEA\uD808\uDCEB\uD808\uDCEC\uD808\uDCED\uD808\uDCEE\uD808\uDCEF\uD808\uDCF0\uD808\uDCF1\uD808\uDCF2\uD808\uDCF3\uD808\uDCF4\uD808\uDCF5\uD808\uDCF6\uD808\uDCF7\uD808\uDCF8\uD808\uDCF9\uD808\uDCFA\uD808\uDCFB\uD808\uDCFC\uD808\uDCFD\uD808\uDCFE\uD808\uDCFF\uD808\uDD00\uD808\uDD01\uD808\uDD02\uD808\uDD03\uD808\uDD04\uD808\uDD05\uD808\uDD06\uD808\uDD07\uD808\uDD08\uD808\uDD09\uD808\uDD0A\uD808\uDD0B\uD808\uDD0C\uD808\uDD0D\uD808\uDD0E\uD808\uDD0F\uD808\uDD10\uD808\uDD11\uD808\uDD12\uD808\uDD13\uD808\uDD14\uD808\uDD15\uD808\uDD16\uD808\uDD17\uD808\uDD18\uD808\uDD19\uD808\uDD1A\uD808\uDD1B\uD808\uDD1C\uD808\uDD1D\uD808\uDD1E\uD808\uDD1F\uD808\uDD20\uD808\uDD21\uD808\uDD22\uD808\uDD23\uD808\uDD24\uD808\uDD25\uD808\uDD26\uD808\uDD27\uD808\uDD28\uD808\uDD29\uD808\uDD2A\uD808\uDD2B\uD808\uDD2C\uD808\uDD2D\uD808\uDD2E\uD808\uDD2F\uD808\uDD30\uD808\uDD31\uD808\uDD32\uD808\uDD33\uD808\uDD34\uD808\uDD35\uD808\uDD36\uD808\uDD37\uD808\uDD38\uD808\uDD39\uD808\uDD3A\uD808\uDD3B\uD808\uDD3C\uD808\uDD3D\uD808\uDD3E\uD808\uDD3F\uD808\uDD40\uD808\uDD41\uD808\uDD42\uD808\uDD43\uD808\uDD44\uD808\uDD45\uD808\uDD46\uD808\uDD47\uD808\uDD48\uD808\uDD49\uD808\uDD4A\uD808\uDD4B\uD808\uDD4C\uD808\uDD4D\uD808\uDD4E\uD808\uDD4F\uD808\uDD50\uD808\uDD51\uD808\uDD52\uD808\uDD53\uD808\uDD54\uD808\uDD55\uD808\uDD56\uD808\uDD57\uD808\uDD58\uD808\uDD59\uD808\uDD5A\uD808\uDD5B\uD808\uDD5C\uD808\uDD5D\uD808\uDD5E\uD808\uDD5F\uD808\uDD60\uD808\uDD61\uD808\uDD62\uD808\uDD63\uD808\uDD64\uD808\uDD65\uD808\uDD66\uD808\uDD67\uD808\uDD68\uD808\uDD69\uD808\uDD6A\uD808\uDD6B\uD808\uDD6C\uD808\uDD6D\uD808\uDD6E\uD808\uDD6F\uD808\uDD70\uD808\uDD71\uD808\uDD72\uD808\uDD73\uD808\uDD74\uD808\uDD75\uD808\uDD76\uD808\uDD77\uD808\uDD78\uD808\uDD79\uD808\uDD7A\uD808\uDD7B\uD808\uDD7C\uD808\uDD7D\uD808\uDD7E\uD808\uDD7F\uD808\uDD80\uD808\uDD81\uD808\uDD82\uD808\uDD83\uD808\uDD84\uD808\uDD85\uD808\uDD86\uD808\uDD87\uD808\uDD88\uD808\uDD89\uD808\uDD8A\uD808\uDD8B\uD808\uDD8C\uD808\uDD8D\uD808\uDD8E\uD808\uDD8F\uD808\uDD90\uD808\uDD91\uD808\uDD92\uD808\uDD93\uD808\uDD94\uD808\uDD95\uD808\uDD96\uD808\uDD97\uD808\uDD98\uD808\uDD99\uD808\uDD9A\uD808\uDD9B\uD808\uDD9C\uD808\uDD9D\uD808\uDD9E\uD808\uDD9F\uD808\uDDA0\uD808\uDDA1\uD808\uDDA2\uD808\uDDA3\uD808\uDDA4\uD808\uDDA5\uD808\uDDA6\uD808\uDDA7\uD808\uDDA8\uD808\uDDA9\uD808\uDDAA\uD808\uDDAB\uD808\uDDAC\uD808\uDDAD\uD808\uDDAE\uD808\uDDAF\uD808\uDDB0\uD808\uDDB1\uD808\uDDB2\uD808\uDDB3\uD808\uDDB4\uD808\uDDB5\uD808\uDDB6\uD808\uDDB7\uD808\uDDB8\uD808\uDDB9\uD808\uDDBA\uD808\uDDBB\uD808\uDDBC\uD808\uDDBD\uD808\uDDBE\uD808\uDDBF\uD808\uDDC0\uD808\uDDC1\uD808\uDDC2\uD808\uDDC3\uD808\uDDC4\uD808\uDDC5\uD808\uDDC6\uD808\uDDC7\uD808\uDDC8\uD808\uDDC9\uD808\uDDCA\uD808\uDDCB\uD808\uDDCC\uD808\uDDCD\uD808\uDDCE\uD808\uDDCF\uD808\uDDD0\uD808\uDDD1\uD808\uDDD2\uD808\uDDD3\uD808\uDDD4\uD808\uDDD5\uD808\uDDD6\uD808\uDDD7\uD808\uDDD8\uD808\uDDD9\uD808\uDDDA\uD808\uDDDB\uD808\uDDDC\uD808\uDDDD\uD808\uDDDE\uD808\uDDDF\uD808\uDDE0\uD808\uDDE1\uD808\uDDE2\uD808\uDDE3\uD808\uDDE4\uD808\uDDE5\uD808\uDDE6\uD808\uDDE7\uD808\uDDE8\uD808\uDDE9\uD808\uDDEA\uD808\uDDEB\uD808\uDDEC\uD808\uDDED\uD808\uDDEE\uD808\uDDEF\uD808\uDDF0\uD808\uDDF1\uD808\uDDF2\uD808\uDDF3\uD808\uDDF4\uD808\uDDF5\uD808\uDDF6\uD808\uDDF7\uD808\uDDF8\uD808\uDDF9\uD808\uDDFA\uD808\uDDFB\uD808\uDDFC\uD808\uDDFD\uD808\uDDFE\uD808\uDDFF\uD808\uDE00\uD808\uDE01\uD808\uDE02\uD808\uDE03\uD808\uDE04\uD808\uDE05\uD808\uDE06\uD808\uDE07\uD808\uDE08\uD808\uDE09\uD808\uDE0A\uD808\uDE0B\uD808\uDE0C\uD808\uDE0D\uD808\uDE0E\uD808\uDE0F\uD808\uDE10\uD808\uDE11\uD808\uDE12\uD808\uDE13\uD808\uDE14\uD808\uDE15\uD808\uDE16\uD808\uDE17\uD808\uDE18\uD808\uDE19\uD808\uDE1A\uD808\uDE1B\uD808\uDE1C\uD808\uDE1D\uD808\uDE1E\uD808\uDE1F\uD808\uDE20\uD808\uDE21\uD808\uDE22\uD808\uDE23\uD808\uDE24\uD808\uDE25\uD808\uDE26\uD808\uDE27\uD808\uDE28\uD808\uDE29\uD808\uDE2A\uD808\uDE2B\uD808\uDE2C\uD808\uDE2D\uD808\uDE2E\uD808\uDE2F\uD808\uDE30\uD808\uDE31\uD808\uDE32\uD808\uDE33\uD808\uDE34\uD808\uDE35\uD808\uDE36\uD808\uDE37\uD808\uDE38\uD808\uDE39\uD808\uDE3A\uD808\uDE3B\uD808\uDE3C\uD808\uDE3D\uD808\uDE3E\uD808\uDE3F\uD808\uDE40\uD808\uDE41\uD808\uDE42\uD808\uDE43\uD808\uDE44\uD808\uDE45\uD808\uDE46\uD808\uDE47\uD808\uDE48\uD808\uDE49\uD808\uDE4A\uD808\uDE4B\uD808\uDE4C\uD808\uDE4D\uD808\uDE4E\uD808\uDE4F\uD808\uDE50\uD808\uDE51\uD808\uDE52\uD808\uDE53\uD808\uDE54\uD808\uDE55\uD808\uDE56\uD808\uDE57\uD808\uDE58\uD808\uDE59\uD808\uDE5A\uD808\uDE5B\uD808\uDE5C\uD808\uDE5D\uD808\uDE5E\uD808\uDE5F\uD808\uDE60\uD808\uDE61\uD808\uDE62\uD808\uDE63\uD808\uDE64\uD808\uDE65\uD808\uDE66\uD808\uDE67\uD808\uDE68\uD808\uDE69\uD808\uDE6A\uD808\uDE6B\uD808\uDE6C\uD808\uDE6D\uD808\uDE6E\uD808\uDE6F\uD808\uDE70\uD808\uDE71\uD808\uDE72\uD808\uDE73\uD808\uDE74\uD808\uDE75\uD808\uDE76\uD808\uDE77\uD808\uDE78\uD808\uDE79\uD808\uDE7A\uD808\uDE7B\uD808\uDE7C\uD808\uDE7D\uD808\uDE7E\uD808\uDE7F\uD808\uDE80\uD808\uDE81\uD808\uDE82\uD808\uDE83\uD808\uDE84\uD808\uDE85\uD808\uDE86\uD808\uDE87\uD808\uDE88\uD808\uDE89\uD808\uDE8A\uD808\uDE8B\uD808\uDE8C\uD808\uDE8D\uD808\uDE8E\uD808\uDE8F\uD808\uDE90\uD808\uDE91\uD808\uDE92\uD808\uDE93\uD808\uDE94\uD808\uDE95\uD808\uDE96\uD808\uDE97\uD808\uDE98\uD808\uDE99\uD808\uDE9A\uD808\uDE9B\uD808\uDE9C\uD808\uDE9D\uD808\uDE9E\uD808\uDE9F\uD808\uDEA0\uD808\uDEA1\uD808\uDEA2\uD808\uDEA3\uD808\uDEA4\uD808\uDEA5\uD808\uDEA6\uD808\uDEA7\uD808\uDEA8\uD808\uDEA9\uD808\uDEAA\uD808\uDEAB\uD808\uDEAC\uD808\uDEAD\uD808\uDEAE\uD808\uDEAF\uD808\uDEB0\uD808\uDEB1\uD808\uDEB2\uD808\uDEB3\uD808\uDEB4\uD808\uDEB5\uD808\uDEB6\uD808\uDEB7\uD808\uDEB8\uD808\uDEB9\uD808\uDEBA\uD808\uDEBB\uD808\uDEBC\uD808\uDEBD\uD808\uDEBE\uD808\uDEBF\uD808\uDEC0\uD808\uDEC1\uD808\uDEC2\uD808\uDEC3\uD808\uDEC4\uD808\uDEC5\uD808\uDEC6\uD808\uDEC7\uD808\uDEC8\uD808\uDEC9\uD808\uDECA\uD808\uDECB\uD808\uDECC\uD808\uDECD\uD808\uDECE\uD808\uDECF\uD808\uDED0\uD808\uDED1\uD808\uDED2\uD808\uDED3\uD808\uDED4\uD808\uDED5\uD808\uDED6\uD808\uDED7\uD808\uDED8\uD808\uDED9\uD808\uDEDA\uD808\uDEDB\uD808\uDEDC\uD808\uDEDD\uD808\uDEDE\uD808\uDEDF\uD808\uDEE0\uD808\uDEE1\uD808\uDEE2\uD808\uDEE3\uD808\uDEE4\uD808\uDEE5\uD808\uDEE6\uD808\uDEE7\uD808\uDEE8\uD808\uDEE9\uD808\uDEEA\uD808\uDEEB\uD808\uDEEC\uD808\uDEED\uD808\uDEEE\uD808\uDEEF\uD808\uDEF0\uD808\uDEF1\uD808\uDEF2\uD808\uDEF3\uD808\uDEF4\uD808\uDEF5\uD808\uDEF6\uD808\uDEF7\uD808\uDEF8\uD808\uDEF9\uD808\uDEFA\uD808\uDEFB\uD808\uDEFC\uD808\uDEFD\uD808\uDEFE\uD808\uDEFF\uD808\uDF00\uD808\uDF01\uD808\uDF02\uD808\uDF03\uD808\uDF04\uD808\uDF05\uD808\uDF06\uD808\uDF07\uD808\uDF08\uD808\uDF09\uD808\uDF0A\uD808\uDF0B\uD808\uDF0C\uD808\uDF0D\uD808\uDF0E\uD808\uDF0F\uD808\uDF10\uD808\uDF11\uD808\uDF12\uD808\uDF13\uD808\uDF14\uD808\uDF15\uD808\uDF16\uD808\uDF17\uD808\uDF18\uD808\uDF19\uD808\uDF1A\uD808\uDF1B\uD808\uDF1C\uD808\uDF1D\uD808\uDF1E\uD808\uDF1F\uD808\uDF20\uD808\uDF21\uD808\uDF22\uD808\uDF23\uD808\uDF24\uD808\uDF25\uD808\uDF26\uD808\uDF27\uD808\uDF28\uD808\uDF29\uD808\uDF2A\uD808\uDF2B\uD808\uDF2C\uD808\uDF2D\uD808\uDF2E\uD808\uDF2F\uD808\uDF30\uD808\uDF31\uD808\uDF32\uD808\uDF33\uD808\uDF34\uD808\uDF35\uD808\uDF36\uD808\uDF37\uD808\uDF38\uD808\uDF39\uD808\uDF3A\uD808\uDF3B\uD808\uDF3C\uD808\uDF3D\uD808\uDF3E\uD808\uDF3F\uD808\uDF40\uD808\uDF41\uD808\uDF42\uD808\uDF43\uD808\uDF44\uD808\uDF45\uD808\uDF46\uD808\uDF47\uD808\uDF48\uD808\uDF49\uD808\uDF4A\uD808\uDF4B\uD808\uDF4C\uD808\uDF4D\uD808\uDF4E\uD808\uDF4F\uD808\uDF50\uD808\uDF51\uD808\uDF52\uD808\uDF53\uD808\uDF54\uD808\uDF55\uD808\uDF56\uD808\uDF57\uD808\uDF58\uD808\uDF59\uD808\uDF5A\uD808\uDF5B\uD808\uDF5C\uD808\uDF5D\uD808\uDF5E\uD808\uDF5F\uD808\uDF60\uD808\uDF61\uD808\uDF62\uD808\uDF63\uD808\uDF64\uD808\uDF65\uD808\uDF66\uD808\uDF67\uD808\uDF68\uD808\uDF69\uD808\uDF6A\uD808\uDF6B\uD808\uDF6C\uD808\uDF6D\uD808\uDF6E\uD80C\uDC00\uD80C\uDC01\uD80C\uDC02\uD80C\uDC03\uD80C\uDC04\uD80C\uDC05\uD80C\uDC06\uD80C\uDC07\uD80C\uDC08\uD80C\uDC09\uD80C\uDC0A\uD80C\uDC0B\uD80C\uDC0C\uD80C\uDC0D\uD80C\uDC0E\uD80C\uDC0F\uD80C\uDC10\uD80C\uDC11\uD80C\uDC12\uD80C\uDC13\uD80C\uDC14\uD80C\uDC15\uD80C\uDC16\uD80C\uDC17\uD80C\uDC18\uD80C\uDC19\uD80C\uDC1A\uD80C\uDC1B\uD80C\uDC1C\uD80C\uDC1D\uD80C\uDC1E\uD80C\uDC1F\uD80C\uDC20\uD80C\uDC21\uD80C\uDC22\uD80C\uDC23\uD80C\uDC24\uD80C\uDC25\uD80C\uDC26\uD80C\uDC27\uD80C\uDC28\uD80C\uDC29\uD80C\uDC2A\uD80C\uDC2B\uD80C\uDC2C\uD80C\uDC2D\uD80C\uDC2E\uD80C\uDC2F\uD80C\uDC30\uD80C\uDC31\uD80C\uDC32\uD80C\uDC33\uD80C\uDC34\uD80C\uDC35\uD80C\uDC36\uD80C\uDC37\uD80C\uDC38\uD80C\uDC39\uD80C\uDC3A\uD80C\uDC3B\uD80C\uDC3C\uD80C\uDC3D\uD80C\uDC3E\uD80C\uDC3F\uD80C\uDC40\uD80C\uDC41\uD80C\uDC42\uD80C\uDC43\uD80C\uDC44\uD80C\uDC45\uD80C\uDC46\uD80C\uDC47\uD80C\uDC48\uD80C\uDC49\uD80C\uDC4A\uD80C\uDC4B\uD80C\uDC4C\uD80C\uDC4D\uD80C\uDC4E\uD80C\uDC4F\uD80C\uDC50\uD80C\uDC51\uD80C\uDC52\uD80C\uDC53\uD80C\uDC54\uD80C\uDC55\uD80C\uDC56\uD80C\uDC57\uD80C\uDC58\uD80C\uDC59\uD80C\uDC5A\uD80C\uDC5B\uD80C\uDC5C\uD80C\uDC5D\uD80C\uDC5E\uD80C\uDC5F\uD80C\uDC60\uD80C\uDC61\uD80C\uDC62\uD80C\uDC63\uD80C\uDC64\uD80C\uDC65\uD80C\uDC66\uD80C\uDC67\uD80C\uDC68\uD80C\uDC69\uD80C\uDC6A\uD80C\uDC6B\uD80C\uDC6C\uD80C\uDC6D\uD80C\uDC6E\uD80C\uDC6F\uD80C\uDC70\uD80C\uDC71\uD80C\uDC72\uD80C\uDC73\uD80C\uDC74\uD80C\uDC75\uD80C\uDC76\uD80C\uDC77\uD80C\uDC78\uD80C\uDC79\uD80C\uDC7A\uD80C\uDC7B\uD80C\uDC7C\uD80C\uDC7D\uD80C\uDC7E\uD80C\uDC7F\uD80C\uDC80\uD80C\uDC81\uD80C\uDC82\uD80C\uDC83\uD80C\uDC84\uD80C\uDC85\uD80C\uDC86\uD80C\uDC87\uD80C\uDC88\uD80C\uDC89\uD80C\uDC8A\uD80C\uDC8B\uD80C\uDC8C\uD80C\uDC8D\uD80C\uDC8E\uD80C\uDC8F\uD80C\uDC90\uD80C\uDC91\uD80C\uDC92\uD80C\uDC93\uD80C\uDC94\uD80C\uDC95\uD80C\uDC96\uD80C\uDC97\uD80C\uDC98\uD80C\uDC99\uD80C\uDC9A\uD80C\uDC9B\uD80C\uDC9C\uD80C\uDC9D\uD80C\uDC9E\uD80C\uDC9F\uD80C\uDCA0\uD80C\uDCA1\uD80C\uDCA2\uD80C\uDCA3\uD80C\uDCA4\uD80C\uDCA5\uD80C\uDCA6\uD80C\uDCA7\uD80C\uDCA8\uD80C\uDCA9\uD80C\uDCAA\uD80C\uDCAB\uD80C\uDCAC\uD80C\uDCAD\uD80C\uDCAE\uD80C\uDCAF\uD80C\uDCB0\uD80C\uDCB1\uD80C\uDCB2\uD80C\uDCB3\uD80C\uDCB4\uD80C\uDCB5\uD80C\uDCB6\uD80C\uDCB7\uD80C\uDCB8\uD80C\uDCB9\uD80C\uDCBA\uD80C\uDCBB\uD80C\uDCBC\uD80C\uDCBD\uD80C\uDCBE\uD80C\uDCBF\uD80C\uDCC0\uD80C\uDCC1\uD80C\uDCC2\uD80C\uDCC3\uD80C\uDCC4\uD80C\uDCC5\uD80C\uDCC6\uD80C\uDCC7\uD80C\uDCC8\uD80C\uDCC9\uD80C\uDCCA\uD80C\uDCCB\uD80C\uDCCC\uD80C\uDCCD\uD80C\uDCCE\uD80C\uDCCF\uD80C\uDCD0\uD80C\uDCD1\uD80C\uDCD2\uD80C\uDCD3\uD80C\uDCD4\uD80C\uDCD5\uD80C\uDCD6\uD80C\uDCD7\uD80C\uDCD8\uD80C\uDCD9\uD80C\uDCDA\uD80C\uDCDB\uD80C\uDCDC\uD80C\uDCDD\uD80C\uDCDE\uD80C\uDCDF\uD80C\uDCE0\uD80C\uDCE1\uD80C\uDCE2\uD80C\uDCE3\uD80C\uDCE4\uD80C\uDCE5\uD80C\uDCE6\uD80C\uDCE7\uD80C\uDCE8\uD80C\uDCE9\uD80C\uDCEA\uD80C\uDCEB\uD80C\uDCEC\uD80C\uDCED\uD80C\uDCEE\uD80C\uDCEF\uD80C\uDCF0\uD80C\uDCF1\uD80C\uDCF2\uD80C\uDCF3\uD80C\uDCF4\uD80C\uDCF5\uD80C\uDCF6\uD80C\uDCF7\uD80C\uDCF8\uD80C\uDCF9\uD80C\uDCFA\uD80C\uDCFB\uD80C\uDCFC\uD80C\uDCFD\uD80C\uDCFE\uD80C\uDCFF\uD80C\uDD00\uD80C\uDD01\uD80C\uDD02\uD80C\uDD03\uD80C\uDD04\uD80C\uDD05\uD80C\uDD06\uD80C\uDD07\uD80C\uDD08\uD80C\uDD09\uD80C\uDD0A\uD80C\uDD0B\uD80C\uDD0C\uD80C\uDD0D\uD80C\uDD0E\uD80C\uDD0F\uD80C\uDD10\uD80C\uDD11\uD80C\uDD12\uD80C\uDD13\uD80C\uDD14\uD80C\uDD15\uD80C\uDD16\uD80C\uDD17\uD80C\uDD18\uD80C\uDD19\uD80C\uDD1A\uD80C\uDD1B\uD80C\uDD1C\uD80C\uDD1D\uD80C\uDD1E\uD80C\uDD1F\uD80C\uDD20\uD80C\uDD21\uD80C\uDD22\uD80C\uDD23\uD80C\uDD24\uD80C\uDD25\uD80C\uDD26\uD80C\uDD27\uD80C\uDD28\uD80C\uDD29\uD80C\uDD2A\uD80C\uDD2B\uD80C\uDD2C\uD80C\uDD2D\uD80C\uDD2E\uD80C\uDD2F\uD80C\uDD30\uD80C\uDD31\uD80C\uDD32\uD80C\uDD33\uD80C\uDD34\uD80C\uDD35\uD80C\uDD36\uD80C\uDD37\uD80C\uDD38\uD80C\uDD39\uD80C\uDD3A\uD80C\uDD3B\uD80C\uDD3C\uD80C\uDD3D\uD80C\uDD3E\uD80C\uDD3F\uD80C\uDD40\uD80C\uDD41\uD80C\uDD42\uD80C\uDD43\uD80C\uDD44\uD80C\uDD45\uD80C\uDD46\uD80C\uDD47\uD80C\uDD48\uD80C\uDD49\uD80C\uDD4A\uD80C\uDD4B\uD80C\uDD4C\uD80C\uDD4D\uD80C\uDD4E\uD80C\uDD4F\uD80C\uDD50\uD80C\uDD51\uD80C\uDD52\uD80C\uDD53\uD80C\uDD54\uD80C\uDD55\uD80C\uDD56\uD80C\uDD57\uD80C\uDD58\uD80C\uDD59\uD80C\uDD5A\uD80C\uDD5B\uD80C\uDD5C\uD80C\uDD5D\uD80C\uDD5E\uD80C\uDD5F\uD80C\uDD60\uD80C\uDD61\uD80C\uDD62\uD80C\uDD63\uD80C\uDD64\uD80C\uDD65\uD80C\uDD66\uD80C\uDD67\uD80C\uDD68\uD80C\uDD69\uD80C\uDD6A\uD80C\uDD6B\uD80C\uDD6C\uD80C\uDD6D\uD80C\uDD6E\uD80C\uDD6F\uD80C\uDD70\uD80C\uDD71\uD80C\uDD72\uD80C\uDD73\uD80C\uDD74\uD80C\uDD75\uD80C\uDD76\uD80C\uDD77\uD80C\uDD78\uD80C\uDD79\uD80C\uDD7A\uD80C\uDD7B\uD80C\uDD7C\uD80C\uDD7D\uD80C\uDD7E\uD80C\uDD7F\uD80C\uDD80\uD80C\uDD81\uD80C\uDD82\uD80C\uDD83\uD80C\uDD84\uD80C\uDD85\uD80C\uDD86\uD80C\uDD87\uD80C\uDD88\uD80C\uDD89\uD80C\uDD8A\uD80C\uDD8B\uD80C\uDD8C\uD80C\uDD8D\uD80C\uDD8E\uD80C\uDD8F\uD80C\uDD90\uD80C\uDD91\uD80C\uDD92\uD80C\uDD93\uD80C\uDD94\uD80C\uDD95\uD80C\uDD96\uD80C\uDD97\uD80C\uDD98\uD80C\uDD99\uD80C\uDD9A\uD80C\uDD9B\uD80C\uDD9C\uD80C\uDD9D\uD80C\uDD9E\uD80C\uDD9F\uD80C\uDDA0\uD80C\uDDA1\uD80C\uDDA2\uD80C\uDDA3\uD80C\uDDA4\uD80C\uDDA5\uD80C\uDDA6\uD80C\uDDA7\uD80C\uDDA8\uD80C\uDDA9\uD80C\uDDAA\uD80C\uDDAB\uD80C\uDDAC\uD80C\uDDAD\uD80C\uDDAE\uD80C\uDDAF\uD80C\uDDB0\uD80C\uDDB1\uD80C\uDDB2\uD80C\uDDB3\uD80C\uDDB4\uD80C\uDDB5\uD80C\uDDB6\uD80C\uDDB7\uD80C\uDDB8\uD80C\uDDB9\uD80C\uDDBA\uD80C\uDDBB\uD80C\uDDBC\uD80C\uDDBD\uD80C\uDDBE\uD80C\uDDBF\uD80C\uDDC0\uD80C\uDDC1\uD80C\uDDC2\uD80C\uDDC3\uD80C\uDDC4\uD80C\uDDC5\uD80C\uDDC6\uD80C\uDDC7\uD80C\uDDC8\uD80C\uDDC9\uD80C\uDDCA\uD80C\uDDCB\uD80C\uDDCC\uD80C\uDDCD\uD80C\uDDCE\uD80C\uDDCF\uD80C\uDDD0\uD80C\uDDD1\uD80C\uDDD2\uD80C\uDDD3\uD80C\uDDD4\uD80C\uDDD5\uD80C\uDDD6\uD80C\uDDD7\uD80C\uDDD8\uD80C\uDDD9\uD80C\uDDDA\uD80C\uDDDB\uD80C\uDDDC\uD80C\uDDDD\uD80C\uDDDE\uD80C\uDDDF\uD80C\uDDE0\uD80C\uDDE1\uD80C\uDDE2\uD80C\uDDE3\uD80C\uDDE4\uD80C\uDDE5\uD80C\uDDE6\uD80C\uDDE7\uD80C\uDDE8\uD80C\uDDE9\uD80C\uDDEA\uD80C\uDDEB\uD80C\uDDEC\uD80C\uDDED\uD80C\uDDEE\uD80C\uDDEF\uD80C\uDDF0\uD80C\uDDF1\uD80C\uDDF2\uD80C\uDDF3\uD80C\uDDF4\uD80C\uDDF5\uD80C\uDDF6\uD80C\uDDF7\uD80C\uDDF8\uD80C\uDDF9\uD80C\uDDFA\uD80C\uDDFB\uD80C\uDDFC\uD80C\uDDFD\uD80C\uDDFE\uD80C\uDDFF\uD80C\uDE00\uD80C\uDE01\uD80C\uDE02\uD80C\uDE03\uD80C\uDE04\uD80C\uDE05\uD80C\uDE06\uD80C\uDE07\uD80C\uDE08\uD80C\uDE09\uD80C\uDE0A\uD80C\uDE0B\uD80C\uDE0C\uD80C\uDE0D\uD80C\uDE0E\uD80C\uDE0F\uD80C\uDE10\uD80C\uDE11\uD80C\uDE12\uD80C\uDE13\uD80C\uDE14\uD80C\uDE15\uD80C\uDE16\uD80C\uDE17\uD80C\uDE18\uD80C\uDE19\uD80C\uDE1A\uD80C\uDE1B\uD80C\uDE1C\uD80C\uDE1D\uD80C\uDE1E\uD80C\uDE1F\uD80C\uDE20\uD80C\uDE21\uD80C\uDE22\uD80C\uDE23\uD80C\uDE24\uD80C\uDE25\uD80C\uDE26\uD80C\uDE27\uD80C\uDE28\uD80C\uDE29\uD80C\uDE2A\uD80C\uDE2B\uD80C\uDE2C\uD80C\uDE2D\uD80C\uDE2E\uD80C\uDE2F\uD80C\uDE30\uD80C\uDE31\uD80C\uDE32\uD80C\uDE33\uD80C\uDE34\uD80C\uDE35\uD80C\uDE36\uD80C\uDE37\uD80C\uDE38\uD80C\uDE39\uD80C\uDE3A\uD80C\uDE3B\uD80C\uDE3C\uD80C\uDE3D\uD80C\uDE3E\uD80C\uDE3F\uD80C\uDE40\uD80C\uDE41\uD80C\uDE42\uD80C\uDE43\uD80C\uDE44\uD80C\uDE45\uD80C\uDE46\uD80C\uDE47\uD80C\uDE48\uD80C\uDE49\uD80C\uDE4A\uD80C\uDE4B\uD80C\uDE4C\uD80C\uDE4D\uD80C\uDE4E\uD80C\uDE4F\uD80C\uDE50\uD80C\uDE51\uD80C\uDE52\uD80C\uDE53\uD80C\uDE54\uD80C\uDE55\uD80C\uDE56\uD80C\uDE57\uD80C\uDE58\uD80C\uDE59\uD80C\uDE5A\uD80C\uDE5B\uD80C\uDE5C\uD80C\uDE5D\uD80C\uDE5E\uD80C\uDE5F\uD80C\uDE60\uD80C\uDE61\uD80C\uDE62\uD80C\uDE63\uD80C\uDE64\uD80C\uDE65\uD80C\uDE66\uD80C\uDE67\uD80C\uDE68\uD80C\uDE69\uD80C\uDE6A\uD80C\uDE6B\uD80C\uDE6C\uD80C\uDE6D\uD80C\uDE6E\uD80C\uDE6F\uD80C\uDE70\uD80C\uDE71\uD80C\uDE72\uD80C\uDE73\uD80C\uDE74\uD80C\uDE75\uD80C\uDE76\uD80C\uDE77\uD80C\uDE78\uD80C\uDE79\uD80C\uDE7A\uD80C\uDE7B\uD80C\uDE7C\uD80C\uDE7D\uD80C\uDE7E\uD80C\uDE7F\uD80C\uDE80\uD80C\uDE81\uD80C\uDE82\uD80C\uDE83\uD80C\uDE84\uD80C\uDE85\uD80C\uDE86\uD80C\uDE87\uD80C\uDE88\uD80C\uDE89\uD80C\uDE8A\uD80C\uDE8B\uD80C\uDE8C\uD80C\uDE8D\uD80C\uDE8E\uD80C\uDE8F\uD80C\uDE90\uD80C\uDE91\uD80C\uDE92\uD80C\uDE93\uD80C\uDE94\uD80C\uDE95\uD80C\uDE96\uD80C\uDE97\uD80C\uDE98\uD80C\uDE99\uD80C\uDE9A\uD80C\uDE9B\uD80C\uDE9C\uD80C\uDE9D\uD80C\uDE9E\uD80C\uDE9F\uD80C\uDEA0\uD80C\uDEA1\uD80C\uDEA2\uD80C\uDEA3\uD80C\uDEA4\uD80C\uDEA5\uD80C\uDEA6\uD80C\uDEA7\uD80C\uDEA8\uD80C\uDEA9\uD80C\uDEAA\uD80C\uDEAB\uD80C\uDEAC\uD80C\uDEAD\uD80C\uDEAE\uD80C\uDEAF\uD80C\uDEB0\uD80C\uDEB1\uD80C\uDEB2\uD80C\uDEB3\uD80C\uDEB4\uD80C\uDEB5\uD80C\uDEB6\uD80C\uDEB7\uD80C\uDEB8\uD80C\uDEB9\uD80C\uDEBA\uD80C\uDEBB\uD80C\uDEBC\uD80C\uDEBD\uD80C\uDEBE\uD80C\uDEBF\uD80C\uDEC0\uD80C\uDEC1\uD80C\uDEC2\uD80C\uDEC3\uD80C\uDEC4\uD80C\uDEC5\uD80C\uDEC6\uD80C\uDEC7\uD80C\uDEC8\uD80C\uDEC9\uD80C\uDECA\uD80C\uDECB\uD80C\uDECC\uD80C\uDECD\uD80C\uDECE\uD80C\uDECF\uD80C\uDED0\uD80C\uDED1\uD80C\uDED2\uD80C\uDED3\uD80C\uDED4\uD80C\uDED5\uD80C\uDED6\uD80C\uDED7\uD80C\uDED8\uD80C\uDED9\uD80C\uDEDA\uD80C\uDEDB\uD80C\uDEDC\uD80C\uDEDD\uD80C\uDEDE\uD80C\uDEDF\uD80C\uDEE0\uD80C\uDEE1\uD80C\uDEE2\uD80C\uDEE3\uD80C\uDEE4\uD80C\uDEE5\uD80C\uDEE6\uD80C\uDEE7\uD80C\uDEE8\uD80C\uDEE9\uD80C\uDEEA\uD80C\uDEEB\uD80C\uDEEC\uD80C\uDEED\uD80C\uDEEE\uD80C\uDEEF\uD80C\uDEF0\uD80C\uDEF1\uD80C\uDEF2\uD80C\uDEF3\uD80C\uDEF4\uD80C\uDEF5\uD80C\uDEF6\uD80C\uDEF7\uD80C\uDEF8\uD80C\uDEF9\uD80C\uDEFA\uD80C\uDEFB\uD80C\uDEFC\uD80C\uDEFD\uD80C\uDEFE\uD80C\uDEFF\uD80C\uDF00\uD80C\uDF01\uD80C\uDF02\uD80C\uDF03\uD80C\uDF04\uD80C\uDF05\uD80C\uDF06\uD80C\uDF07\uD80C\uDF08\uD80C\uDF09\uD80C\uDF0A\uD80C\uDF0B\uD80C\uDF0C\uD80C\uDF0D\uD80C\uDF0E\uD80C\uDF0F\uD80C\uDF10\uD80C\uDF11\uD80C\uDF12\uD80C\uDF13\uD80C\uDF14\uD80C\uDF15\uD80C\uDF16\uD80C\uDF17\uD80C\uDF18\uD80C\uDF19\uD80C\uDF1A\uD80C\uDF1B\uD80C\uDF1C\uD80C\uDF1D\uD80C\uDF1E\uD80C\uDF1F\uD80C\uDF20\uD80C\uDF21\uD80C\uDF22\uD80C\uDF23\uD80C\uDF24\uD80C\uDF25\uD80C\uDF26\uD80C\uDF27\uD80C\uDF28\uD80C\uDF29\uD80C\uDF2A\uD80C\uDF2B\uD80C\uDF2C\uD80C\uDF2D\uD80C\uDF2E\uD80C\uDF2F\uD80C\uDF30\uD80C\uDF31\uD80C\uDF32\uD80C\uDF33\uD80C\uDF34\uD80C\uDF35\uD80C\uDF36\uD80C\uDF37\uD80C\uDF38\uD80C\uDF39\uD80C\uDF3A\uD80C\uDF3B\uD80C\uDF3C\uD80C\uDF3D\uD80C\uDF3E\uD80C\uDF3F\uD80C\uDF40\uD80C\uDF41\uD80C\uDF42\uD80C\uDF43\uD80C\uDF44\uD80C\uDF45\uD80C\uDF46\uD80C\uDF47\uD80C\uDF48\uD80C\uDF49\uD80C\uDF4A\uD80C\uDF4B\uD80C\uDF4C\uD80C\uDF4D\uD80C\uDF4E\uD80C\uDF4F\uD80C\uDF50\uD80C\uDF51\uD80C\uDF52\uD80C\uDF53\uD80C\uDF54\uD80C\uDF55\uD80C\uDF56\uD80C\uDF57\uD80C\uDF58\uD80C\uDF59\uD80C\uDF5A\uD80C\uDF5B\uD80C\uDF5C\uD80C\uDF5D\uD80C\uDF5E\uD80C\uDF5F\uD80C\uDF60\uD80C\uDF61\uD80C\uDF62\uD80C\uDF63\uD80C\uDF64\uD80C\uDF65\uD80C\uDF66\uD80C\uDF67\uD80C\uDF68\uD80C\uDF69\uD80C\uDF6A\uD80C\uDF6B\uD80C\uDF6C\uD80C\uDF6D\uD80C\uDF6E\uD80C\uDF6F\uD80C\uDF70\uD80C\uDF71\uD80C\uDF72\uD80C\uDF73\uD80C\uDF74\uD80C\uDF75\uD80C\uDF76\uD80C\uDF77\uD80C\uDF78\uD80C\uDF79\uD80C\uDF7A\uD80C\uDF7B\uD80C\uDF7C\uD80C\uDF7D\uD80C\uDF7E\uD80C\uDF7F\uD80C\uDF80\uD80C\uDF81\uD80C\uDF82\uD80C\uDF83\uD80C\uDF84\uD80C\uDF85\uD80C\uDF86\uD80C\uDF87\uD80C\uDF88\uD80C\uDF89\uD80C\uDF8A\uD80C\uDF8B\uD80C\uDF8C\uD80C\uDF8D\uD80C\uDF8E\uD80C\uDF8F\uD80C\uDF90\uD80C\uDF91\uD80C\uDF92\uD80C\uDF93\uD80C\uDF94\uD80C\uDF95\uD80C\uDF96\uD80C\uDF97\uD80C\uDF98\uD80C\uDF99\uD80C\uDF9A\uD80C\uDF9B\uD80C\uDF9C\uD80C\uDF9D\uD80C\uDF9E\uD80C\uDF9F\uD80C\uDFA0\uD80C\uDFA1\uD80C\uDFA2\uD80C\uDFA3\uD80C\uDFA4\uD80C\uDFA5\uD80C\uDFA6\uD80C\uDFA7\uD80C\uDFA8\uD80C\uDFA9\uD80C\uDFAA\uD80C\uDFAB\uD80C\uDFAC\uD80C\uDFAD\uD80C\uDFAE\uD80C\uDFAF\uD80C\uDFB0\uD80C\uDFB1\uD80C\uDFB2\uD80C\uDFB3\uD80C\uDFB4\uD80C\uDFB5\uD80C\uDFB6\uD80C\uDFB7\uD80C\uDFB8\uD80C\uDFB9\uD80C\uDFBA\uD80C\uDFBB\uD80C\uDFBC\uD80C\uDFBD\uD80C\uDFBE\uD80C\uDFBF\uD80C\uDFC0\uD80C\uDFC1\uD80C\uDFC2\uD80C\uDFC3\uD80C\uDFC4\uD80C\uDFC5\uD80C\uDFC6\uD80C\uDFC7\uD80C\uDFC8\uD80C\uDFC9\uD80C\uDFCA\uD80C\uDFCB\uD80C\uDFCC\uD80C\uDFCD\uD80C\uDFCE\uD80C\uDFCF\uD80C\uDFD0\uD80C\uDFD1\uD80C\uDFD2\uD80C\uDFD3\uD80C\uDFD4\uD80C\uDFD5\uD80C\uDFD6\uD80C\uDFD7\uD80C\uDFD8\uD80C\uDFD9\uD80C\uDFDA\uD80C\uDFDB\uD80C\uDFDC\uD80C\uDFDD\uD80C\uDFDE\uD80C\uDFDF\uD80C\uDFE0\uD80C\uDFE1\uD80C\uDFE2\uD80C\uDFE3\uD80C\uDFE4\uD80C\uDFE5\uD80C\uDFE6\uD80C\uDFE7\uD80C\uDFE8\uD80C\uDFE9\uD80C\uDFEA\uD80C\uDFEB\uD80C\uDFEC\uD80C\uDFED\uD80C\uDFEE\uD80C\uDFEF\uD80C\uDFF0\uD80C\uDFF1\uD80C\uDFF2\uD80C\uDFF3\uD80C\uDFF4\uD80C\uDFF5\uD80C\uDFF6\uD80C\uDFF7\uD80C\uDFF8\uD80C\uDFF9\uD80C\uDFFA\uD80C\uDFFB\uD80C\uDFFC\uD80C\uDFFD\uD80C\uDFFE\uD80C\uDFFF\uD80D\uDC00\uD80D\uDC01\uD80D\uDC02\uD80D\uDC03\uD80D\uDC04\uD80D\uDC05\uD80D\uDC06\uD80D\uDC07\uD80D\uDC08\uD80D\uDC09\uD80D\uDC0A\uD80D\uDC0B\uD80D\uDC0C\uD80D\uDC0D\uD80D\uDC0E\uD80D\uDC0F\uD80D\uDC10\uD80D\uDC11\uD80D\uDC12\uD80D\uDC13\uD80D\uDC14\uD80D\uDC15\uD80D\uDC16\uD80D\uDC17\uD80D\uDC18\uD80D\uDC19\uD80D\uDC1A\uD80D\uDC1B\uD80D\uDC1C\uD80D\uDC1D\uD80D\uDC1E\uD80D\uDC1F\uD80D\uDC20\uD80D\uDC21\uD80D\uDC22\uD80D\uDC23\uD80D\uDC24\uD80D\uDC25\uD80D\uDC26\uD80D\uDC27\uD80D\uDC28\uD80D\uDC29\uD80D\uDC2A\uD80D\uDC2B\uD80D\uDC2C\uD80D\uDC2D\uD80D\uDC2E\uD81A\uDC00\uD81A\uDC01\uD81A\uDC02\uD81A\uDC03\uD81A\uDC04\uD81A\uDC05\uD81A\uDC06\uD81A\uDC07\uD81A\uDC08\uD81A\uDC09\uD81A\uDC0A\uD81A\uDC0B\uD81A\uDC0C\uD81A\uDC0D\uD81A\uDC0E\uD81A\uDC0F\uD81A\uDC10\uD81A\uDC11\uD81A\uDC12\uD81A\uDC13\uD81A\uDC14\uD81A\uDC15\uD81A\uDC16\uD81A\uDC17\uD81A\uDC18\uD81A\uDC19\uD81A\uDC1A\uD81A\uDC1B\uD81A\uDC1C\uD81A\uDC1D\uD81A\uDC1E\uD81A\uDC1F\uD81A\uDC20\uD81A\uDC21\uD81A\uDC22\uD81A\uDC23\uD81A\uDC24\uD81A\uDC25\uD81A\uDC26\uD81A\uDC27\uD81A\uDC28\uD81A\uDC29\uD81A\uDC2A\uD81A\uDC2B\uD81A\uDC2C\uD81A\uDC2D\uD81A\uDC2E\uD81A\uDC2F\uD81A\uDC30\uD81A\uDC31\uD81A\uDC32\uD81A\uDC33\uD81A\uDC34\uD81A\uDC35\uD81A\uDC36\uD81A\uDC37\uD81A\uDC38\uD81A\uDC39\uD81A\uDC3A\uD81A\uDC3B\uD81A\uDC3C\uD81A\uDC3D\uD81A\uDC3E\uD81A\uDC3F\uD81A\uDC40\uD81A\uDC41\uD81A\uDC42\uD81A\uDC43\uD81A\uDC44\uD81A\uDC45\uD81A\uDC46\uD81A\uDC47\uD81A\uDC48\uD81A\uDC49\uD81A\uDC4A\uD81A\uDC4B\uD81A\uDC4C\uD81A\uDC4D\uD81A\uDC4E\uD81A\uDC4F\uD81A\uDC50\uD81A\uDC51\uD81A\uDC52\uD81A\uDC53\uD81A\uDC54\uD81A\uDC55\uD81A\uDC56\uD81A\uDC57\uD81A\uDC58\uD81A\uDC59\uD81A\uDC5A\uD81A\uDC5B\uD81A\uDC5C\uD81A\uDC5D\uD81A\uDC5E\uD81A\uDC5F\uD81A\uDC60\uD81A\uDC61\uD81A\uDC62\uD81A\uDC63\uD81A\uDC64\uD81A\uDC65\uD81A\uDC66\uD81A\uDC67\uD81A\uDC68\uD81A\uDC69\uD81A\uDC6A\uD81A\uDC6B\uD81A\uDC6C\uD81A\uDC6D\uD81A\uDC6E\uD81A\uDC6F\uD81A\uDC70\uD81A\uDC71\uD81A\uDC72\uD81A\uDC73\uD81A\uDC74\uD81A\uDC75\uD81A\uDC76\uD81A\uDC77\uD81A\uDC78\uD81A\uDC79\uD81A\uDC7A\uD81A\uDC7B\uD81A\uDC7C\uD81A\uDC7D\uD81A\uDC7E\uD81A\uDC7F\uD81A\uDC80\uD81A\uDC81\uD81A\uDC82\uD81A\uDC83\uD81A\uDC84\uD81A\uDC85\uD81A\uDC86\uD81A\uDC87\uD81A\uDC88\uD81A\uDC89\uD81A\uDC8A\uD81A\uDC8B\uD81A\uDC8C\uD81A\uDC8D\uD81A\uDC8E\uD81A\uDC8F\uD81A\uDC90\uD81A\uDC91\uD81A\uDC92\uD81A\uDC93\uD81A\uDC94\uD81A\uDC95\uD81A\uDC96\uD81A\uDC97\uD81A\uDC98\uD81A\uDC99\uD81A\uDC9A\uD81A\uDC9B\uD81A\uDC9C\uD81A\uDC9D\uD81A\uDC9E\uD81A\uDC9F\uD81A\uDCA0\uD81A\uDCA1\uD81A\uDCA2\uD81A\uDCA3\uD81A\uDCA4\uD81A\uDCA5\uD81A\uDCA6\uD81A\uDCA7\uD81A\uDCA8\uD81A\uDCA9\uD81A\uDCAA\uD81A\uDCAB\uD81A\uDCAC\uD81A\uDCAD\uD81A\uDCAE\uD81A\uDCAF\uD81A\uDCB0\uD81A\uDCB1\uD81A\uDCB2\uD81A\uDCB3\uD81A\uDCB4\uD81A\uDCB5\uD81A\uDCB6\uD81A\uDCB7\uD81A\uDCB8\uD81A\uDCB9\uD81A\uDCBA\uD81A\uDCBB\uD81A\uDCBC\uD81A\uDCBD\uD81A\uDCBE\uD81A\uDCBF\uD81A\uDCC0\uD81A\uDCC1\uD81A\uDCC2\uD81A\uDCC3\uD81A\uDCC4\uD81A\uDCC5\uD81A\uDCC6\uD81A\uDCC7\uD81A\uDCC8\uD81A\uDCC9\uD81A\uDCCA\uD81A\uDCCB\uD81A\uDCCC\uD81A\uDCCD\uD81A\uDCCE\uD81A\uDCCF\uD81A\uDCD0\uD81A\uDCD1\uD81A\uDCD2\uD81A\uDCD3\uD81A\uDCD4\uD81A\uDCD5\uD81A\uDCD6\uD81A\uDCD7\uD81A\uDCD8\uD81A\uDCD9\uD81A\uDCDA\uD81A\uDCDB\uD81A\uDCDC\uD81A\uDCDD\uD81A\uDCDE\uD81A\uDCDF\uD81A\uDCE0\uD81A\uDCE1\uD81A\uDCE2\uD81A\uDCE3\uD81A\uDCE4\uD81A\uDCE5\uD81A\uDCE6\uD81A\uDCE7\uD81A\uDCE8\uD81A\uDCE9\uD81A\uDCEA\uD81A\uDCEB\uD81A\uDCEC\uD81A\uDCED\uD81A\uDCEE\uD81A\uDCEF\uD81A\uDCF0\uD81A\uDCF1\uD81A\uDCF2\uD81A\uDCF3\uD81A\uDCF4\uD81A\uDCF5\uD81A\uDCF6\uD81A\uDCF7\uD81A\uDCF8\uD81A\uDCF9\uD81A\uDCFA\uD81A\uDCFB\uD81A\uDCFC\uD81A\uDCFD\uD81A\uDCFE\uD81A\uDCFF\uD81A\uDD00\uD81A\uDD01\uD81A\uDD02\uD81A\uDD03\uD81A\uDD04\uD81A\uDD05\uD81A\uDD06\uD81A\uDD07\uD81A\uDD08\uD81A\uDD09\uD81A\uDD0A\uD81A\uDD0B\uD81A\uDD0C\uD81A\uDD0D\uD81A\uDD0E\uD81A\uDD0F\uD81A\uDD10\uD81A\uDD11\uD81A\uDD12\uD81A\uDD13\uD81A\uDD14\uD81A\uDD15\uD81A\uDD16\uD81A\uDD17\uD81A\uDD18\uD81A\uDD19\uD81A\uDD1A\uD81A\uDD1B\uD81A\uDD1C\uD81A\uDD1D\uD81A\uDD1E\uD81A\uDD1F\uD81A\uDD20\uD81A\uDD21\uD81A\uDD22\uD81A\uDD23\uD81A\uDD24\uD81A\uDD25\uD81A\uDD26\uD81A\uDD27\uD81A\uDD28\uD81A\uDD29\uD81A\uDD2A\uD81A\uDD2B\uD81A\uDD2C\uD81A\uDD2D\uD81A\uDD2E\uD81A\uDD2F\uD81A\uDD30\uD81A\uDD31\uD81A\uDD32\uD81A\uDD33\uD81A\uDD34\uD81A\uDD35\uD81A\uDD36\uD81A\uDD37\uD81A\uDD38\uD81A\uDD39\uD81A\uDD3A\uD81A\uDD3B\uD81A\uDD3C\uD81A\uDD3D\uD81A\uDD3E\uD81A\uDD3F\uD81A\uDD40\uD81A\uDD41\uD81A\uDD42\uD81A\uDD43\uD81A\uDD44\uD81A\uDD45\uD81A\uDD46\uD81A\uDD47\uD81A\uDD48\uD81A\uDD49\uD81A\uDD4A\uD81A\uDD4B\uD81A\uDD4C\uD81A\uDD4D\uD81A\uDD4E\uD81A\uDD4F\uD81A\uDD50\uD81A\uDD51\uD81A\uDD52\uD81A\uDD53\uD81A\uDD54\uD81A\uDD55\uD81A\uDD56\uD81A\uDD57\uD81A\uDD58\uD81A\uDD59\uD81A\uDD5A\uD81A\uDD5B\uD81A\uDD5C\uD81A\uDD5D\uD81A\uDD5E\uD81A\uDD5F\uD81A\uDD60\uD81A\uDD61\uD81A\uDD62\uD81A\uDD63\uD81A\uDD64\uD81A\uDD65\uD81A\uDD66\uD81A\uDD67\uD81A\uDD68\uD81A\uDD69\uD81A\uDD6A\uD81A\uDD6B\uD81A\uDD6C\uD81A\uDD6D\uD81A\uDD6E\uD81A\uDD6F\uD81A\uDD70\uD81A\uDD71\uD81A\uDD72\uD81A\uDD73\uD81A\uDD74\uD81A\uDD75\uD81A\uDD76\uD81A\uDD77\uD81A\uDD78\uD81A\uDD79\uD81A\uDD7A\uD81A\uDD7B\uD81A\uDD7C\uD81A\uDD7D\uD81A\uDD7E\uD81A\uDD7F\uD81A\uDD80\uD81A\uDD81\uD81A\uDD82\uD81A\uDD83\uD81A\uDD84\uD81A\uDD85\uD81A\uDD86\uD81A\uDD87\uD81A\uDD88\uD81A\uDD89\uD81A\uDD8A\uD81A\uDD8B\uD81A\uDD8C\uD81A\uDD8D\uD81A\uDD8E\uD81A\uDD8F\uD81A\uDD90\uD81A\uDD91\uD81A\uDD92\uD81A\uDD93\uD81A\uDD94\uD81A\uDD95\uD81A\uDD96\uD81A\uDD97\uD81A\uDD98\uD81A\uDD99\uD81A\uDD9A\uD81A\uDD9B\uD81A\uDD9C\uD81A\uDD9D\uD81A\uDD9E\uD81A\uDD9F\uD81A\uDDA0\uD81A\uDDA1\uD81A\uDDA2\uD81A\uDDA3\uD81A\uDDA4\uD81A\uDDA5\uD81A\uDDA6\uD81A\uDDA7\uD81A\uDDA8\uD81A\uDDA9\uD81A\uDDAA\uD81A\uDDAB\uD81A\uDDAC\uD81A\uDDAD\uD81A\uDDAE\uD81A\uDDAF\uD81A\uDDB0\uD81A\uDDB1\uD81A\uDDB2\uD81A\uDDB3\uD81A\uDDB4\uD81A\uDDB5\uD81A\uDDB6\uD81A\uDDB7\uD81A\uDDB8\uD81A\uDDB9\uD81A\uDDBA\uD81A\uDDBB\uD81A\uDDBC\uD81A\uDDBD\uD81A\uDDBE\uD81A\uDDBF\uD81A\uDDC0\uD81A\uDDC1\uD81A\uDDC2\uD81A\uDDC3\uD81A\uDDC4\uD81A\uDDC5\uD81A\uDDC6\uD81A\uDDC7\uD81A\uDDC8\uD81A\uDDC9\uD81A\uDDCA\uD81A\uDDCB\uD81A\uDDCC\uD81A\uDDCD\uD81A\uDDCE\uD81A\uDDCF\uD81A\uDDD0\uD81A\uDDD1\uD81A\uDDD2\uD81A\uDDD3\uD81A\uDDD4\uD81A\uDDD5\uD81A\uDDD6\uD81A\uDDD7\uD81A\uDDD8\uD81A\uDDD9\uD81A\uDDDA\uD81A\uDDDB\uD81A\uDDDC\uD81A\uDDDD\uD81A\uDDDE\uD81A\uDDDF\uD81A\uDDE0\uD81A\uDDE1\uD81A\uDDE2\uD81A\uDDE3\uD81A\uDDE4\uD81A\uDDE5\uD81A\uDDE6\uD81A\uDDE7\uD81A\uDDE8\uD81A\uDDE9\uD81A\uDDEA\uD81A\uDDEB\uD81A\uDDEC\uD81A\uDDED\uD81A\uDDEE\uD81A\uDDEF\uD81A\uDDF0\uD81A\uDDF1\uD81A\uDDF2\uD81A\uDDF3\uD81A\uDDF4\uD81A\uDDF5\uD81A\uDDF6\uD81A\uDDF7\uD81A\uDDF8\uD81A\uDDF9\uD81A\uDDFA\uD81A\uDDFB\uD81A\uDDFC\uD81A\uDDFD\uD81A\uDDFE\uD81A\uDDFF\uD81A\uDE00\uD81A\uDE01\uD81A\uDE02\uD81A\uDE03\uD81A\uDE04\uD81A\uDE05\uD81A\uDE06\uD81A\uDE07\uD81A\uDE08\uD81A\uDE09\uD81A\uDE0A\uD81A\uDE0B\uD81A\uDE0C\uD81A\uDE0D\uD81A\uDE0E\uD81A\uDE0F\uD81A\uDE10\uD81A\uDE11\uD81A\uDE12\uD81A\uDE13\uD81A\uDE14\uD81A\uDE15\uD81A\uDE16\uD81A\uDE17\uD81A\uDE18\uD81A\uDE19\uD81A\uDE1A\uD81A\uDE1B\uD81A\uDE1C\uD81A\uDE1D\uD81A\uDE1E\uD81A\uDE1F\uD81A\uDE20\uD81A\uDE21\uD81A\uDE22\uD81A\uDE23\uD81A\uDE24\uD81A\uDE25\uD81A\uDE26\uD81A\uDE27\uD81A\uDE28\uD81A\uDE29\uD81A\uDE2A\uD81A\uDE2B\uD81A\uDE2C\uD81A\uDE2D\uD81A\uDE2E\uD81A\uDE2F\uD81A\uDE30\uD81A\uDE31\uD81A\uDE32\uD81A\uDE33\uD81A\uDE34\uD81A\uDE35\uD81A\uDE36\uD81A\uDE37\uD81A\uDE38\uD82C\uDC00\uD82C\uDC01\uD840\uDC00\uD869\uDED6\uD869\uDF00\uD86D\uDF34\uD86D\uDF40\uD86E\uDC1D\u4E3D\u4E38\u4E41\uD840\uDD22\u4F60\u4FAE\u4FBB\u5002\u507A\u5099\u50E7\u50CF\u349E\uD841\uDE3A\u514D\u5154\u5164\u5177\uD841\uDD1C\u34B9\u5167\u518D\uD841\uDD4B\u5197\u51A4\u4ECC\u51AC\u51B5\uD864\uDDDF\u51F5\u5203\u34DF\u523B\u5246\u5272\u5277\u3515\u52C7\u52C9\u52E4\u52FA\u5305\u5306\u5317\u5349\u5351\u535A\u5373\u537D\u537F\u537F\u537F\uD842\uDE2C\u7070\u53CA\u53DF\uD842\uDF63\u53EB\u53F1\u5406\u549E\u5438\u5448\u5468\u54A2\u54F6\u5510\u5553\u5563\u5584\u5584\u5599\u55AB\u55B3\u55C2\u5716\u5606\u5717\u5651\u5674\u5207\u58EE\u57CE\u57F4\u580D\u578B\u5832\u5831\u58AC\uD845\uDCE4\u58F2\u58F7\u5906\u591A\u5922\u5962\uD845\uDEA8\uD845\uDEEA\u59EC\u5A1B\u5A27\u59D8\u5A66\u36EE\u36FC\u5B08\u5B3E\u5B3E\uD846\uDDC8\u5BC3\u5BD8\u5BE7\u5BF3\uD846\uDF18\u5BFF\u5C06\u5F53\u5C22\u3781\u5C60\u5C6E\u5CC0\u5C8D\uD847\uDDE4\u5D43\uD847\uDDE6\u5D6E\u5D6B\u5D7C\u5DE1\u5DE2\u382F\u5DFD\u5E28\u5E3D\u5E69\u3862\uD848\uDD83\u387C\u5EB0\u5EB3\u5EB6\u5ECA\uD868\uDF92\u5EFE\uD848\uDF31\uD848\uDF31\u8201\u5F22\u5F22\u38C7\uD84C\uDEB8\uD858\uDDDA\u5F62\u5F6B\u38E3\u5F9A\u5FCD\u5FD7\u5FF9\u6081\u393A\u391C\u6094\uD849\uDED4\u60C7\u6148\u614C\u614E\u614C\u617A\u618E\u61B2\u61A4\u61AF\u61DE\u61F2\u61F6\u6210\u621B\u625D\u62B1\u62D4\u6350\uD84A\uDF0C\u633D\u62FC\u6368\u6383\u63E4\uD84A\uDFF1\u6422\u63C5\u63A9\u3A2E\u6469\u647E\u649D\u6477\u3A6C\u654F\u656C\uD84C\uDC0A\u65E3\u66F8\u6649\u3B19\u6691\u3B08\u3AE4\u5192\u5195\u6700\u669C\u80AD\u43D9\u6717\u671B\u6721\u675E\u6753\uD84C\uDFC3\u3B49\u67FA\u6785\u6852\u6885\uD84D\uDC6D\u688E\u681F\u6914\u3B9D\u6942\u69A3\u69EA\u6AA8\uD84D\uDEA3\u6ADB\u3C18\u6B21\uD84E\uDCA7\u6B54\u3C4E\u6B72\u6B9F\u6BBA\u6BBB\uD84E\uDE8D\uD847\uDD0B\uD84E\uDEFA\u6C4E\uD84F\uDCBC\u6CBF\u6CCD\u6C67\u6D16\u6D3E\u6D77\u6D41\u6D69\u6D78\u6D85\uD84F\uDD1E\u6D34\u6E2F\u6E6E\u3D33\u6ECB\u6EC7\uD84F\uDED1\u6DF9\u6F6E\uD84F\uDF5E\uD84F\uDF8E\u6FC6\u7039\u701E\u701B\u3D96\u704A\u707D\u7077\u70AD\uD841\uDD25\u7145\uD850\uDE63\u719C\uD850\uDFAB\u7228\u7235\u7250\uD851\uDE08\u7280\u7295\uD851\uDF35\uD852\uDC14\u737A\u738B\u3EAC\u73A5\u3EB8\u3EB8\u7447\u745C\u7471\u7485\u74CA\u3F1B\u7524\uD853\uDC36\u753E\uD853\uDC92\u7570\uD848\uDD9F\u7610\uD853\uDFA1\uD853\uDFB8\uD854\uDC44\u3FFC\u4008\u76F4\uD854\uDCF3\uD854\uDCF2\uD854\uDD19\uD854\uDD33\u771E\u771F\u771F\u774A\u4039\u778B\u4046\u4096\uD855\uDC1D\u784E\u788C\u78CC\u40E3\uD855\uDE26\u7956\uD855\uDE9A\uD855\uDEC5\u798F\u79EB\u412F\u7A40\u7A4A\u7A4F\uD856\uDD7C\uD856\uDEA7\uD856\uDEA7\u7AEE\u4202\uD856\uDFAB\u7BC6\u7BC9\u4227\uD857\uDC80\u7CD2\u42A0\u7CE8\u7CE3\u7D00\uD857\uDF86\u7D63\u4301\u7DC7\u7E02\u7E45\u4334\uD858\uDE28\uD858\uDE47\u4359\uD858\uDED9\u7F7A\uD858\uDF3E\u7F95\u7FFA\u8005\uD859\uDCDA\uD859\uDD23\u8060\uD859\uDDA8\u8070\uD84C\uDF5F\u43D5\u80B2\u8103\u440B\u813E\u5AB5\uD859\uDFA7\uD859\uDFB5\uD84C\uDF93\uD84C\uDF9C\u8201\u8204\u8F9E\u446B\u8291\u828B\u829D\u52B3\u82B1\u82B3\u82BD\u82E6\uD85A\uDF3C\u82E5\u831D\u8363\u83AD\u8323\u83BD\u83E7\u8457\u8353\u83CA\u83CC\u83DC\uD85B\uDC36\uD85B\uDD6B\uD85B\uDCD5\u452B\u84F1\u84F3\u8516\uD85C\uDFCA\u8564\uD85B\uDF2C\u455D\u4561\uD85B\uDFB1\uD85C\uDCD2\u456B\u8650\u865C\u8667\u8669\u86A9\u8688\u870E\u86E2\u8779\u8728\u876B\u8786\u45D7\u87E1\u8801\u45F9\u8860\u8863\uD85D\uDE67\u88D7\u88DE\u4635\u88FA\u34BB\uD85E\uDCAE\uD85E\uDD66\u46BE\u46C7\u8AA0\u8AED\u8B8A\u8C55\uD85F\uDCA8\u8CAB\u8CC1\u8D1B\u8D77\uD85F\uDF2F\uD842\uDC04\u8DCB\u8DBC\u8DF0\uD842\uDCDE\u8ED4\u8F38\uD861\uDDD2\uD861\uDDED\u9094\u90F1\u9111\uD861\uDF2E\u911B\u9238\u92D7\u92D8\u927C\u93F9\u9415\uD862\uDFFA\u958B\u4995\u95B7\uD863\uDD77\u49E6\u96C3\u5DB2\u9723\uD864\uDD45\uD864\uDE1A\u4A6E\u4A76\u97E0\uD865\uDC0A\u4AB2\uD865\uDC96\u980B\u980B\u9829\uD865\uDDB6\u98E2\u4B33\u9929\u99A7\u99C2\u99FE\u4BCE\uD866\uDF30\u9B12\u9C40\u9CFD\u4CCE\u4CED\u9D67\uD868\uDCCE\u4CF8\uD868\uDD05\uD868\uDE0E\uD868\uDE91\u9EBB\u4D56\u9EF9\u9EFE\u9F05\u9F0F\u9F16\u9F3B\uD869\uDE00'
			},
			{
				'description': 'Unicode character category [Lt]: letter, titlecase',
				'decoded': [453, 456, 459, 498, 8072, 8073, 8074, 8075, 8076, 8077, 8078, 8079, 8088, 8089, 8090, 8091, 8092, 8093, 8094, 8095, 8104, 8105, 8106, 8107, 8108, 8109, 8110, 8111, 8124, 8140, 8188],
				'encoded': '\u01C5\u01C8\u01CB\u01F2\u1F88\u1F89\u1F8A\u1F8B\u1F8C\u1F8D\u1F8E\u1F8F\u1F98\u1F99\u1F9A\u1F9B\u1F9C\u1F9D\u1F9E\u1F9F\u1FA8\u1FA9\u1FAA\u1FAB\u1FAC\u1FAD\u1FAE\u1FAF\u1FBC\u1FCC\u1FFC'
			},
			{
				'description': 'Unicode character category [Lu]: letter, uppercase',
				'decoded': [65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207, 208, 209, 210, 211, 212, 213, 214, 216, 217, 218, 219, 220, 221, 222, 256, 258, 260, 262, 264, 266, 268, 270, 272, 274, 276, 278, 280, 282, 284, 286, 288, 290, 292, 294, 296, 298, 300, 302, 304, 306, 308, 310, 313, 315, 317, 319, 321, 323, 325, 327, 330, 332, 334, 336, 338, 340, 342, 344, 346, 348, 350, 352, 354, 356, 358, 360, 362, 364, 366, 368, 370, 372, 374, 376, 377, 379, 381, 385, 386, 388, 390, 391, 393, 394, 395, 398, 399, 400, 401, 403, 404, 406, 407, 408, 412, 413, 415, 416, 418, 420, 422, 423, 425, 428, 430, 431, 433, 434, 435, 437, 439, 440, 444, 452, 455, 458, 461, 463, 465, 467, 469, 471, 473, 475, 478, 480, 482, 484, 486, 488, 490, 492, 494, 497, 500, 502, 503, 504, 506, 508, 510, 512, 514, 516, 518, 520, 522, 524, 526, 528, 530, 532, 534, 536, 538, 540, 542, 544, 546, 548, 550, 552, 554, 556, 558, 560, 562, 570, 571, 573, 574, 577, 579, 580, 581, 582, 584, 586, 588, 590, 880, 882, 886, 902, 904, 905, 906, 908, 910, 911, 913, 914, 915, 916, 917, 918, 919, 920, 921, 922, 923, 924, 925, 926, 927, 928, 929, 931, 932, 933, 934, 935, 936, 937, 938, 939, 975, 978, 979, 980, 984, 986, 988, 990, 992, 994, 996, 998, 1000, 1002, 1004, 1006, 1012, 1015, 1017, 1018, 1021, 1022, 1023, 1024, 1025, 1026, 1027, 1028, 1029, 1030, 1031, 1032, 1033, 1034, 1035, 1036, 1037, 1038, 1039, 1040, 1041, 1042, 1043, 1044, 1045, 1046, 1047, 1048, 1049, 1050, 1051, 1052, 1053, 1054, 1055, 1056, 1057, 1058, 1059, 1060, 1061, 1062, 1063, 1064, 1065, 1066, 1067, 1068, 1069, 1070, 1071, 1120, 1122, 1124, 1126, 1128, 1130, 1132, 1134, 1136, 1138, 1140, 1142, 1144, 1146, 1148, 1150, 1152, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1202, 1204, 1206, 1208, 1210, 1212, 1214, 1216, 1217, 1219, 1221, 1223, 1225, 1227, 1229, 1232, 1234, 1236, 1238, 1240, 1242, 1244, 1246, 1248, 1250, 1252, 1254, 1256, 1258, 1260, 1262, 1264, 1266, 1268, 1270, 1272, 1274, 1276, 1278, 1280, 1282, 1284, 1286, 1288, 1290, 1292, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1329, 1330, 1331, 1332, 1333, 1334, 1335, 1336, 1337, 1338, 1339, 1340, 1341, 1342, 1343, 1344, 1345, 1346, 1347, 1348, 1349, 1350, 1351, 1352, 1353, 1354, 1355, 1356, 1357, 1358, 1359, 1360, 1361, 1362, 1363, 1364, 1365, 1366, 4256, 4257, 4258, 4259, 4260, 4261, 4262, 4263, 4264, 4265, 4266, 4267, 4268, 4269, 4270, 4271, 4272, 4273, 4274, 4275, 4276, 4277, 4278, 4279, 4280, 4281, 4282, 4283, 4284, 4285, 4286, 4287, 4288, 4289, 4290, 4291, 4292, 4293, 7680, 7682, 7684, 7686, 7688, 7690, 7692, 7694, 7696, 7698, 7700, 7702, 7704, 7706, 7708, 7710, 7712, 7714, 7716, 7718, 7720, 7722, 7724, 7726, 7728, 7730, 7732, 7734, 7736, 7738, 7740, 7742, 7744, 7746, 7748, 7750, 7752, 7754, 7756, 7758, 7760, 7762, 7764, 7766, 7768, 7770, 7772, 7774, 7776, 7778, 7780, 7782, 7784, 7786, 7788, 7790, 7792, 7794, 7796, 7798, 7800, 7802, 7804, 7806, 7808, 7810, 7812, 7814, 7816, 7818, 7820, 7822, 7824, 7826, 7828, 7838, 7840, 7842, 7844, 7846, 7848, 7850, 7852, 7854, 7856, 7858, 7860, 7862, 7864, 7866, 7868, 7870, 7872, 7874, 7876, 7878, 7880, 7882, 7884, 7886, 7888, 7890, 7892, 7894, 7896, 7898, 7900, 7902, 7904, 7906, 7908, 7910, 7912, 7914, 7916, 7918, 7920, 7922, 7924, 7926, 7928, 7930, 7932, 7934, 7944, 7945, 7946, 7947, 7948, 7949, 7950, 7951, 7960, 7961, 7962, 7963, 7964, 7965, 7976, 7977, 7978, 7979, 7980, 7981, 7982, 7983, 7992, 7993, 7994, 7995, 7996, 7997, 7998, 7999, 8008, 8009, 8010, 8011, 8012, 8013, 8025, 8027, 8029, 8031, 8040, 8041, 8042, 8043, 8044, 8045, 8046, 8047, 8120, 8121, 8122, 8123, 8136, 8137, 8138, 8139, 8152, 8153, 8154, 8155, 8168, 8169, 8170, 8171, 8172, 8184, 8185, 8186, 8187, 8450, 8455, 8459, 8460, 8461, 8464, 8465, 8466, 8469, 8473, 8474, 8475, 8476, 8477, 8484, 8486, 8488, 8490, 8491, 8492, 8493, 8496, 8497, 8498, 8499, 8510, 8511, 8517, 8579, 11264, 11265, 11266, 11267, 11268, 11269, 11270, 11271, 11272, 11273, 11274, 11275, 11276, 11277, 11278, 11279, 11280, 11281, 11282, 11283, 11284, 11285, 11286, 11287, 11288, 11289, 11290, 11291, 11292, 11293, 11294, 11295, 11296, 11297, 11298, 11299, 11300, 11301, 11302, 11303, 11304, 11305, 11306, 11307, 11308, 11309, 11310, 11360, 11362, 11363, 11364, 11367, 11369, 11371, 11373, 11374, 11375, 11376, 11378, 11381, 11390, 11391, 11392, 11394, 11396, 11398, 11400, 11402, 11404, 11406, 11408, 11410, 11412, 11414, 11416, 11418, 11420, 11422, 11424, 11426, 11428, 11430, 11432, 11434, 11436, 11438, 11440, 11442, 11444, 11446, 11448, 11450, 11452, 11454, 11456, 11458, 11460, 11462, 11464, 11466, 11468, 11470, 11472, 11474, 11476, 11478, 11480, 11482, 11484, 11486, 11488, 11490, 11499, 11501, 42560, 42562, 42564, 42566, 42568, 42570, 42572, 42574, 42576, 42578, 42580, 42582, 42584, 42586, 42588, 42590, 42592, 42594, 42596, 42598, 42600, 42602, 42604, 42624, 42626, 42628, 42630, 42632, 42634, 42636, 42638, 42640, 42642, 42644, 42646, 42786, 42788, 42790, 42792, 42794, 42796, 42798, 42802, 42804, 42806, 42808, 42810, 42812, 42814, 42816, 42818, 42820, 42822, 42824, 42826, 42828, 42830, 42832, 42834, 42836, 42838, 42840, 42842, 42844, 42846, 42848, 42850, 42852, 42854, 42856, 42858, 42860, 42862, 42873, 42875, 42877, 42878, 42880, 42882, 42884, 42886, 42891, 42893, 42896, 42912, 42914, 42916, 42918, 42920, 65313, 65314, 65315, 65316, 65317, 65318, 65319, 65320, 65321, 65322, 65323, 65324, 65325, 65326, 65327, 65328, 65329, 65330, 65331, 65332, 65333, 65334, 65335, 65336, 65337, 65338, 66560, 66561, 66562, 66563, 66564, 66565, 66566, 66567, 66568, 66569, 66570, 66571, 66572, 66573, 66574, 66575, 66576, 66577, 66578, 66579, 66580, 66581, 66582, 66583, 66584, 66585, 66586, 66587, 66588, 66589, 66590, 66591, 66592, 66593, 66594, 66595, 66596, 66597, 66598, 66599, 119808, 119809, 119810, 119811, 119812, 119813, 119814, 119815, 119816, 119817, 119818, 119819, 119820, 119821, 119822, 119823, 119824, 119825, 119826, 119827, 119828, 119829, 119830, 119831, 119832, 119833, 119860, 119861, 119862, 119863, 119864, 119865, 119866, 119867, 119868, 119869, 119870, 119871, 119872, 119873, 119874, 119875, 119876, 119877, 119878, 119879, 119880, 119881, 119882, 119883, 119884, 119885, 119912, 119913, 119914, 119915, 119916, 119917, 119918, 119919, 119920, 119921, 119922, 119923, 119924, 119925, 119926, 119927, 119928, 119929, 119930, 119931, 119932, 119933, 119934, 119935, 119936, 119937, 119964, 119966, 119967, 119970, 119973, 119974, 119977, 119978, 119979, 119980, 119982, 119983, 119984, 119985, 119986, 119987, 119988, 119989, 120016, 120017, 120018, 120019, 120020, 120021, 120022, 120023, 120024, 120025, 120026, 120027, 120028, 120029, 120030, 120031, 120032, 120033, 120034, 120035, 120036, 120037, 120038, 120039, 120040, 120041, 120068, 120069, 120071, 120072, 120073, 120074, 120077, 120078, 120079, 120080, 120081, 120082, 120083, 120084, 120086, 120087, 120088, 120089, 120090, 120091, 120092, 120120, 120121, 120123, 120124, 120125, 120126, 120128, 120129, 120130, 120131, 120132, 120134, 120138, 120139, 120140, 120141, 120142, 120143, 120144, 120172, 120173, 120174, 120175, 120176, 120177, 120178, 120179, 120180, 120181, 120182, 120183, 120184, 120185, 120186, 120187, 120188, 120189, 120190, 120191, 120192, 120193, 120194, 120195, 120196, 120197, 120224, 120225, 120226, 120227, 120228, 120229, 120230, 120231, 120232, 120233, 120234, 120235, 120236, 120237, 120238, 120239, 120240, 120241, 120242, 120243, 120244, 120245, 120246, 120247, 120248, 120249, 120276, 120277, 120278, 120279, 120280, 120281, 120282, 120283, 120284, 120285, 120286, 120287, 120288, 120289, 120290, 120291, 120292, 120293, 120294, 120295, 120296, 120297, 120298, 120299, 120300, 120301, 120328, 120329, 120330, 120331, 120332, 120333, 120334, 120335, 120336, 120337, 120338, 120339, 120340, 120341, 120342, 120343, 120344, 120345, 120346, 120347, 120348, 120349, 120350, 120351, 120352, 120353, 120380, 120381, 120382, 120383, 120384, 120385, 120386, 120387, 120388, 120389, 120390, 120391, 120392, 120393, 120394, 120395, 120396, 120397, 120398, 120399, 120400, 120401, 120402, 120403, 120404, 120405, 120432, 120433, 120434, 120435, 120436, 120437, 120438, 120439, 120440, 120441, 120442, 120443, 120444, 120445, 120446, 120447, 120448, 120449, 120450, 120451, 120452, 120453, 120454, 120455, 120456, 120457, 120488, 120489, 120490, 120491, 120492, 120493, 120494, 120495, 120496, 120497, 120498, 120499, 120500, 120501, 120502, 120503, 120504, 120505, 120506, 120507, 120508, 120509, 120510, 120511, 120512, 120546, 120547, 120548, 120549, 120550, 120551, 120552, 120553, 120554, 120555, 120556, 120557, 120558, 120559, 120560, 120561, 120562, 120563, 120564, 120565, 120566, 120567, 120568, 120569, 120570, 120604, 120605, 120606, 120607, 120608, 120609, 120610, 120611, 120612, 120613, 120614, 120615, 120616, 120617, 120618, 120619, 120620, 120621, 120622, 120623, 120624, 120625, 120626, 120627, 120628, 120662, 120663, 120664, 120665, 120666, 120667, 120668, 120669, 120670, 120671, 120672, 120673, 120674, 120675, 120676, 120677, 120678, 120679, 120680, 120681, 120682, 120683, 120684, 120685, 120686, 120720, 120721, 120722, 120723, 120724, 120725, 120726, 120727, 120728, 120729, 120730, 120731, 120732, 120733, 120734, 120735, 120736, 120737, 120738, 120739, 120740, 120741, 120742, 120743, 120744, 120778],
				'encoded': 'ABCDEFGHIJKLMNOPQRSTUVWXYZ\xC0\xC1\xC2\xC3\xC4\xC5\xC6\xC7\xC8\xC9\xCA\xCB\xCC\xCD\xCE\xCF\xD0\xD1\xD2\xD3\xD4\xD5\xD6\xD8\xD9\xDA\xDB\xDC\xDD\xDE\u0100\u0102\u0104\u0106\u0108\u010A\u010C\u010E\u0110\u0112\u0114\u0116\u0118\u011A\u011C\u011E\u0120\u0122\u0124\u0126\u0128\u012A\u012C\u012E\u0130\u0132\u0134\u0136\u0139\u013B\u013D\u013F\u0141\u0143\u0145\u0147\u014A\u014C\u014E\u0150\u0152\u0154\u0156\u0158\u015A\u015C\u015E\u0160\u0162\u0164\u0166\u0168\u016A\u016C\u016E\u0170\u0172\u0174\u0176\u0178\u0179\u017B\u017D\u0181\u0182\u0184\u0186\u0187\u0189\u018A\u018B\u018E\u018F\u0190\u0191\u0193\u0194\u0196\u0197\u0198\u019C\u019D\u019F\u01A0\u01A2\u01A4\u01A6\u01A7\u01A9\u01AC\u01AE\u01AF\u01B1\u01B2\u01B3\u01B5\u01B7\u01B8\u01BC\u01C4\u01C7\u01CA\u01CD\u01CF\u01D1\u01D3\u01D5\u01D7\u01D9\u01DB\u01DE\u01E0\u01E2\u01E4\u01E6\u01E8\u01EA\u01EC\u01EE\u01F1\u01F4\u01F6\u01F7\u01F8\u01FA\u01FC\u01FE\u0200\u0202\u0204\u0206\u0208\u020A\u020C\u020E\u0210\u0212\u0214\u0216\u0218\u021A\u021C\u021E\u0220\u0222\u0224\u0226\u0228\u022A\u022C\u022E\u0230\u0232\u023A\u023B\u023D\u023E\u0241\u0243\u0244\u0245\u0246\u0248\u024A\u024C\u024E\u0370\u0372\u0376\u0386\u0388\u0389\u038A\u038C\u038E\u038F\u0391\u0392\u0393\u0394\u0395\u0396\u0397\u0398\u0399\u039A\u039B\u039C\u039D\u039E\u039F\u03A0\u03A1\u03A3\u03A4\u03A5\u03A6\u03A7\u03A8\u03A9\u03AA\u03AB\u03CF\u03D2\u03D3\u03D4\u03D8\u03DA\u03DC\u03DE\u03E0\u03E2\u03E4\u03E6\u03E8\u03EA\u03EC\u03EE\u03F4\u03F7\u03F9\u03FA\u03FD\u03FE\u03FF\u0400\u0401\u0402\u0403\u0404\u0405\u0406\u0407\u0408\u0409\u040A\u040B\u040C\u040D\u040E\u040F\u0410\u0411\u0412\u0413\u0414\u0415\u0416\u0417\u0418\u0419\u041A\u041B\u041C\u041D\u041E\u041F\u0420\u0421\u0422\u0423\u0424\u0425\u0426\u0427\u0428\u0429\u042A\u042B\u042C\u042D\u042E\u042F\u0460\u0462\u0464\u0466\u0468\u046A\u046C\u046E\u0470\u0472\u0474\u0476\u0478\u047A\u047C\u047E\u0480\u048A\u048C\u048E\u0490\u0492\u0494\u0496\u0498\u049A\u049C\u049E\u04A0\u04A2\u04A4\u04A6\u04A8\u04AA\u04AC\u04AE\u04B0\u04B2\u04B4\u04B6\u04B8\u04BA\u04BC\u04BE\u04C0\u04C1\u04C3\u04C5\u04C7\u04C9\u04CB\u04CD\u04D0\u04D2\u04D4\u04D6\u04D8\u04DA\u04DC\u04DE\u04E0\u04E2\u04E4\u04E6\u04E8\u04EA\u04EC\u04EE\u04F0\u04F2\u04F4\u04F6\u04F8\u04FA\u04FC\u04FE\u0500\u0502\u0504\u0506\u0508\u050A\u050C\u050E\u0510\u0512\u0514\u0516\u0518\u051A\u051C\u051E\u0520\u0522\u0524\u0526\u0531\u0532\u0533\u0534\u0535\u0536\u0537\u0538\u0539\u053A\u053B\u053C\u053D\u053E\u053F\u0540\u0541\u0542\u0543\u0544\u0545\u0546\u0547\u0548\u0549\u054A\u054B\u054C\u054D\u054E\u054F\u0550\u0551\u0552\u0553\u0554\u0555\u0556\u10A0\u10A1\u10A2\u10A3\u10A4\u10A5\u10A6\u10A7\u10A8\u10A9\u10AA\u10AB\u10AC\u10AD\u10AE\u10AF\u10B0\u10B1\u10B2\u10B3\u10B4\u10B5\u10B6\u10B7\u10B8\u10B9\u10BA\u10BB\u10BC\u10BD\u10BE\u10BF\u10C0\u10C1\u10C2\u10C3\u10C4\u10C5\u1E00\u1E02\u1E04\u1E06\u1E08\u1E0A\u1E0C\u1E0E\u1E10\u1E12\u1E14\u1E16\u1E18\u1E1A\u1E1C\u1E1E\u1E20\u1E22\u1E24\u1E26\u1E28\u1E2A\u1E2C\u1E2E\u1E30\u1E32\u1E34\u1E36\u1E38\u1E3A\u1E3C\u1E3E\u1E40\u1E42\u1E44\u1E46\u1E48\u1E4A\u1E4C\u1E4E\u1E50\u1E52\u1E54\u1E56\u1E58\u1E5A\u1E5C\u1E5E\u1E60\u1E62\u1E64\u1E66\u1E68\u1E6A\u1E6C\u1E6E\u1E70\u1E72\u1E74\u1E76\u1E78\u1E7A\u1E7C\u1E7E\u1E80\u1E82\u1E84\u1E86\u1E88\u1E8A\u1E8C\u1E8E\u1E90\u1E92\u1E94\u1E9E\u1EA0\u1EA2\u1EA4\u1EA6\u1EA8\u1EAA\u1EAC\u1EAE\u1EB0\u1EB2\u1EB4\u1EB6\u1EB8\u1EBA\u1EBC\u1EBE\u1EC0\u1EC2\u1EC4\u1EC6\u1EC8\u1ECA\u1ECC\u1ECE\u1ED0\u1ED2\u1ED4\u1ED6\u1ED8\u1EDA\u1EDC\u1EDE\u1EE0\u1EE2\u1EE4\u1EE6\u1EE8\u1EEA\u1EEC\u1EEE\u1EF0\u1EF2\u1EF4\u1EF6\u1EF8\u1EFA\u1EFC\u1EFE\u1F08\u1F09\u1F0A\u1F0B\u1F0C\u1F0D\u1F0E\u1F0F\u1F18\u1F19\u1F1A\u1F1B\u1F1C\u1F1D\u1F28\u1F29\u1F2A\u1F2B\u1F2C\u1F2D\u1F2E\u1F2F\u1F38\u1F39\u1F3A\u1F3B\u1F3C\u1F3D\u1F3E\u1F3F\u1F48\u1F49\u1F4A\u1F4B\u1F4C\u1F4D\u1F59\u1F5B\u1F5D\u1F5F\u1F68\u1F69\u1F6A\u1F6B\u1F6C\u1F6D\u1F6E\u1F6F\u1FB8\u1FB9\u1FBA\u1FBB\u1FC8\u1FC9\u1FCA\u1FCB\u1FD8\u1FD9\u1FDA\u1FDB\u1FE8\u1FE9\u1FEA\u1FEB\u1FEC\u1FF8\u1FF9\u1FFA\u1FFB\u2102\u2107\u210B\u210C\u210D\u2110\u2111\u2112\u2115\u2119\u211A\u211B\u211C\u211D\u2124\u2126\u2128\u212A\u212B\u212C\u212D\u2130\u2131\u2132\u2133\u213E\u213F\u2145\u2183\u2C00\u2C01\u2C02\u2C03\u2C04\u2C05\u2C06\u2C07\u2C08\u2C09\u2C0A\u2C0B\u2C0C\u2C0D\u2C0E\u2C0F\u2C10\u2C11\u2C12\u2C13\u2C14\u2C15\u2C16\u2C17\u2C18\u2C19\u2C1A\u2C1B\u2C1C\u2C1D\u2C1E\u2C1F\u2C20\u2C21\u2C22\u2C23\u2C24\u2C25\u2C26\u2C27\u2C28\u2C29\u2C2A\u2C2B\u2C2C\u2C2D\u2C2E\u2C60\u2C62\u2C63\u2C64\u2C67\u2C69\u2C6B\u2C6D\u2C6E\u2C6F\u2C70\u2C72\u2C75\u2C7E\u2C7F\u2C80\u2C82\u2C84\u2C86\u2C88\u2C8A\u2C8C\u2C8E\u2C90\u2C92\u2C94\u2C96\u2C98\u2C9A\u2C9C\u2C9E\u2CA0\u2CA2\u2CA4\u2CA6\u2CA8\u2CAA\u2CAC\u2CAE\u2CB0\u2CB2\u2CB4\u2CB6\u2CB8\u2CBA\u2CBC\u2CBE\u2CC0\u2CC2\u2CC4\u2CC6\u2CC8\u2CCA\u2CCC\u2CCE\u2CD0\u2CD2\u2CD4\u2CD6\u2CD8\u2CDA\u2CDC\u2CDE\u2CE0\u2CE2\u2CEB\u2CED\uA640\uA642\uA644\uA646\uA648\uA64A\uA64C\uA64E\uA650\uA652\uA654\uA656\uA658\uA65A\uA65C\uA65E\uA660\uA662\uA664\uA666\uA668\uA66A\uA66C\uA680\uA682\uA684\uA686\uA688\uA68A\uA68C\uA68E\uA690\uA692\uA694\uA696\uA722\uA724\uA726\uA728\uA72A\uA72C\uA72E\uA732\uA734\uA736\uA738\uA73A\uA73C\uA73E\uA740\uA742\uA744\uA746\uA748\uA74A\uA74C\uA74E\uA750\uA752\uA754\uA756\uA758\uA75A\uA75C\uA75E\uA760\uA762\uA764\uA766\uA768\uA76A\uA76C\uA76E\uA779\uA77B\uA77D\uA77E\uA780\uA782\uA784\uA786\uA78B\uA78D\uA790\uA7A0\uA7A2\uA7A4\uA7A6\uA7A8\uFF21\uFF22\uFF23\uFF24\uFF25\uFF26\uFF27\uFF28\uFF29\uFF2A\uFF2B\uFF2C\uFF2D\uFF2E\uFF2F\uFF30\uFF31\uFF32\uFF33\uFF34\uFF35\uFF36\uFF37\uFF38\uFF39\uFF3A\uD801\uDC00\uD801\uDC01\uD801\uDC02\uD801\uDC03\uD801\uDC04\uD801\uDC05\uD801\uDC06\uD801\uDC07\uD801\uDC08\uD801\uDC09\uD801\uDC0A\uD801\uDC0B\uD801\uDC0C\uD801\uDC0D\uD801\uDC0E\uD801\uDC0F\uD801\uDC10\uD801\uDC11\uD801\uDC12\uD801\uDC13\uD801\uDC14\uD801\uDC15\uD801\uDC16\uD801\uDC17\uD801\uDC18\uD801\uDC19\uD801\uDC1A\uD801\uDC1B\uD801\uDC1C\uD801\uDC1D\uD801\uDC1E\uD801\uDC1F\uD801\uDC20\uD801\uDC21\uD801\uDC22\uD801\uDC23\uD801\uDC24\uD801\uDC25\uD801\uDC26\uD801\uDC27\uD835\uDC00\uD835\uDC01\uD835\uDC02\uD835\uDC03\uD835\uDC04\uD835\uDC05\uD835\uDC06\uD835\uDC07\uD835\uDC08\uD835\uDC09\uD835\uDC0A\uD835\uDC0B\uD835\uDC0C\uD835\uDC0D\uD835\uDC0E\uD835\uDC0F\uD835\uDC10\uD835\uDC11\uD835\uDC12\uD835\uDC13\uD835\uDC14\uD835\uDC15\uD835\uDC16\uD835\uDC17\uD835\uDC18\uD835\uDC19\uD835\uDC34\uD835\uDC35\uD835\uDC36\uD835\uDC37\uD835\uDC38\uD835\uDC39\uD835\uDC3A\uD835\uDC3B\uD835\uDC3C\uD835\uDC3D\uD835\uDC3E\uD835\uDC3F\uD835\uDC40\uD835\uDC41\uD835\uDC42\uD835\uDC43\uD835\uDC44\uD835\uDC45\uD835\uDC46\uD835\uDC47\uD835\uDC48\uD835\uDC49\uD835\uDC4A\uD835\uDC4B\uD835\uDC4C\uD835\uDC4D\uD835\uDC68\uD835\uDC69\uD835\uDC6A\uD835\uDC6B\uD835\uDC6C\uD835\uDC6D\uD835\uDC6E\uD835\uDC6F\uD835\uDC70\uD835\uDC71\uD835\uDC72\uD835\uDC73\uD835\uDC74\uD835\uDC75\uD835\uDC76\uD835\uDC77\uD835\uDC78\uD835\uDC79\uD835\uDC7A\uD835\uDC7B\uD835\uDC7C\uD835\uDC7D\uD835\uDC7E\uD835\uDC7F\uD835\uDC80\uD835\uDC81\uD835\uDC9C\uD835\uDC9E\uD835\uDC9F\uD835\uDCA2\uD835\uDCA5\uD835\uDCA6\uD835\uDCA9\uD835\uDCAA\uD835\uDCAB\uD835\uDCAC\uD835\uDCAE\uD835\uDCAF\uD835\uDCB0\uD835\uDCB1\uD835\uDCB2\uD835\uDCB3\uD835\uDCB4\uD835\uDCB5\uD835\uDCD0\uD835\uDCD1\uD835\uDCD2\uD835\uDCD3\uD835\uDCD4\uD835\uDCD5\uD835\uDCD6\uD835\uDCD7\uD835\uDCD8\uD835\uDCD9\uD835\uDCDA\uD835\uDCDB\uD835\uDCDC\uD835\uDCDD\uD835\uDCDE\uD835\uDCDF\uD835\uDCE0\uD835\uDCE1\uD835\uDCE2\uD835\uDCE3\uD835\uDCE4\uD835\uDCE5\uD835\uDCE6\uD835\uDCE7\uD835\uDCE8\uD835\uDCE9\uD835\uDD04\uD835\uDD05\uD835\uDD07\uD835\uDD08\uD835\uDD09\uD835\uDD0A\uD835\uDD0D\uD835\uDD0E\uD835\uDD0F\uD835\uDD10\uD835\uDD11\uD835\uDD12\uD835\uDD13\uD835\uDD14\uD835\uDD16\uD835\uDD17\uD835\uDD18\uD835\uDD19\uD835\uDD1A\uD835\uDD1B\uD835\uDD1C\uD835\uDD38\uD835\uDD39\uD835\uDD3B\uD835\uDD3C\uD835\uDD3D\uD835\uDD3E\uD835\uDD40\uD835\uDD41\uD835\uDD42\uD835\uDD43\uD835\uDD44\uD835\uDD46\uD835\uDD4A\uD835\uDD4B\uD835\uDD4C\uD835\uDD4D\uD835\uDD4E\uD835\uDD4F\uD835\uDD50\uD835\uDD6C\uD835\uDD6D\uD835\uDD6E\uD835\uDD6F\uD835\uDD70\uD835\uDD71\uD835\uDD72\uD835\uDD73\uD835\uDD74\uD835\uDD75\uD835\uDD76\uD835\uDD77\uD835\uDD78\uD835\uDD79\uD835\uDD7A\uD835\uDD7B\uD835\uDD7C\uD835\uDD7D\uD835\uDD7E\uD835\uDD7F\uD835\uDD80\uD835\uDD81\uD835\uDD82\uD835\uDD83\uD835\uDD84\uD835\uDD85\uD835\uDDA0\uD835\uDDA1\uD835\uDDA2\uD835\uDDA3\uD835\uDDA4\uD835\uDDA5\uD835\uDDA6\uD835\uDDA7\uD835\uDDA8\uD835\uDDA9\uD835\uDDAA\uD835\uDDAB\uD835\uDDAC\uD835\uDDAD\uD835\uDDAE\uD835\uDDAF\uD835\uDDB0\uD835\uDDB1\uD835\uDDB2\uD835\uDDB3\uD835\uDDB4\uD835\uDDB5\uD835\uDDB6\uD835\uDDB7\uD835\uDDB8\uD835\uDDB9\uD835\uDDD4\uD835\uDDD5\uD835\uDDD6\uD835\uDDD7\uD835\uDDD8\uD835\uDDD9\uD835\uDDDA\uD835\uDDDB\uD835\uDDDC\uD835\uDDDD\uD835\uDDDE\uD835\uDDDF\uD835\uDDE0\uD835\uDDE1\uD835\uDDE2\uD835\uDDE3\uD835\uDDE4\uD835\uDDE5\uD835\uDDE6\uD835\uDDE7\uD835\uDDE8\uD835\uDDE9\uD835\uDDEA\uD835\uDDEB\uD835\uDDEC\uD835\uDDED\uD835\uDE08\uD835\uDE09\uD835\uDE0A\uD835\uDE0B\uD835\uDE0C\uD835\uDE0D\uD835\uDE0E\uD835\uDE0F\uD835\uDE10\uD835\uDE11\uD835\uDE12\uD835\uDE13\uD835\uDE14\uD835\uDE15\uD835\uDE16\uD835\uDE17\uD835\uDE18\uD835\uDE19\uD835\uDE1A\uD835\uDE1B\uD835\uDE1C\uD835\uDE1D\uD835\uDE1E\uD835\uDE1F\uD835\uDE20\uD835\uDE21\uD835\uDE3C\uD835\uDE3D\uD835\uDE3E\uD835\uDE3F\uD835\uDE40\uD835\uDE41\uD835\uDE42\uD835\uDE43\uD835\uDE44\uD835\uDE45\uD835\uDE46\uD835\uDE47\uD835\uDE48\uD835\uDE49\uD835\uDE4A\uD835\uDE4B\uD835\uDE4C\uD835\uDE4D\uD835\uDE4E\uD835\uDE4F\uD835\uDE50\uD835\uDE51\uD835\uDE52\uD835\uDE53\uD835\uDE54\uD835\uDE55\uD835\uDE70\uD835\uDE71\uD835\uDE72\uD835\uDE73\uD835\uDE74\uD835\uDE75\uD835\uDE76\uD835\uDE77\uD835\uDE78\uD835\uDE79\uD835\uDE7A\uD835\uDE7B\uD835\uDE7C\uD835\uDE7D\uD835\uDE7E\uD835\uDE7F\uD835\uDE80\uD835\uDE81\uD835\uDE82\uD835\uDE83\uD835\uDE84\uD835\uDE85\uD835\uDE86\uD835\uDE87\uD835\uDE88\uD835\uDE89\uD835\uDEA8\uD835\uDEA9\uD835\uDEAA\uD835\uDEAB\uD835\uDEAC\uD835\uDEAD\uD835\uDEAE\uD835\uDEAF\uD835\uDEB0\uD835\uDEB1\uD835\uDEB2\uD835\uDEB3\uD835\uDEB4\uD835\uDEB5\uD835\uDEB6\uD835\uDEB7\uD835\uDEB8\uD835\uDEB9\uD835\uDEBA\uD835\uDEBB\uD835\uDEBC\uD835\uDEBD\uD835\uDEBE\uD835\uDEBF\uD835\uDEC0\uD835\uDEE2\uD835\uDEE3\uD835\uDEE4\uD835\uDEE5\uD835\uDEE6\uD835\uDEE7\uD835\uDEE8\uD835\uDEE9\uD835\uDEEA\uD835\uDEEB\uD835\uDEEC\uD835\uDEED\uD835\uDEEE\uD835\uDEEF\uD835\uDEF0\uD835\uDEF1\uD835\uDEF2\uD835\uDEF3\uD835\uDEF4\uD835\uDEF5\uD835\uDEF6\uD835\uDEF7\uD835\uDEF8\uD835\uDEF9\uD835\uDEFA\uD835\uDF1C\uD835\uDF1D\uD835\uDF1E\uD835\uDF1F\uD835\uDF20\uD835\uDF21\uD835\uDF22\uD835\uDF23\uD835\uDF24\uD835\uDF25\uD835\uDF26\uD835\uDF27\uD835\uDF28\uD835\uDF29\uD835\uDF2A\uD835\uDF2B\uD835\uDF2C\uD835\uDF2D\uD835\uDF2E\uD835\uDF2F\uD835\uDF30\uD835\uDF31\uD835\uDF32\uD835\uDF33\uD835\uDF34\uD835\uDF56\uD835\uDF57\uD835\uDF58\uD835\uDF59\uD835\uDF5A\uD835\uDF5B\uD835\uDF5C\uD835\uDF5D\uD835\uDF5E\uD835\uDF5F\uD835\uDF60\uD835\uDF61\uD835\uDF62\uD835\uDF63\uD835\uDF64\uD835\uDF65\uD835\uDF66\uD835\uDF67\uD835\uDF68\uD835\uDF69\uD835\uDF6A\uD835\uDF6B\uD835\uDF6C\uD835\uDF6D\uD835\uDF6E\uD835\uDF90\uD835\uDF91\uD835\uDF92\uD835\uDF93\uD835\uDF94\uD835\uDF95\uD835\uDF96\uD835\uDF97\uD835\uDF98\uD835\uDF99\uD835\uDF9A\uD835\uDF9B\uD835\uDF9C\uD835\uDF9D\uD835\uDF9E\uD835\uDF9F\uD835\uDFA0\uD835\uDFA1\uD835\uDFA2\uD835\uDFA3\uD835\uDFA4\uD835\uDFA5\uD835\uDFA6\uD835\uDFA7\uD835\uDFA8\uD835\uDFCA'
			},
			{
				'description': 'Unicode character category [Mc]: mark, spacing combining',
				'decoded': [2307, 2363, 2366, 2367, 2368, 2377, 2378, 2379, 2380, 2382, 2383, 2434, 2435, 2494, 2495, 2496, 2503, 2504, 2507, 2508, 2519, 2563, 2622, 2623, 2624, 2691, 2750, 2751, 2752, 2761, 2763, 2764, 2818, 2819, 2878, 2880, 2887, 2888, 2891, 2892, 2903, 3006, 3007, 3009, 3010, 3014, 3015, 3016, 3018, 3019, 3020, 3031, 3073, 3074, 3075, 3137, 3138, 3139, 3140, 3202, 3203, 3262, 3264, 3265, 3266, 3267, 3268, 3271, 3272, 3274, 3275, 3285, 3286, 3330, 3331, 3390, 3391, 3392, 3398, 3399, 3400, 3402, 3403, 3404, 3415, 3458, 3459, 3535, 3536, 3537, 3544, 3545, 3546, 3547, 3548, 3549, 3550, 3551, 3570, 3571, 3902, 3903, 3967, 4139, 4140, 4145, 4152, 4155, 4156, 4182, 4183, 4194, 4195, 4196, 4199, 4200, 4201, 4202, 4203, 4204, 4205, 4227, 4228, 4231, 4232, 4233, 4234, 4235, 4236, 4239, 4250, 4251, 4252, 6070, 6078, 6079, 6080, 6081, 6082, 6083, 6084, 6085, 6087, 6088, 6435, 6436, 6437, 6438, 6441, 6442, 6443, 6448, 6449, 6451, 6452, 6453, 6454, 6455, 6456, 6576, 6577, 6578, 6579, 6580, 6581, 6582, 6583, 6584, 6585, 6586, 6587, 6588, 6589, 6590, 6591, 6592, 6600, 6601, 6681, 6682, 6683, 6741, 6743, 6753, 6755, 6756, 6765, 6766, 6767, 6768, 6769, 6770, 6916, 6965, 6971, 6973, 6974, 6975, 6976, 6977, 6979, 6980, 7042, 7073, 7078, 7079, 7082, 7143, 7146, 7147, 7148, 7150, 7154, 7155, 7204, 7205, 7206, 7207, 7208, 7209, 7210, 7211, 7220, 7221, 7393, 7410, 43043, 43044, 43047, 43136, 43137, 43188, 43189, 43190, 43191, 43192, 43193, 43194, 43195, 43196, 43197, 43198, 43199, 43200, 43201, 43202, 43203, 43346, 43347, 43395, 43444, 43445, 43450, 43451, 43453, 43454, 43455, 43456, 43567, 43568, 43571, 43572, 43597, 43643, 44003, 44004, 44006, 44007, 44009, 44010, 44012, 69632, 69634, 69762, 69808, 69809, 69810, 69815, 69816, 119141, 119142, 119149, 119150, 119151, 119152, 119153, 119154],
				'encoded': '\u0903\u093B\u093E\u093F\u0940\u0949\u094A\u094B\u094C\u094E\u094F\u0982\u0983\u09BE\u09BF\u09C0\u09C7\u09C8\u09CB\u09CC\u09D7\u0A03\u0A3E\u0A3F\u0A40\u0A83\u0ABE\u0ABF\u0AC0\u0AC9\u0ACB\u0ACC\u0B02\u0B03\u0B3E\u0B40\u0B47\u0B48\u0B4B\u0B4C\u0B57\u0BBE\u0BBF\u0BC1\u0BC2\u0BC6\u0BC7\u0BC8\u0BCA\u0BCB\u0BCC\u0BD7\u0C01\u0C02\u0C03\u0C41\u0C42\u0C43\u0C44\u0C82\u0C83\u0CBE\u0CC0\u0CC1\u0CC2\u0CC3\u0CC4\u0CC7\u0CC8\u0CCA\u0CCB\u0CD5\u0CD6\u0D02\u0D03\u0D3E\u0D3F\u0D40\u0D46\u0D47\u0D48\u0D4A\u0D4B\u0D4C\u0D57\u0D82\u0D83\u0DCF\u0DD0\u0DD1\u0DD8\u0DD9\u0DDA\u0DDB\u0DDC\u0DDD\u0DDE\u0DDF\u0DF2\u0DF3\u0F3E\u0F3F\u0F7F\u102B\u102C\u1031\u1038\u103B\u103C\u1056\u1057\u1062\u1063\u1064\u1067\u1068\u1069\u106A\u106B\u106C\u106D\u1083\u1084\u1087\u1088\u1089\u108A\u108B\u108C\u108F\u109A\u109B\u109C\u17B6\u17BE\u17BF\u17C0\u17C1\u17C2\u17C3\u17C4\u17C5\u17C7\u17C8\u1923\u1924\u1925\u1926\u1929\u192A\u192B\u1930\u1931\u1933\u1934\u1935\u1936\u1937\u1938\u19B0\u19B1\u19B2\u19B3\u19B4\u19B5\u19B6\u19B7\u19B8\u19B9\u19BA\u19BB\u19BC\u19BD\u19BE\u19BF\u19C0\u19C8\u19C9\u1A19\u1A1A\u1A1B\u1A55\u1A57\u1A61\u1A63\u1A64\u1A6D\u1A6E\u1A6F\u1A70\u1A71\u1A72\u1B04\u1B35\u1B3B\u1B3D\u1B3E\u1B3F\u1B40\u1B41\u1B43\u1B44\u1B82\u1BA1\u1BA6\u1BA7\u1BAA\u1BE7\u1BEA\u1BEB\u1BEC\u1BEE\u1BF2\u1BF3\u1C24\u1C25\u1C26\u1C27\u1C28\u1C29\u1C2A\u1C2B\u1C34\u1C35\u1CE1\u1CF2\uA823\uA824\uA827\uA880\uA881\uA8B4\uA8B5\uA8B6\uA8B7\uA8B8\uA8B9\uA8BA\uA8BB\uA8BC\uA8BD\uA8BE\uA8BF\uA8C0\uA8C1\uA8C2\uA8C3\uA952\uA953\uA983\uA9B4\uA9B5\uA9BA\uA9BB\uA9BD\uA9BE\uA9BF\uA9C0\uAA2F\uAA30\uAA33\uAA34\uAA4D\uAA7B\uABE3\uABE4\uABE6\uABE7\uABE9\uABEA\uABEC\uD804\uDC00\uD804\uDC02\uD804\uDC82\uD804\uDCB0\uD804\uDCB1\uD804\uDCB2\uD804\uDCB7\uD804\uDCB8\uD834\uDD65\uD834\uDD66\uD834\uDD6D\uD834\uDD6E\uD834\uDD6F\uD834\uDD70\uD834\uDD71\uD834\uDD72'
			},
			{
				'description': 'Unicode character category [Me]: mark, enclosing',
				'decoded': [1160, 1161, 8413, 8414, 8415, 8416, 8418, 8419, 8420, 42608, 42609, 42610],
				'encoded': '\u0488\u0489\u20DD\u20DE\u20DF\u20E0\u20E2\u20E3\u20E4\uA670\uA671\uA672'
			},
			{
				'description': 'Unicode character category [Mn]: mark, non-spacing',
				'decoded': [768, 769, 770, 771, 772, 773, 774, 775, 776, 777, 778, 779, 780, 781, 782, 783, 784, 785, 786, 787, 788, 789, 790, 791, 792, 793, 794, 795, 796, 797, 798, 799, 800, 801, 802, 803, 804, 805, 806, 807, 808, 809, 810, 811, 812, 813, 814, 815, 816, 817, 818, 819, 820, 821, 822, 823, 824, 825, 826, 827, 828, 829, 830, 831, 832, 833, 834, 835, 836, 837, 838, 839, 840, 841, 842, 843, 844, 845, 846, 847, 848, 849, 850, 851, 852, 853, 854, 855, 856, 857, 858, 859, 860, 861, 862, 863, 864, 865, 866, 867, 868, 869, 870, 871, 872, 873, 874, 875, 876, 877, 878, 879, 1155, 1156, 1157, 1158, 1159, 1425, 1426, 1427, 1428, 1429, 1430, 1431, 1432, 1433, 1434, 1435, 1436, 1437, 1438, 1439, 1440, 1441, 1442, 1443, 1444, 1445, 1446, 1447, 1448, 1449, 1450, 1451, 1452, 1453, 1454, 1455, 1456, 1457, 1458, 1459, 1460, 1461, 1462, 1463, 1464, 1465, 1466, 1467, 1468, 1469, 1471, 1473, 1474, 1476, 1477, 1479, 1552, 1553, 1554, 1555, 1556, 1557, 1558, 1559, 1560, 1561, 1562, 1611, 1612, 1613, 1614, 1615, 1616, 1617, 1618, 1619, 1620, 1621, 1622, 1623, 1624, 1625, 1626, 1627, 1628, 1629, 1630, 1631, 1648, 1750, 1751, 1752, 1753, 1754, 1755, 1756, 1759, 1760, 1761, 1762, 1763, 1764, 1767, 1768, 1770, 1771, 1772, 1773, 1809, 1840, 1841, 1842, 1843, 1844, 1845, 1846, 1847, 1848, 1849, 1850, 1851, 1852, 1853, 1854, 1855, 1856, 1857, 1858, 1859, 1860, 1861, 1862, 1863, 1864, 1865, 1866, 1958, 1959, 1960, 1961, 1962, 1963, 1964, 1965, 1966, 1967, 1968, 2027, 2028, 2029, 2030, 2031, 2032, 2033, 2034, 2035, 2070, 2071, 2072, 2073, 2075, 2076, 2077, 2078, 2079, 2080, 2081, 2082, 2083, 2085, 2086, 2087, 2089, 2090, 2091, 2092, 2093, 2137, 2138, 2139, 2304, 2305, 2306, 2362, 2364, 2369, 2370, 2371, 2372, 2373, 2374, 2375, 2376, 2381, 2385, 2386, 2387, 2388, 2389, 2390, 2391, 2402, 2403, 2433, 2492, 2497, 2498, 2499, 2500, 2509, 2530, 2531, 2561, 2562, 2620, 2625, 2626, 2631, 2632, 2635, 2636, 2637, 2641, 2672, 2673, 2677, 2689, 2690, 2748, 2753, 2754, 2755, 2756, 2757, 2759, 2760, 2765, 2786, 2787, 2817, 2876, 2879, 2881, 2882, 2883, 2884, 2893, 2902, 2914, 2915, 2946, 3008, 3021, 3134, 3135, 3136, 3142, 3143, 3144, 3146, 3147, 3148, 3149, 3157, 3158, 3170, 3171, 3260, 3263, 3270, 3276, 3277, 3298, 3299, 3393, 3394, 3395, 3396, 3405, 3426, 3427, 3530, 3538, 3539, 3540, 3542, 3633, 3636, 3637, 3638, 3639, 3640, 3641, 3642, 3655, 3656, 3657, 3658, 3659, 3660, 3661, 3662, 3761, 3764, 3765, 3766, 3767, 3768, 3769, 3771, 3772, 3784, 3785, 3786, 3787, 3788, 3789, 3864, 3865, 3893, 3895, 3897, 3953, 3954, 3955, 3956, 3957, 3958, 3959, 3960, 3961, 3962, 3963, 3964, 3965, 3966, 3968, 3969, 3970, 3971, 3972, 3974, 3975, 3981, 3982, 3983, 3984, 3985, 3986, 3987, 3988, 3989, 3990, 3991, 3993, 3994, 3995, 3996, 3997, 3998, 3999, 4000, 4001, 4002, 4003, 4004, 4005, 4006, 4007, 4008, 4009, 4010, 4011, 4012, 4013, 4014, 4015, 4016, 4017, 4018, 4019, 4020, 4021, 4022, 4023, 4024, 4025, 4026, 4027, 4028, 4038, 4141, 4142, 4143, 4144, 4146, 4147, 4148, 4149, 4150, 4151, 4153, 4154, 4157, 4158, 4184, 4185, 4190, 4191, 4192, 4209, 4210, 4211, 4212, 4226, 4229, 4230, 4237, 4253, 4957, 4958, 4959, 5906, 5907, 5908, 5938, 5939, 5940, 5970, 5971, 6002, 6003, 6071, 6072, 6073, 6074, 6075, 6076, 6077, 6086, 6089, 6090, 6091, 6092, 6093, 6094, 6095, 6096, 6097, 6098, 6099, 6109, 6155, 6156, 6157, 6313, 6432, 6433, 6434, 6439, 6440, 6450, 6457, 6458, 6459, 6679, 6680, 6742, 6744, 6745, 6746, 6747, 6748, 6749, 6750, 6752, 6754, 6757, 6758, 6759, 6760, 6761, 6762, 6763, 6764, 6771, 6772, 6773, 6774, 6775, 6776, 6777, 6778, 6779, 6780, 6783, 6912, 6913, 6914, 6915, 6964, 6966, 6967, 6968, 6969, 6970, 6972, 6978, 7019, 7020, 7021, 7022, 7023, 7024, 7025, 7026, 7027, 7040, 7041, 7074, 7075, 7076, 7077, 7080, 7081, 7142, 7144, 7145, 7149, 7151, 7152, 7153, 7212, 7213, 7214, 7215, 7216, 7217, 7218, 7219, 7222, 7223, 7376, 7377, 7378, 7380, 7381, 7382, 7383, 7384, 7385, 7386, 7387, 7388, 7389, 7390, 7391, 7392, 7394, 7395, 7396, 7397, 7398, 7399, 7400, 7405, 7616, 7617, 7618, 7619, 7620, 7621, 7622, 7623, 7624, 7625, 7626, 7627, 7628, 7629, 7630, 7631, 7632, 7633, 7634, 7635, 7636, 7637, 7638, 7639, 7640, 7641, 7642, 7643, 7644, 7645, 7646, 7647, 7648, 7649, 7650, 7651, 7652, 7653, 7654, 7676, 7677, 7678, 7679, 8400, 8401, 8402, 8403, 8404, 8405, 8406, 8407, 8408, 8409, 8410, 8411, 8412, 8417, 8421, 8422, 8423, 8424, 8425, 8426, 8427, 8428, 8429, 8430, 8431, 8432, 11503, 11504, 11505, 11647, 11744, 11745, 11746, 11747, 11748, 11749, 11750, 11751, 11752, 11753, 11754, 11755, 11756, 11757, 11758, 11759, 11760, 11761, 11762, 11763, 11764, 11765, 11766, 11767, 11768, 11769, 11770, 11771, 11772, 11773, 11774, 11775, 12330, 12331, 12332, 12333, 12334, 12335, 12441, 12442, 42607, 42620, 42621, 42736, 42737, 43010, 43014, 43019, 43045, 43046, 43204, 43232, 43233, 43234, 43235, 43236, 43237, 43238, 43239, 43240, 43241, 43242, 43243, 43244, 43245, 43246, 43247, 43248, 43249, 43302, 43303, 43304, 43305, 43306, 43307, 43308, 43309, 43335, 43336, 43337, 43338, 43339, 43340, 43341, 43342, 43343, 43344, 43345, 43392, 43393, 43394, 43443, 43446, 43447, 43448, 43449, 43452, 43561, 43562, 43563, 43564, 43565, 43566, 43569, 43570, 43573, 43574, 43587, 43596, 43696, 43698, 43699, 43700, 43703, 43704, 43710, 43711, 43713, 44005, 44008, 44013, 64286, 65024, 65025, 65026, 65027, 65028, 65029, 65030, 65031, 65032, 65033, 65034, 65035, 65036, 65037, 65038, 65039, 65056, 65057, 65058, 65059, 65060, 65061, 65062, 66045, 68097, 68098, 68099, 68101, 68102, 68108, 68109, 68110, 68111, 68152, 68153, 68154, 68159, 69633, 69688, 69689, 69690, 69691, 69692, 69693, 69694, 69695, 69696, 69697, 69698, 69699, 69700, 69701, 69702, 69760, 69761, 69811, 69812, 69813, 69814, 69817, 69818, 119143, 119144, 119145, 119163, 119164, 119165, 119166, 119167, 119168, 119169, 119170, 119173, 119174, 119175, 119176, 119177, 119178, 119179, 119210, 119211, 119212, 119213, 119362, 119363, 119364, 917760, 917761, 917762, 917763, 917764, 917765, 917766, 917767, 917768, 917769, 917770, 917771, 917772, 917773, 917774, 917775, 917776, 917777, 917778, 917779, 917780, 917781, 917782, 917783, 917784, 917785, 917786, 917787, 917788, 917789, 917790, 917791, 917792, 917793, 917794, 917795, 917796, 917797, 917798, 917799, 917800, 917801, 917802, 917803, 917804, 917805, 917806, 917807, 917808, 917809, 917810, 917811, 917812, 917813, 917814, 917815, 917816, 917817, 917818, 917819, 917820, 917821, 917822, 917823, 917824, 917825, 917826, 917827, 917828, 917829, 917830, 917831, 917832, 917833, 917834, 917835, 917836, 917837, 917838, 917839, 917840, 917841, 917842, 917843, 917844, 917845, 917846, 917847, 917848, 917849, 917850, 917851, 917852, 917853, 917854, 917855, 917856, 917857, 917858, 917859, 917860, 917861, 917862, 917863, 917864, 917865, 917866, 917867, 917868, 917869, 917870, 917871, 917872, 917873, 917874, 917875, 917876, 917877, 917878, 917879, 917880, 917881, 917882, 917883, 917884, 917885, 917886, 917887, 917888, 917889, 917890, 917891, 917892, 917893, 917894, 917895, 917896, 917897, 917898, 917899, 917900, 917901, 917902, 917903, 917904, 917905, 917906, 917907, 917908, 917909, 917910, 917911, 917912, 917913, 917914, 917915, 917916, 917917, 917918, 917919, 917920, 917921, 917922, 917923, 917924, 917925, 917926, 917927, 917928, 917929, 917930, 917931, 917932, 917933, 917934, 917935, 917936, 917937, 917938, 917939, 917940, 917941, 917942, 917943, 917944, 917945, 917946, 917947, 917948, 917949, 917950, 917951, 917952, 917953, 917954, 917955, 917956, 917957, 917958, 917959, 917960, 917961, 917962, 917963, 917964, 917965, 917966, 917967, 917968, 917969, 917970, 917971, 917972, 917973, 917974, 917975, 917976, 917977, 917978, 917979, 917980, 917981, 917982, 917983, 917984, 917985, 917986, 917987, 917988, 917989, 917990, 917991, 917992, 917993, 917994, 917995, 917996, 917997, 917998, 917999],
				'encoded': '\u0300\u0301\u0302\u0303\u0304\u0305\u0306\u0307\u0308\u0309\u030A\u030B\u030C\u030D\u030E\u030F\u0310\u0311\u0312\u0313\u0314\u0315\u0316\u0317\u0318\u0319\u031A\u031B\u031C\u031D\u031E\u031F\u0320\u0321\u0322\u0323\u0324\u0325\u0326\u0327\u0328\u0329\u032A\u032B\u032C\u032D\u032E\u032F\u0330\u0331\u0332\u0333\u0334\u0335\u0336\u0337\u0338\u0339\u033A\u033B\u033C\u033D\u033E\u033F\u0340\u0341\u0342\u0343\u0344\u0345\u0346\u0347\u0348\u0349\u034A\u034B\u034C\u034D\u034E\u034F\u0350\u0351\u0352\u0353\u0354\u0355\u0356\u0357\u0358\u0359\u035A\u035B\u035C\u035D\u035E\u035F\u0360\u0361\u0362\u0363\u0364\u0365\u0366\u0367\u0368\u0369\u036A\u036B\u036C\u036D\u036E\u036F\u0483\u0484\u0485\u0486\u0487\u0591\u0592\u0593\u0594\u0595\u0596\u0597\u0598\u0599\u059A\u059B\u059C\u059D\u059E\u059F\u05A0\u05A1\u05A2\u05A3\u05A4\u05A5\u05A6\u05A7\u05A8\u05A9\u05AA\u05AB\u05AC\u05AD\u05AE\u05AF\u05B0\u05B1\u05B2\u05B3\u05B4\u05B5\u05B6\u05B7\u05B8\u05B9\u05BA\u05BB\u05BC\u05BD\u05BF\u05C1\u05C2\u05C4\u05C5\u05C7\u0610\u0611\u0612\u0613\u0614\u0615\u0616\u0617\u0618\u0619\u061A\u064B\u064C\u064D\u064E\u064F\u0650\u0651\u0652\u0653\u0654\u0655\u0656\u0657\u0658\u0659\u065A\u065B\u065C\u065D\u065E\u065F\u0670\u06D6\u06D7\u06D8\u06D9\u06DA\u06DB\u06DC\u06DF\u06E0\u06E1\u06E2\u06E3\u06E4\u06E7\u06E8\u06EA\u06EB\u06EC\u06ED\u0711\u0730\u0731\u0732\u0733\u0734\u0735\u0736\u0737\u0738\u0739\u073A\u073B\u073C\u073D\u073E\u073F\u0740\u0741\u0742\u0743\u0744\u0745\u0746\u0747\u0748\u0749\u074A\u07A6\u07A7\u07A8\u07A9\u07AA\u07AB\u07AC\u07AD\u07AE\u07AF\u07B0\u07EB\u07EC\u07ED\u07EE\u07EF\u07F0\u07F1\u07F2\u07F3\u0816\u0817\u0818\u0819\u081B\u081C\u081D\u081E\u081F\u0820\u0821\u0822\u0823\u0825\u0826\u0827\u0829\u082A\u082B\u082C\u082D\u0859\u085A\u085B\u0900\u0901\u0902\u093A\u093C\u0941\u0942\u0943\u0944\u0945\u0946\u0947\u0948\u094D\u0951\u0952\u0953\u0954\u0955\u0956\u0957\u0962\u0963\u0981\u09BC\u09C1\u09C2\u09C3\u09C4\u09CD\u09E2\u09E3\u0A01\u0A02\u0A3C\u0A41\u0A42\u0A47\u0A48\u0A4B\u0A4C\u0A4D\u0A51\u0A70\u0A71\u0A75\u0A81\u0A82\u0ABC\u0AC1\u0AC2\u0AC3\u0AC4\u0AC5\u0AC7\u0AC8\u0ACD\u0AE2\u0AE3\u0B01\u0B3C\u0B3F\u0B41\u0B42\u0B43\u0B44\u0B4D\u0B56\u0B62\u0B63\u0B82\u0BC0\u0BCD\u0C3E\u0C3F\u0C40\u0C46\u0C47\u0C48\u0C4A\u0C4B\u0C4C\u0C4D\u0C55\u0C56\u0C62\u0C63\u0CBC\u0CBF\u0CC6\u0CCC\u0CCD\u0CE2\u0CE3\u0D41\u0D42\u0D43\u0D44\u0D4D\u0D62\u0D63\u0DCA\u0DD2\u0DD3\u0DD4\u0DD6\u0E31\u0E34\u0E35\u0E36\u0E37\u0E38\u0E39\u0E3A\u0E47\u0E48\u0E49\u0E4A\u0E4B\u0E4C\u0E4D\u0E4E\u0EB1\u0EB4\u0EB5\u0EB6\u0EB7\u0EB8\u0EB9\u0EBB\u0EBC\u0EC8\u0EC9\u0ECA\u0ECB\u0ECC\u0ECD\u0F18\u0F19\u0F35\u0F37\u0F39\u0F71\u0F72\u0F73\u0F74\u0F75\u0F76\u0F77\u0F78\u0F79\u0F7A\u0F7B\u0F7C\u0F7D\u0F7E\u0F80\u0F81\u0F82\u0F83\u0F84\u0F86\u0F87\u0F8D\u0F8E\u0F8F\u0F90\u0F91\u0F92\u0F93\u0F94\u0F95\u0F96\u0F97\u0F99\u0F9A\u0F9B\u0F9C\u0F9D\u0F9E\u0F9F\u0FA0\u0FA1\u0FA2\u0FA3\u0FA4\u0FA5\u0FA6\u0FA7\u0FA8\u0FA9\u0FAA\u0FAB\u0FAC\u0FAD\u0FAE\u0FAF\u0FB0\u0FB1\u0FB2\u0FB3\u0FB4\u0FB5\u0FB6\u0FB7\u0FB8\u0FB9\u0FBA\u0FBB\u0FBC\u0FC6\u102D\u102E\u102F\u1030\u1032\u1033\u1034\u1035\u1036\u1037\u1039\u103A\u103D\u103E\u1058\u1059\u105E\u105F\u1060\u1071\u1072\u1073\u1074\u1082\u1085\u1086\u108D\u109D\u135D\u135E\u135F\u1712\u1713\u1714\u1732\u1733\u1734\u1752\u1753\u1772\u1773\u17B7\u17B8\u17B9\u17BA\u17BB\u17BC\u17BD\u17C6\u17C9\u17CA\u17CB\u17CC\u17CD\u17CE\u17CF\u17D0\u17D1\u17D2\u17D3\u17DD\u180B\u180C\u180D\u18A9\u1920\u1921\u1922\u1927\u1928\u1932\u1939\u193A\u193B\u1A17\u1A18\u1A56\u1A58\u1A59\u1A5A\u1A5B\u1A5C\u1A5D\u1A5E\u1A60\u1A62\u1A65\u1A66\u1A67\u1A68\u1A69\u1A6A\u1A6B\u1A6C\u1A73\u1A74\u1A75\u1A76\u1A77\u1A78\u1A79\u1A7A\u1A7B\u1A7C\u1A7F\u1B00\u1B01\u1B02\u1B03\u1B34\u1B36\u1B37\u1B38\u1B39\u1B3A\u1B3C\u1B42\u1B6B\u1B6C\u1B6D\u1B6E\u1B6F\u1B70\u1B71\u1B72\u1B73\u1B80\u1B81\u1BA2\u1BA3\u1BA4\u1BA5\u1BA8\u1BA9\u1BE6\u1BE8\u1BE9\u1BED\u1BEF\u1BF0\u1BF1\u1C2C\u1C2D\u1C2E\u1C2F\u1C30\u1C31\u1C32\u1C33\u1C36\u1C37\u1CD0\u1CD1\u1CD2\u1CD4\u1CD5\u1CD6\u1CD7\u1CD8\u1CD9\u1CDA\u1CDB\u1CDC\u1CDD\u1CDE\u1CDF\u1CE0\u1CE2\u1CE3\u1CE4\u1CE5\u1CE6\u1CE7\u1CE8\u1CED\u1DC0\u1DC1\u1DC2\u1DC3\u1DC4\u1DC5\u1DC6\u1DC7\u1DC8\u1DC9\u1DCA\u1DCB\u1DCC\u1DCD\u1DCE\u1DCF\u1DD0\u1DD1\u1DD2\u1DD3\u1DD4\u1DD5\u1DD6\u1DD7\u1DD8\u1DD9\u1DDA\u1DDB\u1DDC\u1DDD\u1DDE\u1DDF\u1DE0\u1DE1\u1DE2\u1DE3\u1DE4\u1DE5\u1DE6\u1DFC\u1DFD\u1DFE\u1DFF\u20D0\u20D1\u20D2\u20D3\u20D4\u20D5\u20D6\u20D7\u20D8\u20D9\u20DA\u20DB\u20DC\u20E1\u20E5\u20E6\u20E7\u20E8\u20E9\u20EA\u20EB\u20EC\u20ED\u20EE\u20EF\u20F0\u2CEF\u2CF0\u2CF1\u2D7F\u2DE0\u2DE1\u2DE2\u2DE3\u2DE4\u2DE5\u2DE6\u2DE7\u2DE8\u2DE9\u2DEA\u2DEB\u2DEC\u2DED\u2DEE\u2DEF\u2DF0\u2DF1\u2DF2\u2DF3\u2DF4\u2DF5\u2DF6\u2DF7\u2DF8\u2DF9\u2DFA\u2DFB\u2DFC\u2DFD\u2DFE\u2DFF\u302A\u302B\u302C\u302D\u302E\u302F\u3099\u309A\uA66F\uA67C\uA67D\uA6F0\uA6F1\uA802\uA806\uA80B\uA825\uA826\uA8C4\uA8E0\uA8E1\uA8E2\uA8E3\uA8E4\uA8E5\uA8E6\uA8E7\uA8E8\uA8E9\uA8EA\uA8EB\uA8EC\uA8ED\uA8EE\uA8EF\uA8F0\uA8F1\uA926\uA927\uA928\uA929\uA92A\uA92B\uA92C\uA92D\uA947\uA948\uA949\uA94A\uA94B\uA94C\uA94D\uA94E\uA94F\uA950\uA951\uA980\uA981\uA982\uA9B3\uA9B6\uA9B7\uA9B8\uA9B9\uA9BC\uAA29\uAA2A\uAA2B\uAA2C\uAA2D\uAA2E\uAA31\uAA32\uAA35\uAA36\uAA43\uAA4C\uAAB0\uAAB2\uAAB3\uAAB4\uAAB7\uAAB8\uAABE\uAABF\uAAC1\uABE5\uABE8\uABED\uFB1E\uFE00\uFE01\uFE02\uFE03\uFE04\uFE05\uFE06\uFE07\uFE08\uFE09\uFE0A\uFE0B\uFE0C\uFE0D\uFE0E\uFE0F\uFE20\uFE21\uFE22\uFE23\uFE24\uFE25\uFE26\uD800\uDDFD\uD802\uDE01\uD802\uDE02\uD802\uDE03\uD802\uDE05\uD802\uDE06\uD802\uDE0C\uD802\uDE0D\uD802\uDE0E\uD802\uDE0F\uD802\uDE38\uD802\uDE39\uD802\uDE3A\uD802\uDE3F\uD804\uDC01\uD804\uDC38\uD804\uDC39\uD804\uDC3A\uD804\uDC3B\uD804\uDC3C\uD804\uDC3D\uD804\uDC3E\uD804\uDC3F\uD804\uDC40\uD804\uDC41\uD804\uDC42\uD804\uDC43\uD804\uDC44\uD804\uDC45\uD804\uDC46\uD804\uDC80\uD804\uDC81\uD804\uDCB3\uD804\uDCB4\uD804\uDCB5\uD804\uDCB6\uD804\uDCB9\uD804\uDCBA\uD834\uDD67\uD834\uDD68\uD834\uDD69\uD834\uDD7B\uD834\uDD7C\uD834\uDD7D\uD834\uDD7E\uD834\uDD7F\uD834\uDD80\uD834\uDD81\uD834\uDD82\uD834\uDD85\uD834\uDD86\uD834\uDD87\uD834\uDD88\uD834\uDD89\uD834\uDD8A\uD834\uDD8B\uD834\uDDAA\uD834\uDDAB\uD834\uDDAC\uD834\uDDAD\uD834\uDE42\uD834\uDE43\uD834\uDE44\uDB40\uDD00\uDB40\uDD01\uDB40\uDD02\uDB40\uDD03\uDB40\uDD04\uDB40\uDD05\uDB40\uDD06\uDB40\uDD07\uDB40\uDD08\uDB40\uDD09\uDB40\uDD0A\uDB40\uDD0B\uDB40\uDD0C\uDB40\uDD0D\uDB40\uDD0E\uDB40\uDD0F\uDB40\uDD10\uDB40\uDD11\uDB40\uDD12\uDB40\uDD13\uDB40\uDD14\uDB40\uDD15\uDB40\uDD16\uDB40\uDD17\uDB40\uDD18\uDB40\uDD19\uDB40\uDD1A\uDB40\uDD1B\uDB40\uDD1C\uDB40\uDD1D\uDB40\uDD1E\uDB40\uDD1F\uDB40\uDD20\uDB40\uDD21\uDB40\uDD22\uDB40\uDD23\uDB40\uDD24\uDB40\uDD25\uDB40\uDD26\uDB40\uDD27\uDB40\uDD28\uDB40\uDD29\uDB40\uDD2A\uDB40\uDD2B\uDB40\uDD2C\uDB40\uDD2D\uDB40\uDD2E\uDB40\uDD2F\uDB40\uDD30\uDB40\uDD31\uDB40\uDD32\uDB40\uDD33\uDB40\uDD34\uDB40\uDD35\uDB40\uDD36\uDB40\uDD37\uDB40\uDD38\uDB40\uDD39\uDB40\uDD3A\uDB40\uDD3B\uDB40\uDD3C\uDB40\uDD3D\uDB40\uDD3E\uDB40\uDD3F\uDB40\uDD40\uDB40\uDD41\uDB40\uDD42\uDB40\uDD43\uDB40\uDD44\uDB40\uDD45\uDB40\uDD46\uDB40\uDD47\uDB40\uDD48\uDB40\uDD49\uDB40\uDD4A\uDB40\uDD4B\uDB40\uDD4C\uDB40\uDD4D\uDB40\uDD4E\uDB40\uDD4F\uDB40\uDD50\uDB40\uDD51\uDB40\uDD52\uDB40\uDD53\uDB40\uDD54\uDB40\uDD55\uDB40\uDD56\uDB40\uDD57\uDB40\uDD58\uDB40\uDD59\uDB40\uDD5A\uDB40\uDD5B\uDB40\uDD5C\uDB40\uDD5D\uDB40\uDD5E\uDB40\uDD5F\uDB40\uDD60\uDB40\uDD61\uDB40\uDD62\uDB40\uDD63\uDB40\uDD64\uDB40\uDD65\uDB40\uDD66\uDB40\uDD67\uDB40\uDD68\uDB40\uDD69\uDB40\uDD6A\uDB40\uDD6B\uDB40\uDD6C\uDB40\uDD6D\uDB40\uDD6E\uDB40\uDD6F\uDB40\uDD70\uDB40\uDD71\uDB40\uDD72\uDB40\uDD73\uDB40\uDD74\uDB40\uDD75\uDB40\uDD76\uDB40\uDD77\uDB40\uDD78\uDB40\uDD79\uDB40\uDD7A\uDB40\uDD7B\uDB40\uDD7C\uDB40\uDD7D\uDB40\uDD7E\uDB40\uDD7F\uDB40\uDD80\uDB40\uDD81\uDB40\uDD82\uDB40\uDD83\uDB40\uDD84\uDB40\uDD85\uDB40\uDD86\uDB40\uDD87\uDB40\uDD88\uDB40\uDD89\uDB40\uDD8A\uDB40\uDD8B\uDB40\uDD8C\uDB40\uDD8D\uDB40\uDD8E\uDB40\uDD8F\uDB40\uDD90\uDB40\uDD91\uDB40\uDD92\uDB40\uDD93\uDB40\uDD94\uDB40\uDD95\uDB40\uDD96\uDB40\uDD97\uDB40\uDD98\uDB40\uDD99\uDB40\uDD9A\uDB40\uDD9B\uDB40\uDD9C\uDB40\uDD9D\uDB40\uDD9E\uDB40\uDD9F\uDB40\uDDA0\uDB40\uDDA1\uDB40\uDDA2\uDB40\uDDA3\uDB40\uDDA4\uDB40\uDDA5\uDB40\uDDA6\uDB40\uDDA7\uDB40\uDDA8\uDB40\uDDA9\uDB40\uDDAA\uDB40\uDDAB\uDB40\uDDAC\uDB40\uDDAD\uDB40\uDDAE\uDB40\uDDAF\uDB40\uDDB0\uDB40\uDDB1\uDB40\uDDB2\uDB40\uDDB3\uDB40\uDDB4\uDB40\uDDB5\uDB40\uDDB6\uDB40\uDDB7\uDB40\uDDB8\uDB40\uDDB9\uDB40\uDDBA\uDB40\uDDBB\uDB40\uDDBC\uDB40\uDDBD\uDB40\uDDBE\uDB40\uDDBF\uDB40\uDDC0\uDB40\uDDC1\uDB40\uDDC2\uDB40\uDDC3\uDB40\uDDC4\uDB40\uDDC5\uDB40\uDDC6\uDB40\uDDC7\uDB40\uDDC8\uDB40\uDDC9\uDB40\uDDCA\uDB40\uDDCB\uDB40\uDDCC\uDB40\uDDCD\uDB40\uDDCE\uDB40\uDDCF\uDB40\uDDD0\uDB40\uDDD1\uDB40\uDDD2\uDB40\uDDD3\uDB40\uDDD4\uDB40\uDDD5\uDB40\uDDD6\uDB40\uDDD7\uDB40\uDDD8\uDB40\uDDD9\uDB40\uDDDA\uDB40\uDDDB\uDB40\uDDDC\uDB40\uDDDD\uDB40\uDDDE\uDB40\uDDDF\uDB40\uDDE0\uDB40\uDDE1\uDB40\uDDE2\uDB40\uDDE3\uDB40\uDDE4\uDB40\uDDE5\uDB40\uDDE6\uDB40\uDDE7\uDB40\uDDE8\uDB40\uDDE9\uDB40\uDDEA\uDB40\uDDEB\uDB40\uDDEC\uDB40\uDDED\uDB40\uDDEE\uDB40\uDDEF'
			},
			{
				'description': 'Unicode character category [Nd]: number, decimal digit',
				'decoded': [48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 1632, 1633, 1634, 1635, 1636, 1637, 1638, 1639, 1640, 1641, 1776, 1777, 1778, 1779, 1780, 1781, 1782, 1783, 1784, 1785, 1984, 1985, 1986, 1987, 1988, 1989, 1990, 1991, 1992, 1993, 2406, 2407, 2408, 2409, 2410, 2411, 2412, 2413, 2414, 2415, 2534, 2535, 2536, 2537, 2538, 2539, 2540, 2541, 2542, 2543, 2662, 2663, 2664, 2665, 2666, 2667, 2668, 2669, 2670, 2671, 2790, 2791, 2792, 2793, 2794, 2795, 2796, 2797, 2798, 2799, 2918, 2919, 2920, 2921, 2922, 2923, 2924, 2925, 2926, 2927, 3046, 3047, 3048, 3049, 3050, 3051, 3052, 3053, 3054, 3055, 3174, 3175, 3176, 3177, 3178, 3179, 3180, 3181, 3182, 3183, 3302, 3303, 3304, 3305, 3306, 3307, 3308, 3309, 3310, 3311, 3430, 3431, 3432, 3433, 3434, 3435, 3436, 3437, 3438, 3439, 3664, 3665, 3666, 3667, 3668, 3669, 3670, 3671, 3672, 3673, 3792, 3793, 3794, 3795, 3796, 3797, 3798, 3799, 3800, 3801, 3872, 3873, 3874, 3875, 3876, 3877, 3878, 3879, 3880, 3881, 4160, 4161, 4162, 4163, 4164, 4165, 4166, 4167, 4168, 4169, 4240, 4241, 4242, 4243, 4244, 4245, 4246, 4247, 4248, 4249, 6112, 6113, 6114, 6115, 6116, 6117, 6118, 6119, 6120, 6121, 6160, 6161, 6162, 6163, 6164, 6165, 6166, 6167, 6168, 6169, 6470, 6471, 6472, 6473, 6474, 6475, 6476, 6477, 6478, 6479, 6608, 6609, 6610, 6611, 6612, 6613, 6614, 6615, 6616, 6617, 6784, 6785, 6786, 6787, 6788, 6789, 6790, 6791, 6792, 6793, 6800, 6801, 6802, 6803, 6804, 6805, 6806, 6807, 6808, 6809, 6992, 6993, 6994, 6995, 6996, 6997, 6998, 6999, 7000, 7001, 7088, 7089, 7090, 7091, 7092, 7093, 7094, 7095, 7096, 7097, 7232, 7233, 7234, 7235, 7236, 7237, 7238, 7239, 7240, 7241, 7248, 7249, 7250, 7251, 7252, 7253, 7254, 7255, 7256, 7257, 42528, 42529, 42530, 42531, 42532, 42533, 42534, 42535, 42536, 42537, 43216, 43217, 43218, 43219, 43220, 43221, 43222, 43223, 43224, 43225, 43264, 43265, 43266, 43267, 43268, 43269, 43270, 43271, 43272, 43273, 43472, 43473, 43474, 43475, 43476, 43477, 43478, 43479, 43480, 43481, 43600, 43601, 43602, 43603, 43604, 43605, 43606, 43607, 43608, 43609, 44016, 44017, 44018, 44019, 44020, 44021, 44022, 44023, 44024, 44025, 65296, 65297, 65298, 65299, 65300, 65301, 65302, 65303, 65304, 65305, 66720, 66721, 66722, 66723, 66724, 66725, 66726, 66727, 66728, 66729, 69734, 69735, 69736, 69737, 69738, 69739, 69740, 69741, 69742, 69743, 120782, 120783, 120784, 120785, 120786, 120787, 120788, 120789, 120790, 120791, 120792, 120793, 120794, 120795, 120796, 120797, 120798, 120799, 120800, 120801, 120802, 120803, 120804, 120805, 120806, 120807, 120808, 120809, 120810, 120811, 120812, 120813, 120814, 120815, 120816, 120817, 120818, 120819, 120820, 120821, 120822, 120823, 120824, 120825, 120826, 120827, 120828, 120829, 120830, 120831],
				'encoded': '0123456789\u0660\u0661\u0662\u0663\u0664\u0665\u0666\u0667\u0668\u0669\u06F0\u06F1\u06F2\u06F3\u06F4\u06F5\u06F6\u06F7\u06F8\u06F9\u07C0\u07C1\u07C2\u07C3\u07C4\u07C5\u07C6\u07C7\u07C8\u07C9\u0966\u0967\u0968\u0969\u096A\u096B\u096C\u096D\u096E\u096F\u09E6\u09E7\u09E8\u09E9\u09EA\u09EB\u09EC\u09ED\u09EE\u09EF\u0A66\u0A67\u0A68\u0A69\u0A6A\u0A6B\u0A6C\u0A6D\u0A6E\u0A6F\u0AE6\u0AE7\u0AE8\u0AE9\u0AEA\u0AEB\u0AEC\u0AED\u0AEE\u0AEF\u0B66\u0B67\u0B68\u0B69\u0B6A\u0B6B\u0B6C\u0B6D\u0B6E\u0B6F\u0BE6\u0BE7\u0BE8\u0BE9\u0BEA\u0BEB\u0BEC\u0BED\u0BEE\u0BEF\u0C66\u0C67\u0C68\u0C69\u0C6A\u0C6B\u0C6C\u0C6D\u0C6E\u0C6F\u0CE6\u0CE7\u0CE8\u0CE9\u0CEA\u0CEB\u0CEC\u0CED\u0CEE\u0CEF\u0D66\u0D67\u0D68\u0D69\u0D6A\u0D6B\u0D6C\u0D6D\u0D6E\u0D6F\u0E50\u0E51\u0E52\u0E53\u0E54\u0E55\u0E56\u0E57\u0E58\u0E59\u0ED0\u0ED1\u0ED2\u0ED3\u0ED4\u0ED5\u0ED6\u0ED7\u0ED8\u0ED9\u0F20\u0F21\u0F22\u0F23\u0F24\u0F25\u0F26\u0F27\u0F28\u0F29\u1040\u1041\u1042\u1043\u1044\u1045\u1046\u1047\u1048\u1049\u1090\u1091\u1092\u1093\u1094\u1095\u1096\u1097\u1098\u1099\u17E0\u17E1\u17E2\u17E3\u17E4\u17E5\u17E6\u17E7\u17E8\u17E9\u1810\u1811\u1812\u1813\u1814\u1815\u1816\u1817\u1818\u1819\u1946\u1947\u1948\u1949\u194A\u194B\u194C\u194D\u194E\u194F\u19D0\u19D1\u19D2\u19D3\u19D4\u19D5\u19D6\u19D7\u19D8\u19D9\u1A80\u1A81\u1A82\u1A83\u1A84\u1A85\u1A86\u1A87\u1A88\u1A89\u1A90\u1A91\u1A92\u1A93\u1A94\u1A95\u1A96\u1A97\u1A98\u1A99\u1B50\u1B51\u1B52\u1B53\u1B54\u1B55\u1B56\u1B57\u1B58\u1B59\u1BB0\u1BB1\u1BB2\u1BB3\u1BB4\u1BB5\u1BB6\u1BB7\u1BB8\u1BB9\u1C40\u1C41\u1C42\u1C43\u1C44\u1C45\u1C46\u1C47\u1C48\u1C49\u1C50\u1C51\u1C52\u1C53\u1C54\u1C55\u1C56\u1C57\u1C58\u1C59\uA620\uA621\uA622\uA623\uA624\uA625\uA626\uA627\uA628\uA629\uA8D0\uA8D1\uA8D2\uA8D3\uA8D4\uA8D5\uA8D6\uA8D7\uA8D8\uA8D9\uA900\uA901\uA902\uA903\uA904\uA905\uA906\uA907\uA908\uA909\uA9D0\uA9D1\uA9D2\uA9D3\uA9D4\uA9D5\uA9D6\uA9D7\uA9D8\uA9D9\uAA50\uAA51\uAA52\uAA53\uAA54\uAA55\uAA56\uAA57\uAA58\uAA59\uABF0\uABF1\uABF2\uABF3\uABF4\uABF5\uABF6\uABF7\uABF8\uABF9\uFF10\uFF11\uFF12\uFF13\uFF14\uFF15\uFF16\uFF17\uFF18\uFF19\uD801\uDCA0\uD801\uDCA1\uD801\uDCA2\uD801\uDCA3\uD801\uDCA4\uD801\uDCA5\uD801\uDCA6\uD801\uDCA7\uD801\uDCA8\uD801\uDCA9\uD804\uDC66\uD804\uDC67\uD804\uDC68\uD804\uDC69\uD804\uDC6A\uD804\uDC6B\uD804\uDC6C\uD804\uDC6D\uD804\uDC6E\uD804\uDC6F\uD835\uDFCE\uD835\uDFCF\uD835\uDFD0\uD835\uDFD1\uD835\uDFD2\uD835\uDFD3\uD835\uDFD4\uD835\uDFD5\uD835\uDFD6\uD835\uDFD7\uD835\uDFD8\uD835\uDFD9\uD835\uDFDA\uD835\uDFDB\uD835\uDFDC\uD835\uDFDD\uD835\uDFDE\uD835\uDFDF\uD835\uDFE0\uD835\uDFE1\uD835\uDFE2\uD835\uDFE3\uD835\uDFE4\uD835\uDFE5\uD835\uDFE6\uD835\uDFE7\uD835\uDFE8\uD835\uDFE9\uD835\uDFEA\uD835\uDFEB\uD835\uDFEC\uD835\uDFED\uD835\uDFEE\uD835\uDFEF\uD835\uDFF0\uD835\uDFF1\uD835\uDFF2\uD835\uDFF3\uD835\uDFF4\uD835\uDFF5\uD835\uDFF6\uD835\uDFF7\uD835\uDFF8\uD835\uDFF9\uD835\uDFFA\uD835\uDFFB\uD835\uDFFC\uD835\uDFFD\uD835\uDFFE\uD835\uDFFF'
			},
			{
				'description': 'Unicode character category [Nl]: number, letter',
				'decoded': [5870, 5871, 5872, 8544, 8545, 8546, 8547, 8548, 8549, 8550, 8551, 8552, 8553, 8554, 8555, 8556, 8557, 8558, 8559, 8560, 8561, 8562, 8563, 8564, 8565, 8566, 8567, 8568, 8569, 8570, 8571, 8572, 8573, 8574, 8575, 8576, 8577, 8578, 8581, 8582, 8583, 8584, 12295, 12321, 12322, 12323, 12324, 12325, 12326, 12327, 12328, 12329, 12344, 12345, 12346, 42726, 42727, 42728, 42729, 42730, 42731, 42732, 42733, 42734, 42735, 65856, 65857, 65858, 65859, 65860, 65861, 65862, 65863, 65864, 65865, 65866, 65867, 65868, 65869, 65870, 65871, 65872, 65873, 65874, 65875, 65876, 65877, 65878, 65879, 65880, 65881, 65882, 65883, 65884, 65885, 65886, 65887, 65888, 65889, 65890, 65891, 65892, 65893, 65894, 65895, 65896, 65897, 65898, 65899, 65900, 65901, 65902, 65903, 65904, 65905, 65906, 65907, 65908, 66369, 66378, 66513, 66514, 66515, 66516, 66517, 74752, 74753, 74754, 74755, 74756, 74757, 74758, 74759, 74760, 74761, 74762, 74763, 74764, 74765, 74766, 74767, 74768, 74769, 74770, 74771, 74772, 74773, 74774, 74775, 74776, 74777, 74778, 74779, 74780, 74781, 74782, 74783, 74784, 74785, 74786, 74787, 74788, 74789, 74790, 74791, 74792, 74793, 74794, 74795, 74796, 74797, 74798, 74799, 74800, 74801, 74802, 74803, 74804, 74805, 74806, 74807, 74808, 74809, 74810, 74811, 74812, 74813, 74814, 74815, 74816, 74817, 74818, 74819, 74820, 74821, 74822, 74823, 74824, 74825, 74826, 74827, 74828, 74829, 74830, 74831, 74832, 74833, 74834, 74835, 74836, 74837, 74838, 74839, 74840, 74841, 74842, 74843, 74844, 74845, 74846, 74847, 74848, 74849, 74850],
				'encoded': '\u16EE\u16EF\u16F0\u2160\u2161\u2162\u2163\u2164\u2165\u2166\u2167\u2168\u2169\u216A\u216B\u216C\u216D\u216E\u216F\u2170\u2171\u2172\u2173\u2174\u2175\u2176\u2177\u2178\u2179\u217A\u217B\u217C\u217D\u217E\u217F\u2180\u2181\u2182\u2185\u2186\u2187\u2188\u3007\u3021\u3022\u3023\u3024\u3025\u3026\u3027\u3028\u3029\u3038\u3039\u303A\uA6E6\uA6E7\uA6E8\uA6E9\uA6EA\uA6EB\uA6EC\uA6ED\uA6EE\uA6EF\uD800\uDD40\uD800\uDD41\uD800\uDD42\uD800\uDD43\uD800\uDD44\uD800\uDD45\uD800\uDD46\uD800\uDD47\uD800\uDD48\uD800\uDD49\uD800\uDD4A\uD800\uDD4B\uD800\uDD4C\uD800\uDD4D\uD800\uDD4E\uD800\uDD4F\uD800\uDD50\uD800\uDD51\uD800\uDD52\uD800\uDD53\uD800\uDD54\uD800\uDD55\uD800\uDD56\uD800\uDD57\uD800\uDD58\uD800\uDD59\uD800\uDD5A\uD800\uDD5B\uD800\uDD5C\uD800\uDD5D\uD800\uDD5E\uD800\uDD5F\uD800\uDD60\uD800\uDD61\uD800\uDD62\uD800\uDD63\uD800\uDD64\uD800\uDD65\uD800\uDD66\uD800\uDD67\uD800\uDD68\uD800\uDD69\uD800\uDD6A\uD800\uDD6B\uD800\uDD6C\uD800\uDD6D\uD800\uDD6E\uD800\uDD6F\uD800\uDD70\uD800\uDD71\uD800\uDD72\uD800\uDD73\uD800\uDD74\uD800\uDF41\uD800\uDF4A\uD800\uDFD1\uD800\uDFD2\uD800\uDFD3\uD800\uDFD4\uD800\uDFD5\uD809\uDC00\uD809\uDC01\uD809\uDC02\uD809\uDC03\uD809\uDC04\uD809\uDC05\uD809\uDC06\uD809\uDC07\uD809\uDC08\uD809\uDC09\uD809\uDC0A\uD809\uDC0B\uD809\uDC0C\uD809\uDC0D\uD809\uDC0E\uD809\uDC0F\uD809\uDC10\uD809\uDC11\uD809\uDC12\uD809\uDC13\uD809\uDC14\uD809\uDC15\uD809\uDC16\uD809\uDC17\uD809\uDC18\uD809\uDC19\uD809\uDC1A\uD809\uDC1B\uD809\uDC1C\uD809\uDC1D\uD809\uDC1E\uD809\uDC1F\uD809\uDC20\uD809\uDC21\uD809\uDC22\uD809\uDC23\uD809\uDC24\uD809\uDC25\uD809\uDC26\uD809\uDC27\uD809\uDC28\uD809\uDC29\uD809\uDC2A\uD809\uDC2B\uD809\uDC2C\uD809\uDC2D\uD809\uDC2E\uD809\uDC2F\uD809\uDC30\uD809\uDC31\uD809\uDC32\uD809\uDC33\uD809\uDC34\uD809\uDC35\uD809\uDC36\uD809\uDC37\uD809\uDC38\uD809\uDC39\uD809\uDC3A\uD809\uDC3B\uD809\uDC3C\uD809\uDC3D\uD809\uDC3E\uD809\uDC3F\uD809\uDC40\uD809\uDC41\uD809\uDC42\uD809\uDC43\uD809\uDC44\uD809\uDC45\uD809\uDC46\uD809\uDC47\uD809\uDC48\uD809\uDC49\uD809\uDC4A\uD809\uDC4B\uD809\uDC4C\uD809\uDC4D\uD809\uDC4E\uD809\uDC4F\uD809\uDC50\uD809\uDC51\uD809\uDC52\uD809\uDC53\uD809\uDC54\uD809\uDC55\uD809\uDC56\uD809\uDC57\uD809\uDC58\uD809\uDC59\uD809\uDC5A\uD809\uDC5B\uD809\uDC5C\uD809\uDC5D\uD809\uDC5E\uD809\uDC5F\uD809\uDC60\uD809\uDC61\uD809\uDC62'
			},
			{
				'description': 'Unicode character category [No]: number, other',
				'decoded': [178, 179, 185, 188, 189, 190, 2548, 2549, 2550, 2551, 2552, 2553, 2930, 2931, 2932, 2933, 2934, 2935, 3056, 3057, 3058, 3192, 3193, 3194, 3195, 3196, 3197, 3198, 3440, 3441, 3442, 3443, 3444, 3445, 3882, 3883, 3884, 3885, 3886, 3887, 3888, 3889, 3890, 3891, 4969, 4970, 4971, 4972, 4973, 4974, 4975, 4976, 4977, 4978, 4979, 4980, 4981, 4982, 4983, 4984, 4985, 4986, 4987, 4988, 6128, 6129, 6130, 6131, 6132, 6133, 6134, 6135, 6136, 6137, 6618, 8304, 8308, 8309, 8310, 8311, 8312, 8313, 8320, 8321, 8322, 8323, 8324, 8325, 8326, 8327, 8328, 8329, 8528, 8529, 8530, 8531, 8532, 8533, 8534, 8535, 8536, 8537, 8538, 8539, 8540, 8541, 8542, 8543, 8585, 9312, 9313, 9314, 9315, 9316, 9317, 9318, 9319, 9320, 9321, 9322, 9323, 9324, 9325, 9326, 9327, 9328, 9329, 9330, 9331, 9332, 9333, 9334, 9335, 9336, 9337, 9338, 9339, 9340, 9341, 9342, 9343, 9344, 9345, 9346, 9347, 9348, 9349, 9350, 9351, 9352, 9353, 9354, 9355, 9356, 9357, 9358, 9359, 9360, 9361, 9362, 9363, 9364, 9365, 9366, 9367, 9368, 9369, 9370, 9371, 9450, 9451, 9452, 9453, 9454, 9455, 9456, 9457, 9458, 9459, 9460, 9461, 9462, 9463, 9464, 9465, 9466, 9467, 9468, 9469, 9470, 9471, 10102, 10103, 10104, 10105, 10106, 10107, 10108, 10109, 10110, 10111, 10112, 10113, 10114, 10115, 10116, 10117, 10118, 10119, 10120, 10121, 10122, 10123, 10124, 10125, 10126, 10127, 10128, 10129, 10130, 10131, 11517, 12690, 12691, 12692, 12693, 12832, 12833, 12834, 12835, 12836, 12837, 12838, 12839, 12840, 12841, 12881, 12882, 12883, 12884, 12885, 12886, 12887, 12888, 12889, 12890, 12891, 12892, 12893, 12894, 12895, 12928, 12929, 12930, 12931, 12932, 12933, 12934, 12935, 12936, 12937, 12977, 12978, 12979, 12980, 12981, 12982, 12983, 12984, 12985, 12986, 12987, 12988, 12989, 12990, 12991, 43056, 43057, 43058, 43059, 43060, 43061, 65799, 65800, 65801, 65802, 65803, 65804, 65805, 65806, 65807, 65808, 65809, 65810, 65811, 65812, 65813, 65814, 65815, 65816, 65817, 65818, 65819, 65820, 65821, 65822, 65823, 65824, 65825, 65826, 65827, 65828, 65829, 65830, 65831, 65832, 65833, 65834, 65835, 65836, 65837, 65838, 65839, 65840, 65841, 65842, 65843, 65909, 65910, 65911, 65912, 65930, 66336, 66337, 66338, 66339, 67672, 67673, 67674, 67675, 67676, 67677, 67678, 67679, 67862, 67863, 67864, 67865, 67866, 67867, 68160, 68161, 68162, 68163, 68164, 68165, 68166, 68167, 68221, 68222, 68440, 68441, 68442, 68443, 68444, 68445, 68446, 68447, 68472, 68473, 68474, 68475, 68476, 68477, 68478, 68479, 69216, 69217, 69218, 69219, 69220, 69221, 69222, 69223, 69224, 69225, 69226, 69227, 69228, 69229, 69230, 69231, 69232, 69233, 69234, 69235, 69236, 69237, 69238, 69239, 69240, 69241, 69242, 69243, 69244, 69245, 69246, 69714, 69715, 69716, 69717, 69718, 69719, 69720, 69721, 69722, 69723, 69724, 69725, 69726, 69727, 69728, 69729, 69730, 69731, 69732, 69733, 119648, 119649, 119650, 119651, 119652, 119653, 119654, 119655, 119656, 119657, 119658, 119659, 119660, 119661, 119662, 119663, 119664, 119665, 127232, 127233, 127234, 127235, 127236, 127237, 127238, 127239, 127240, 127241, 127242],
				'encoded': '\xB2\xB3\xB9\xBC\xBD\xBE\u09F4\u09F5\u09F6\u09F7\u09F8\u09F9\u0B72\u0B73\u0B74\u0B75\u0B76\u0B77\u0BF0\u0BF1\u0BF2\u0C78\u0C79\u0C7A\u0C7B\u0C7C\u0C7D\u0C7E\u0D70\u0D71\u0D72\u0D73\u0D74\u0D75\u0F2A\u0F2B\u0F2C\u0F2D\u0F2E\u0F2F\u0F30\u0F31\u0F32\u0F33\u1369\u136A\u136B\u136C\u136D\u136E\u136F\u1370\u1371\u1372\u1373\u1374\u1375\u1376\u1377\u1378\u1379\u137A\u137B\u137C\u17F0\u17F1\u17F2\u17F3\u17F4\u17F5\u17F6\u17F7\u17F8\u17F9\u19DA\u2070\u2074\u2075\u2076\u2077\u2078\u2079\u2080\u2081\u2082\u2083\u2084\u2085\u2086\u2087\u2088\u2089\u2150\u2151\u2152\u2153\u2154\u2155\u2156\u2157\u2158\u2159\u215A\u215B\u215C\u215D\u215E\u215F\u2189\u2460\u2461\u2462\u2463\u2464\u2465\u2466\u2467\u2468\u2469\u246A\u246B\u246C\u246D\u246E\u246F\u2470\u2471\u2472\u2473\u2474\u2475\u2476\u2477\u2478\u2479\u247A\u247B\u247C\u247D\u247E\u247F\u2480\u2481\u2482\u2483\u2484\u2485\u2486\u2487\u2488\u2489\u248A\u248B\u248C\u248D\u248E\u248F\u2490\u2491\u2492\u2493\u2494\u2495\u2496\u2497\u2498\u2499\u249A\u249B\u24EA\u24EB\u24EC\u24ED\u24EE\u24EF\u24F0\u24F1\u24F2\u24F3\u24F4\u24F5\u24F6\u24F7\u24F8\u24F9\u24FA\u24FB\u24FC\u24FD\u24FE\u24FF\u2776\u2777\u2778\u2779\u277A\u277B\u277C\u277D\u277E\u277F\u2780\u2781\u2782\u2783\u2784\u2785\u2786\u2787\u2788\u2789\u278A\u278B\u278C\u278D\u278E\u278F\u2790\u2791\u2792\u2793\u2CFD\u3192\u3193\u3194\u3195\u3220\u3221\u3222\u3223\u3224\u3225\u3226\u3227\u3228\u3229\u3251\u3252\u3253\u3254\u3255\u3256\u3257\u3258\u3259\u325A\u325B\u325C\u325D\u325E\u325F\u3280\u3281\u3282\u3283\u3284\u3285\u3286\u3287\u3288\u3289\u32B1\u32B2\u32B3\u32B4\u32B5\u32B6\u32B7\u32B8\u32B9\u32BA\u32BB\u32BC\u32BD\u32BE\u32BF\uA830\uA831\uA832\uA833\uA834\uA835\uD800\uDD07\uD800\uDD08\uD800\uDD09\uD800\uDD0A\uD800\uDD0B\uD800\uDD0C\uD800\uDD0D\uD800\uDD0E\uD800\uDD0F\uD800\uDD10\uD800\uDD11\uD800\uDD12\uD800\uDD13\uD800\uDD14\uD800\uDD15\uD800\uDD16\uD800\uDD17\uD800\uDD18\uD800\uDD19\uD800\uDD1A\uD800\uDD1B\uD800\uDD1C\uD800\uDD1D\uD800\uDD1E\uD800\uDD1F\uD800\uDD20\uD800\uDD21\uD800\uDD22\uD800\uDD23\uD800\uDD24\uD800\uDD25\uD800\uDD26\uD800\uDD27\uD800\uDD28\uD800\uDD29\uD800\uDD2A\uD800\uDD2B\uD800\uDD2C\uD800\uDD2D\uD800\uDD2E\uD800\uDD2F\uD800\uDD30\uD800\uDD31\uD800\uDD32\uD800\uDD33\uD800\uDD75\uD800\uDD76\uD800\uDD77\uD800\uDD78\uD800\uDD8A\uD800\uDF20\uD800\uDF21\uD800\uDF22\uD800\uDF23\uD802\uDC58\uD802\uDC59\uD802\uDC5A\uD802\uDC5B\uD802\uDC5C\uD802\uDC5D\uD802\uDC5E\uD802\uDC5F\uD802\uDD16\uD802\uDD17\uD802\uDD18\uD802\uDD19\uD802\uDD1A\uD802\uDD1B\uD802\uDE40\uD802\uDE41\uD802\uDE42\uD802\uDE43\uD802\uDE44\uD802\uDE45\uD802\uDE46\uD802\uDE47\uD802\uDE7D\uD802\uDE7E\uD802\uDF58\uD802\uDF59\uD802\uDF5A\uD802\uDF5B\uD802\uDF5C\uD802\uDF5D\uD802\uDF5E\uD802\uDF5F\uD802\uDF78\uD802\uDF79\uD802\uDF7A\uD802\uDF7B\uD802\uDF7C\uD802\uDF7D\uD802\uDF7E\uD802\uDF7F\uD803\uDE60\uD803\uDE61\uD803\uDE62\uD803\uDE63\uD803\uDE64\uD803\uDE65\uD803\uDE66\uD803\uDE67\uD803\uDE68\uD803\uDE69\uD803\uDE6A\uD803\uDE6B\uD803\uDE6C\uD803\uDE6D\uD803\uDE6E\uD803\uDE6F\uD803\uDE70\uD803\uDE71\uD803\uDE72\uD803\uDE73\uD803\uDE74\uD803\uDE75\uD803\uDE76\uD803\uDE77\uD803\uDE78\uD803\uDE79\uD803\uDE7A\uD803\uDE7B\uD803\uDE7C\uD803\uDE7D\uD803\uDE7E\uD804\uDC52\uD804\uDC53\uD804\uDC54\uD804\uDC55\uD804\uDC56\uD804\uDC57\uD804\uDC58\uD804\uDC59\uD804\uDC5A\uD804\uDC5B\uD804\uDC5C\uD804\uDC5D\uD804\uDC5E\uD804\uDC5F\uD804\uDC60\uD804\uDC61\uD804\uDC62\uD804\uDC63\uD804\uDC64\uD804\uDC65\uD834\uDF60\uD834\uDF61\uD834\uDF62\uD834\uDF63\uD834\uDF64\uD834\uDF65\uD834\uDF66\uD834\uDF67\uD834\uDF68\uD834\uDF69\uD834\uDF6A\uD834\uDF6B\uD834\uDF6C\uD834\uDF6D\uD834\uDF6E\uD834\uDF6F\uD834\uDF70\uD834\uDF71\uD83C\uDD00\uD83C\uDD01\uD83C\uDD02\uD83C\uDD03\uD83C\uDD04\uD83C\uDD05\uD83C\uDD06\uD83C\uDD07\uD83C\uDD08\uD83C\uDD09\uD83C\uDD0A'
			},
			{
				'description': 'Unicode character category [Pc]: punctuation, connector',
				'decoded': [95, 8255, 8256, 8276, 65075, 65076, 65101, 65102, 65103, 65343],
				'encoded': '_\u203F\u2040\u2054\uFE33\uFE34\uFE4D\uFE4E\uFE4F\uFF3F'
			},
			{
				'description': 'Unicode character category [Pd]: punctuation, dash',
				'decoded': [45, 1418, 1470, 5120, 6150, 8208, 8209, 8210, 8211, 8212, 8213, 11799, 11802, 12316, 12336, 12448, 65073, 65074, 65112, 65123, 65293],
				'encoded': '-\u058A\u05BE\u1400\u1806\u2010\u2011\u2012\u2013\u2014\u2015\u2E17\u2E1A\u301C\u3030\u30A0\uFE31\uFE32\uFE58\uFE63\uFF0D'
			},
			{
				'description': 'Unicode character category [Pe]: punctuation, close',
				'decoded': [41, 93, 125, 3899, 3901, 5788, 8262, 8318, 8334, 9002, 10089, 10091, 10093, 10095, 10097, 10099, 10101, 10182, 10215, 10217, 10219, 10221, 10223, 10628, 10630, 10632, 10634, 10636, 10638, 10640, 10642, 10644, 10646, 10648, 10713, 10715, 10749, 11811, 11813, 11815, 11817, 12297, 12299, 12301, 12303, 12305, 12309, 12311, 12313, 12315, 12318, 12319, 64831, 65048, 65078, 65080, 65082, 65084, 65086, 65088, 65090, 65092, 65096, 65114, 65116, 65118, 65289, 65341, 65373, 65376, 65379],
				'encoded': ')]}\u0F3B\u0F3D\u169C\u2046\u207E\u208E\u232A\u2769\u276B\u276D\u276F\u2771\u2773\u2775\u27C6\u27E7\u27E9\u27EB\u27ED\u27EF\u2984\u2986\u2988\u298A\u298C\u298E\u2990\u2992\u2994\u2996\u2998\u29D9\u29DB\u29FD\u2E23\u2E25\u2E27\u2E29\u3009\u300B\u300D\u300F\u3011\u3015\u3017\u3019\u301B\u301E\u301F\uFD3F\uFE18\uFE36\uFE38\uFE3A\uFE3C\uFE3E\uFE40\uFE42\uFE44\uFE48\uFE5A\uFE5C\uFE5E\uFF09\uFF3D\uFF5D\uFF60\uFF63'
			},
			{
				'description': 'Unicode character category [Pf]: punctuation, final quote',
				'decoded': [187, 8217, 8221, 8250, 11779, 11781, 11786, 11789, 11805, 11809],
				'encoded': '\xBB\u2019\u201D\u203A\u2E03\u2E05\u2E0A\u2E0D\u2E1D\u2E21'
			},
			{
				'description': 'Unicode character category [Pi]: punctuation, initial quote',
				'decoded': [171, 8216, 8219, 8220, 8223, 8249, 11778, 11780, 11785, 11788, 11804, 11808],
				'encoded': '\xAB\u2018\u201B\u201C\u201F\u2039\u2E02\u2E04\u2E09\u2E0C\u2E1C\u2E20'
			},
			{
				'description': 'Unicode character category [Po]: punctuation, other',
				'decoded': [33, 34, 35, 37, 38, 39, 42, 44, 46, 47, 58, 59, 63, 64, 161, 183, 191, 894, 903, 1370, 1371, 1372, 1373, 1374, 1375, 1417, 1472, 1475, 1478, 1523, 1524, 1545, 1546, 1548, 1549, 1563, 1566, 1567, 1642, 1643, 1644, 1645, 1748, 1792, 1793, 1794, 1795, 1796, 1797, 1798, 1799, 1800, 1801, 1802, 1803, 1804, 1805, 2039, 2040, 2041, 2096, 2097, 2098, 2099, 2100, 2101, 2102, 2103, 2104, 2105, 2106, 2107, 2108, 2109, 2110, 2142, 2404, 2405, 2416, 3572, 3663, 3674, 3675, 3844, 3845, 3846, 3847, 3848, 3849, 3850, 3851, 3852, 3853, 3854, 3855, 3856, 3857, 3858, 3973, 4048, 4049, 4050, 4051, 4052, 4057, 4058, 4170, 4171, 4172, 4173, 4174, 4175, 4347, 4961, 4962, 4963, 4964, 4965, 4966, 4967, 4968, 5741, 5742, 5867, 5868, 5869, 5941, 5942, 6100, 6101, 6102, 6104, 6105, 6106, 6144, 6145, 6146, 6147, 6148, 6149, 6151, 6152, 6153, 6154, 6468, 6469, 6686, 6687, 6816, 6817, 6818, 6819, 6820, 6821, 6822, 6824, 6825, 6826, 6827, 6828, 6829, 7002, 7003, 7004, 7005, 7006, 7007, 7008, 7164, 7165, 7166, 7167, 7227, 7228, 7229, 7230, 7231, 7294, 7295, 7379, 8214, 8215, 8224, 8225, 8226, 8227, 8228, 8229, 8230, 8231, 8240, 8241, 8242, 8243, 8244, 8245, 8246, 8247, 8248, 8251, 8252, 8253, 8254, 8257, 8258, 8259, 8263, 8264, 8265, 8266, 8267, 8268, 8269, 8270, 8271, 8272, 8273, 8275, 8277, 8278, 8279, 8280, 8281, 8282, 8283, 8284, 8285, 8286, 11513, 11514, 11515, 11516, 11518, 11519, 11632, 11776, 11777, 11782, 11783, 11784, 11787, 11790, 11791, 11792, 11793, 11794, 11795, 11796, 11797, 11798, 11800, 11801, 11803, 11806, 11807, 11818, 11819, 11820, 11821, 11822, 11824, 11825, 12289, 12290, 12291, 12349, 12539, 42238, 42239, 42509, 42510, 42511, 42611, 42622, 42738, 42739, 42740, 42741, 42742, 42743, 43124, 43125, 43126, 43127, 43214, 43215, 43256, 43257, 43258, 43310, 43311, 43359, 43457, 43458, 43459, 43460, 43461, 43462, 43463, 43464, 43465, 43466, 43467, 43468, 43469, 43486, 43487, 43612, 43613, 43614, 43615, 43742, 43743, 44011, 65040, 65041, 65042, 65043, 65044, 65045, 65046, 65049, 65072, 65093, 65094, 65097, 65098, 65099, 65100, 65104, 65105, 65106, 65108, 65109, 65110, 65111, 65119, 65120, 65121, 65128, 65130, 65131, 65281, 65282, 65283, 65285, 65286, 65287, 65290, 65292, 65294, 65295, 65306, 65307, 65311, 65312, 65340, 65377, 65380, 65381, 65792, 65793, 66463, 66512, 67671, 67871, 67903, 68176, 68177, 68178, 68179, 68180, 68181, 68182, 68183, 68184, 68223, 68409, 68410, 68411, 68412, 68413, 68414, 68415, 69703, 69704, 69705, 69706, 69707, 69708, 69709, 69819, 69820, 69822, 69823, 69824, 69825, 74864, 74865, 74866, 74867],
				'encoded': '!"#%&\'*,./:;?@\xA1\xB7\xBF\u037E\u0387\u055A\u055B\u055C\u055D\u055E\u055F\u0589\u05C0\u05C3\u05C6\u05F3\u05F4\u0609\u060A\u060C\u060D\u061B\u061E\u061F\u066A\u066B\u066C\u066D\u06D4\u0700\u0701\u0702\u0703\u0704\u0705\u0706\u0707\u0708\u0709\u070A\u070B\u070C\u070D\u07F7\u07F8\u07F9\u0830\u0831\u0832\u0833\u0834\u0835\u0836\u0837\u0838\u0839\u083A\u083B\u083C\u083D\u083E\u085E\u0964\u0965\u0970\u0DF4\u0E4F\u0E5A\u0E5B\u0F04\u0F05\u0F06\u0F07\u0F08\u0F09\u0F0A\u0F0B\u0F0C\u0F0D\u0F0E\u0F0F\u0F10\u0F11\u0F12\u0F85\u0FD0\u0FD1\u0FD2\u0FD3\u0FD4\u0FD9\u0FDA\u104A\u104B\u104C\u104D\u104E\u104F\u10FB\u1361\u1362\u1363\u1364\u1365\u1366\u1367\u1368\u166D\u166E\u16EB\u16EC\u16ED\u1735\u1736\u17D4\u17D5\u17D6\u17D8\u17D9\u17DA\u1800\u1801\u1802\u1803\u1804\u1805\u1807\u1808\u1809\u180A\u1944\u1945\u1A1E\u1A1F\u1AA0\u1AA1\u1AA2\u1AA3\u1AA4\u1AA5\u1AA6\u1AA8\u1AA9\u1AAA\u1AAB\u1AAC\u1AAD\u1B5A\u1B5B\u1B5C\u1B5D\u1B5E\u1B5F\u1B60\u1BFC\u1BFD\u1BFE\u1BFF\u1C3B\u1C3C\u1C3D\u1C3E\u1C3F\u1C7E\u1C7F\u1CD3\u2016\u2017\u2020\u2021\u2022\u2023\u2024\u2025\u2026\u2027\u2030\u2031\u2032\u2033\u2034\u2035\u2036\u2037\u2038\u203B\u203C\u203D\u203E\u2041\u2042\u2043\u2047\u2048\u2049\u204A\u204B\u204C\u204D\u204E\u204F\u2050\u2051\u2053\u2055\u2056\u2057\u2058\u2059\u205A\u205B\u205C\u205D\u205E\u2CF9\u2CFA\u2CFB\u2CFC\u2CFE\u2CFF\u2D70\u2E00\u2E01\u2E06\u2E07\u2E08\u2E0B\u2E0E\u2E0F\u2E10\u2E11\u2E12\u2E13\u2E14\u2E15\u2E16\u2E18\u2E19\u2E1B\u2E1E\u2E1F\u2E2A\u2E2B\u2E2C\u2E2D\u2E2E\u2E30\u2E31\u3001\u3002\u3003\u303D\u30FB\uA4FE\uA4FF\uA60D\uA60E\uA60F\uA673\uA67E\uA6F2\uA6F3\uA6F4\uA6F5\uA6F6\uA6F7\uA874\uA875\uA876\uA877\uA8CE\uA8CF\uA8F8\uA8F9\uA8FA\uA92E\uA92F\uA95F\uA9C1\uA9C2\uA9C3\uA9C4\uA9C5\uA9C6\uA9C7\uA9C8\uA9C9\uA9CA\uA9CB\uA9CC\uA9CD\uA9DE\uA9DF\uAA5C\uAA5D\uAA5E\uAA5F\uAADE\uAADF\uABEB\uFE10\uFE11\uFE12\uFE13\uFE14\uFE15\uFE16\uFE19\uFE30\uFE45\uFE46\uFE49\uFE4A\uFE4B\uFE4C\uFE50\uFE51\uFE52\uFE54\uFE55\uFE56\uFE57\uFE5F\uFE60\uFE61\uFE68\uFE6A\uFE6B\uFF01\uFF02\uFF03\uFF05\uFF06\uFF07\uFF0A\uFF0C\uFF0E\uFF0F\uFF1A\uFF1B\uFF1F\uFF20\uFF3C\uFF61\uFF64\uFF65\uD800\uDD00\uD800\uDD01\uD800\uDF9F\uD800\uDFD0\uD802\uDC57\uD802\uDD1F\uD802\uDD3F\uD802\uDE50\uD802\uDE51\uD802\uDE52\uD802\uDE53\uD802\uDE54\uD802\uDE55\uD802\uDE56\uD802\uDE57\uD802\uDE58\uD802\uDE7F\uD802\uDF39\uD802\uDF3A\uD802\uDF3B\uD802\uDF3C\uD802\uDF3D\uD802\uDF3E\uD802\uDF3F\uD804\uDC47\uD804\uDC48\uD804\uDC49\uD804\uDC4A\uD804\uDC4B\uD804\uDC4C\uD804\uDC4D\uD804\uDCBB\uD804\uDCBC\uD804\uDCBE\uD804\uDCBF\uD804\uDCC0\uD804\uDCC1\uD809\uDC70\uD809\uDC71\uD809\uDC72\uD809\uDC73'
			},
			{
				'description': 'Unicode character category [Ps]: punctuation, open',
				'decoded': [40, 91, 123, 3898, 3900, 5787, 8218, 8222, 8261, 8317, 8333, 9001, 10088, 10090, 10092, 10094, 10096, 10098, 10100, 10181, 10214, 10216, 10218, 10220, 10222, 10627, 10629, 10631, 10633, 10635, 10637, 10639, 10641, 10643, 10645, 10647, 10712, 10714, 10748, 11810, 11812, 11814, 11816, 12296, 12298, 12300, 12302, 12304, 12308, 12310, 12312, 12314, 12317, 64830, 65047, 65077, 65079, 65081, 65083, 65085, 65087, 65089, 65091, 65095, 65113, 65115, 65117, 65288, 65339, 65371, 65375, 65378],
				'encoded': '([{\u0F3A\u0F3C\u169B\u201A\u201E\u2045\u207D\u208D\u2329\u2768\u276A\u276C\u276E\u2770\u2772\u2774\u27C5\u27E6\u27E8\u27EA\u27EC\u27EE\u2983\u2985\u2987\u2989\u298B\u298D\u298F\u2991\u2993\u2995\u2997\u29D8\u29DA\u29FC\u2E22\u2E24\u2E26\u2E28\u3008\u300A\u300C\u300E\u3010\u3014\u3016\u3018\u301A\u301D\uFD3E\uFE17\uFE35\uFE37\uFE39\uFE3B\uFE3D\uFE3F\uFE41\uFE43\uFE47\uFE59\uFE5B\uFE5D\uFF08\uFF3B\uFF5B\uFF5F\uFF62'
			},
			{
				'description': 'Unicode character category [Sc]: symbol, currency',
				'decoded': [36, 162, 163, 164, 165, 1547, 2546, 2547, 2555, 2801, 3065, 3647, 6107, 8352, 8353, 8354, 8355, 8356, 8357, 8358, 8359, 8360, 8361, 8362, 8363, 8364, 8365, 8366, 8367, 8368, 8369, 8370, 8371, 8372, 8373, 8374, 8375, 8376, 8377, 43064, 65020, 65129, 65284, 65504, 65505, 65509, 65510],
				'encoded': '$\xA2\xA3\xA4\xA5\u060B\u09F2\u09F3\u09FB\u0AF1\u0BF9\u0E3F\u17DB\u20A0\u20A1\u20A2\u20A3\u20A4\u20A5\u20A6\u20A7\u20A8\u20A9\u20AA\u20AB\u20AC\u20AD\u20AE\u20AF\u20B0\u20B1\u20B2\u20B3\u20B4\u20B5\u20B6\u20B7\u20B8\u20B9\uA838\uFDFC\uFE69\uFF04\uFFE0\uFFE1\uFFE5\uFFE6'
			},
			{
				'description': 'Unicode character category [Sk]: symbol, modifier',
				'decoded': [94, 96, 168, 175, 180, 184, 706, 707, 708, 709, 722, 723, 724, 725, 726, 727, 728, 729, 730, 731, 732, 733, 734, 735, 741, 742, 743, 744, 745, 746, 747, 749, 751, 752, 753, 754, 755, 756, 757, 758, 759, 760, 761, 762, 763, 764, 765, 766, 767, 885, 900, 901, 8125, 8127, 8128, 8129, 8141, 8142, 8143, 8157, 8158, 8159, 8173, 8174, 8175, 8189, 8190, 12443, 12444, 42752, 42753, 42754, 42755, 42756, 42757, 42758, 42759, 42760, 42761, 42762, 42763, 42764, 42765, 42766, 42767, 42768, 42769, 42770, 42771, 42772, 42773, 42774, 42784, 42785, 42889, 42890, 64434, 64435, 64436, 64437, 64438, 64439, 64440, 64441, 64442, 64443, 64444, 64445, 64446, 64447, 64448, 64449, 65342, 65344, 65507],
				'encoded': '^`\xA8\xAF\xB4\xB8\u02C2\u02C3\u02C4\u02C5\u02D2\u02D3\u02D4\u02D5\u02D6\u02D7\u02D8\u02D9\u02DA\u02DB\u02DC\u02DD\u02DE\u02DF\u02E5\u02E6\u02E7\u02E8\u02E9\u02EA\u02EB\u02ED\u02EF\u02F0\u02F1\u02F2\u02F3\u02F4\u02F5\u02F6\u02F7\u02F8\u02F9\u02FA\u02FB\u02FC\u02FD\u02FE\u02FF\u0375\u0384\u0385\u1FBD\u1FBF\u1FC0\u1FC1\u1FCD\u1FCE\u1FCF\u1FDD\u1FDE\u1FDF\u1FED\u1FEE\u1FEF\u1FFD\u1FFE\u309B\u309C\uA700\uA701\uA702\uA703\uA704\uA705\uA706\uA707\uA708\uA709\uA70A\uA70B\uA70C\uA70D\uA70E\uA70F\uA710\uA711\uA712\uA713\uA714\uA715\uA716\uA720\uA721\uA789\uA78A\uFBB2\uFBB3\uFBB4\uFBB5\uFBB6\uFBB7\uFBB8\uFBB9\uFBBA\uFBBB\uFBBC\uFBBD\uFBBE\uFBBF\uFBC0\uFBC1\uFF3E\uFF40\uFFE3'
			},
			{
				'description': 'Unicode character category [Sm]: symbol, math',
				'decoded': [43, 60, 61, 62, 124, 126, 172, 177, 215, 247, 1014, 1542, 1543, 1544, 8260, 8274, 8314, 8315, 8316, 8330, 8331, 8332, 8472, 8512, 8513, 8514, 8515, 8516, 8523, 8592, 8593, 8594, 8595, 8596, 8602, 8603, 8608, 8611, 8614, 8622, 8654, 8655, 8658, 8660, 8692, 8693, 8694, 8695, 8696, 8697, 8698, 8699, 8700, 8701, 8702, 8703, 8704, 8705, 8706, 8707, 8708, 8709, 8710, 8711, 8712, 8713, 8714, 8715, 8716, 8717, 8718, 8719, 8720, 8721, 8722, 8723, 8724, 8725, 8726, 8727, 8728, 8729, 8730, 8731, 8732, 8733, 8734, 8735, 8736, 8737, 8738, 8739, 8740, 8741, 8742, 8743, 8744, 8745, 8746, 8747, 8748, 8749, 8750, 8751, 8752, 8753, 8754, 8755, 8756, 8757, 8758, 8759, 8760, 8761, 8762, 8763, 8764, 8765, 8766, 8767, 8768, 8769, 8770, 8771, 8772, 8773, 8774, 8775, 8776, 8777, 8778, 8779, 8780, 8781, 8782, 8783, 8784, 8785, 8786, 8787, 8788, 8789, 8790, 8791, 8792, 8793, 8794, 8795, 8796, 8797, 8798, 8799, 8800, 8801, 8802, 8803, 8804, 8805, 8806, 8807, 8808, 8809, 8810, 8811, 8812, 8813, 8814, 8815, 8816, 8817, 8818, 8819, 8820, 8821, 8822, 8823, 8824, 8825, 8826, 8827, 8828, 8829, 8830, 8831, 8832, 8833, 8834, 8835, 8836, 8837, 8838, 8839, 8840, 8841, 8842, 8843, 8844, 8845, 8846, 8847, 8848, 8849, 8850, 8851, 8852, 8853, 8854, 8855, 8856, 8857, 8858, 8859, 8860, 8861, 8862, 8863, 8864, 8865, 8866, 8867, 8868, 8869, 8870, 8871, 8872, 8873, 8874, 8875, 8876, 8877, 8878, 8879, 8880, 8881, 8882, 8883, 8884, 8885, 8886, 8887, 8888, 8889, 8890, 8891, 8892, 8893, 8894, 8895, 8896, 8897, 8898, 8899, 8900, 8901, 8902, 8903, 8904, 8905, 8906, 8907, 8908, 8909, 8910, 8911, 8912, 8913, 8914, 8915, 8916, 8917, 8918, 8919, 8920, 8921, 8922, 8923, 8924, 8925, 8926, 8927, 8928, 8929, 8930, 8931, 8932, 8933, 8934, 8935, 8936, 8937, 8938, 8939, 8940, 8941, 8942, 8943, 8944, 8945, 8946, 8947, 8948, 8949, 8950, 8951, 8952, 8953, 8954, 8955, 8956, 8957, 8958, 8959, 8968, 8969, 8970, 8971, 8992, 8993, 9084, 9115, 9116, 9117, 9118, 9119, 9120, 9121, 9122, 9123, 9124, 9125, 9126, 9127, 9128, 9129, 9130, 9131, 9132, 9133, 9134, 9135, 9136, 9137, 9138, 9139, 9180, 9181, 9182, 9183, 9184, 9185, 9655, 9665, 9720, 9721, 9722, 9723, 9724, 9725, 9726, 9727, 9839, 10176, 10177, 10178, 10179, 10180, 10183, 10184, 10185, 10186, 10188, 10190, 10191, 10192, 10193, 10194, 10195, 10196, 10197, 10198, 10199, 10200, 10201, 10202, 10203, 10204, 10205, 10206, 10207, 10208, 10209, 10210, 10211, 10212, 10213, 10224, 10225, 10226, 10227, 10228, 10229, 10230, 10231, 10232, 10233, 10234, 10235, 10236, 10237, 10238, 10239, 10496, 10497, 10498, 10499, 10500, 10501, 10502, 10503, 10504, 10505, 10506, 10507, 10508, 10509, 10510, 10511, 10512, 10513, 10514, 10515, 10516, 10517, 10518, 10519, 10520, 10521, 10522, 10523, 10524, 10525, 10526, 10527, 10528, 10529, 10530, 10531, 10532, 10533, 10534, 10535, 10536, 10537, 10538, 10539, 10540, 10541, 10542, 10543, 10544, 10545, 10546, 10547, 10548, 10549, 10550, 10551, 10552, 10553, 10554, 10555, 10556, 10557, 10558, 10559, 10560, 10561, 10562, 10563, 10564, 10565, 10566, 10567, 10568, 10569, 10570, 10571, 10572, 10573, 10574, 10575, 10576, 10577, 10578, 10579, 10580, 10581, 10582, 10583, 10584, 10585, 10586, 10587, 10588, 10589, 10590, 10591, 10592, 10593, 10594, 10595, 10596, 10597, 10598, 10599, 10600, 10601, 10602, 10603, 10604, 10605, 10606, 10607, 10608, 10609, 10610, 10611, 10612, 10613, 10614, 10615, 10616, 10617, 10618, 10619, 10620, 10621, 10622, 10623, 10624, 10625, 10626, 10649, 10650, 10651, 10652, 10653, 10654, 10655, 10656, 10657, 10658, 10659, 10660, 10661, 10662, 10663, 10664, 10665, 10666, 10667, 10668, 10669, 10670, 10671, 10672, 10673, 10674, 10675, 10676, 10677, 10678, 10679, 10680, 10681, 10682, 10683, 10684, 10685, 10686, 10687, 10688, 10689, 10690, 10691, 10692, 10693, 10694, 10695, 10696, 10697, 10698, 10699, 10700, 10701, 10702, 10703, 10704, 10705, 10706, 10707, 10708, 10709, 10710, 10711, 10716, 10717, 10718, 10719, 10720, 10721, 10722, 10723, 10724, 10725, 10726, 10727, 10728, 10729, 10730, 10731, 10732, 10733, 10734, 10735, 10736, 10737, 10738, 10739, 10740, 10741, 10742, 10743, 10744, 10745, 10746, 10747, 10750, 10751, 10752, 10753, 10754, 10755, 10756, 10757, 10758, 10759, 10760, 10761, 10762, 10763, 10764, 10765, 10766, 10767, 10768, 10769, 10770, 10771, 10772, 10773, 10774, 10775, 10776, 10777, 10778, 10779, 10780, 10781, 10782, 10783, 10784, 10785, 10786, 10787, 10788, 10789, 10790, 10791, 10792, 10793, 10794, 10795, 10796, 10797, 10798, 10799, 10800, 10801, 10802, 10803, 10804, 10805, 10806, 10807, 10808, 10809, 10810, 10811, 10812, 10813, 10814, 10815, 10816, 10817, 10818, 10819, 10820, 10821, 10822, 10823, 10824, 10825, 10826, 10827, 10828, 10829, 10830, 10831, 10832, 10833, 10834, 10835, 10836, 10837, 10838, 10839, 10840, 10841, 10842, 10843, 10844, 10845, 10846, 10847, 10848, 10849, 10850, 10851, 10852, 10853, 10854, 10855, 10856, 10857, 10858, 10859, 10860, 10861, 10862, 10863, 10864, 10865, 10866, 10867, 10868, 10869, 10870, 10871, 10872, 10873, 10874, 10875, 10876, 10877, 10878, 10879, 10880, 10881, 10882, 10883, 10884, 10885, 10886, 10887, 10888, 10889, 10890, 10891, 10892, 10893, 10894, 10895, 10896, 10897, 10898, 10899, 10900, 10901, 10902, 10903, 10904, 10905, 10906, 10907, 10908, 10909, 10910, 10911, 10912, 10913, 10914, 10915, 10916, 10917, 10918, 10919, 10920, 10921, 10922, 10923, 10924, 10925, 10926, 10927, 10928, 10929, 10930, 10931, 10932, 10933, 10934, 10935, 10936, 10937, 10938, 10939, 10940, 10941, 10942, 10943, 10944, 10945, 10946, 10947, 10948, 10949, 10950, 10951, 10952, 10953, 10954, 10955, 10956, 10957, 10958, 10959, 10960, 10961, 10962, 10963, 10964, 10965, 10966, 10967, 10968, 10969, 10970, 10971, 10972, 10973, 10974, 10975, 10976, 10977, 10978, 10979, 10980, 10981, 10982, 10983, 10984, 10985, 10986, 10987, 10988, 10989, 10990, 10991, 10992, 10993, 10994, 10995, 10996, 10997, 10998, 10999, 11000, 11001, 11002, 11003, 11004, 11005, 11006, 11007, 11056, 11057, 11058, 11059, 11060, 11061, 11062, 11063, 11064, 11065, 11066, 11067, 11068, 11069, 11070, 11071, 11072, 11073, 11074, 11075, 11076, 11079, 11080, 11081, 11082, 11083, 11084, 64297, 65122, 65124, 65125, 65126, 65291, 65308, 65309, 65310, 65372, 65374, 65506, 65513, 65514, 65515, 65516, 120513, 120539, 120571, 120597, 120629, 120655, 120687, 120713, 120745, 120771],
				'encoded': '+<=>|~\xAC\xB1\xD7\xF7\u03F6\u0606\u0607\u0608\u2044\u2052\u207A\u207B\u207C\u208A\u208B\u208C\u2118\u2140\u2141\u2142\u2143\u2144\u214B\u2190\u2191\u2192\u2193\u2194\u219A\u219B\u21A0\u21A3\u21A6\u21AE\u21CE\u21CF\u21D2\u21D4\u21F4\u21F5\u21F6\u21F7\u21F8\u21F9\u21FA\u21FB\u21FC\u21FD\u21FE\u21FF\u2200\u2201\u2202\u2203\u2204\u2205\u2206\u2207\u2208\u2209\u220A\u220B\u220C\u220D\u220E\u220F\u2210\u2211\u2212\u2213\u2214\u2215\u2216\u2217\u2218\u2219\u221A\u221B\u221C\u221D\u221E\u221F\u2220\u2221\u2222\u2223\u2224\u2225\u2226\u2227\u2228\u2229\u222A\u222B\u222C\u222D\u222E\u222F\u2230\u2231\u2232\u2233\u2234\u2235\u2236\u2237\u2238\u2239\u223A\u223B\u223C\u223D\u223E\u223F\u2240\u2241\u2242\u2243\u2244\u2245\u2246\u2247\u2248\u2249\u224A\u224B\u224C\u224D\u224E\u224F\u2250\u2251\u2252\u2253\u2254\u2255\u2256\u2257\u2258\u2259\u225A\u225B\u225C\u225D\u225E\u225F\u2260\u2261\u2262\u2263\u2264\u2265\u2266\u2267\u2268\u2269\u226A\u226B\u226C\u226D\u226E\u226F\u2270\u2271\u2272\u2273\u2274\u2275\u2276\u2277\u2278\u2279\u227A\u227B\u227C\u227D\u227E\u227F\u2280\u2281\u2282\u2283\u2284\u2285\u2286\u2287\u2288\u2289\u228A\u228B\u228C\u228D\u228E\u228F\u2290\u2291\u2292\u2293\u2294\u2295\u2296\u2297\u2298\u2299\u229A\u229B\u229C\u229D\u229E\u229F\u22A0\u22A1\u22A2\u22A3\u22A4\u22A5\u22A6\u22A7\u22A8\u22A9\u22AA\u22AB\u22AC\u22AD\u22AE\u22AF\u22B0\u22B1\u22B2\u22B3\u22B4\u22B5\u22B6\u22B7\u22B8\u22B9\u22BA\u22BB\u22BC\u22BD\u22BE\u22BF\u22C0\u22C1\u22C2\u22C3\u22C4\u22C5\u22C6\u22C7\u22C8\u22C9\u22CA\u22CB\u22CC\u22CD\u22CE\u22CF\u22D0\u22D1\u22D2\u22D3\u22D4\u22D5\u22D6\u22D7\u22D8\u22D9\u22DA\u22DB\u22DC\u22DD\u22DE\u22DF\u22E0\u22E1\u22E2\u22E3\u22E4\u22E5\u22E6\u22E7\u22E8\u22E9\u22EA\u22EB\u22EC\u22ED\u22EE\u22EF\u22F0\u22F1\u22F2\u22F3\u22F4\u22F5\u22F6\u22F7\u22F8\u22F9\u22FA\u22FB\u22FC\u22FD\u22FE\u22FF\u2308\u2309\u230A\u230B\u2320\u2321\u237C\u239B\u239C\u239D\u239E\u239F\u23A0\u23A1\u23A2\u23A3\u23A4\u23A5\u23A6\u23A7\u23A8\u23A9\u23AA\u23AB\u23AC\u23AD\u23AE\u23AF\u23B0\u23B1\u23B2\u23B3\u23DC\u23DD\u23DE\u23DF\u23E0\u23E1\u25B7\u25C1\u25F8\u25F9\u25FA\u25FB\u25FC\u25FD\u25FE\u25FF\u266F\u27C0\u27C1\u27C2\u27C3\u27C4\u27C7\u27C8\u27C9\u27CA\u27CC\u27CE\u27CF\u27D0\u27D1\u27D2\u27D3\u27D4\u27D5\u27D6\u27D7\u27D8\u27D9\u27DA\u27DB\u27DC\u27DD\u27DE\u27DF\u27E0\u27E1\u27E2\u27E3\u27E4\u27E5\u27F0\u27F1\u27F2\u27F3\u27F4\u27F5\u27F6\u27F7\u27F8\u27F9\u27FA\u27FB\u27FC\u27FD\u27FE\u27FF\u2900\u2901\u2902\u2903\u2904\u2905\u2906\u2907\u2908\u2909\u290A\u290B\u290C\u290D\u290E\u290F\u2910\u2911\u2912\u2913\u2914\u2915\u2916\u2917\u2918\u2919\u291A\u291B\u291C\u291D\u291E\u291F\u2920\u2921\u2922\u2923\u2924\u2925\u2926\u2927\u2928\u2929\u292A\u292B\u292C\u292D\u292E\u292F\u2930\u2931\u2932\u2933\u2934\u2935\u2936\u2937\u2938\u2939\u293A\u293B\u293C\u293D\u293E\u293F\u2940\u2941\u2942\u2943\u2944\u2945\u2946\u2947\u2948\u2949\u294A\u294B\u294C\u294D\u294E\u294F\u2950\u2951\u2952\u2953\u2954\u2955\u2956\u2957\u2958\u2959\u295A\u295B\u295C\u295D\u295E\u295F\u2960\u2961\u2962\u2963\u2964\u2965\u2966\u2967\u2968\u2969\u296A\u296B\u296C\u296D\u296E\u296F\u2970\u2971\u2972\u2973\u2974\u2975\u2976\u2977\u2978\u2979\u297A\u297B\u297C\u297D\u297E\u297F\u2980\u2981\u2982\u2999\u299A\u299B\u299C\u299D\u299E\u299F\u29A0\u29A1\u29A2\u29A3\u29A4\u29A5\u29A6\u29A7\u29A8\u29A9\u29AA\u29AB\u29AC\u29AD\u29AE\u29AF\u29B0\u29B1\u29B2\u29B3\u29B4\u29B5\u29B6\u29B7\u29B8\u29B9\u29BA\u29BB\u29BC\u29BD\u29BE\u29BF\u29C0\u29C1\u29C2\u29C3\u29C4\u29C5\u29C6\u29C7\u29C8\u29C9\u29CA\u29CB\u29CC\u29CD\u29CE\u29CF\u29D0\u29D1\u29D2\u29D3\u29D4\u29D5\u29D6\u29D7\u29DC\u29DD\u29DE\u29DF\u29E0\u29E1\u29E2\u29E3\u29E4\u29E5\u29E6\u29E7\u29E8\u29E9\u29EA\u29EB\u29EC\u29ED\u29EE\u29EF\u29F0\u29F1\u29F2\u29F3\u29F4\u29F5\u29F6\u29F7\u29F8\u29F9\u29FA\u29FB\u29FE\u29FF\u2A00\u2A01\u2A02\u2A03\u2A04\u2A05\u2A06\u2A07\u2A08\u2A09\u2A0A\u2A0B\u2A0C\u2A0D\u2A0E\u2A0F\u2A10\u2A11\u2A12\u2A13\u2A14\u2A15\u2A16\u2A17\u2A18\u2A19\u2A1A\u2A1B\u2A1C\u2A1D\u2A1E\u2A1F\u2A20\u2A21\u2A22\u2A23\u2A24\u2A25\u2A26\u2A27\u2A28\u2A29\u2A2A\u2A2B\u2A2C\u2A2D\u2A2E\u2A2F\u2A30\u2A31\u2A32\u2A33\u2A34\u2A35\u2A36\u2A37\u2A38\u2A39\u2A3A\u2A3B\u2A3C\u2A3D\u2A3E\u2A3F\u2A40\u2A41\u2A42\u2A43\u2A44\u2A45\u2A46\u2A47\u2A48\u2A49\u2A4A\u2A4B\u2A4C\u2A4D\u2A4E\u2A4F\u2A50\u2A51\u2A52\u2A53\u2A54\u2A55\u2A56\u2A57\u2A58\u2A59\u2A5A\u2A5B\u2A5C\u2A5D\u2A5E\u2A5F\u2A60\u2A61\u2A62\u2A63\u2A64\u2A65\u2A66\u2A67\u2A68\u2A69\u2A6A\u2A6B\u2A6C\u2A6D\u2A6E\u2A6F\u2A70\u2A71\u2A72\u2A73\u2A74\u2A75\u2A76\u2A77\u2A78\u2A79\u2A7A\u2A7B\u2A7C\u2A7D\u2A7E\u2A7F\u2A80\u2A81\u2A82\u2A83\u2A84\u2A85\u2A86\u2A87\u2A88\u2A89\u2A8A\u2A8B\u2A8C\u2A8D\u2A8E\u2A8F\u2A90\u2A91\u2A92\u2A93\u2A94\u2A95\u2A96\u2A97\u2A98\u2A99\u2A9A\u2A9B\u2A9C\u2A9D\u2A9E\u2A9F\u2AA0\u2AA1\u2AA2\u2AA3\u2AA4\u2AA5\u2AA6\u2AA7\u2AA8\u2AA9\u2AAA\u2AAB\u2AAC\u2AAD\u2AAE\u2AAF\u2AB0\u2AB1\u2AB2\u2AB3\u2AB4\u2AB5\u2AB6\u2AB7\u2AB8\u2AB9\u2ABA\u2ABB\u2ABC\u2ABD\u2ABE\u2ABF\u2AC0\u2AC1\u2AC2\u2AC3\u2AC4\u2AC5\u2AC6\u2AC7\u2AC8\u2AC9\u2ACA\u2ACB\u2ACC\u2ACD\u2ACE\u2ACF\u2AD0\u2AD1\u2AD2\u2AD3\u2AD4\u2AD5\u2AD6\u2AD7\u2AD8\u2AD9\u2ADA\u2ADB\u2ADC\u2ADD\u2ADE\u2ADF\u2AE0\u2AE1\u2AE2\u2AE3\u2AE4\u2AE5\u2AE6\u2AE7\u2AE8\u2AE9\u2AEA\u2AEB\u2AEC\u2AED\u2AEE\u2AEF\u2AF0\u2AF1\u2AF2\u2AF3\u2AF4\u2AF5\u2AF6\u2AF7\u2AF8\u2AF9\u2AFA\u2AFB\u2AFC\u2AFD\u2AFE\u2AFF\u2B30\u2B31\u2B32\u2B33\u2B34\u2B35\u2B36\u2B37\u2B38\u2B39\u2B3A\u2B3B\u2B3C\u2B3D\u2B3E\u2B3F\u2B40\u2B41\u2B42\u2B43\u2B44\u2B47\u2B48\u2B49\u2B4A\u2B4B\u2B4C\uFB29\uFE62\uFE64\uFE65\uFE66\uFF0B\uFF1C\uFF1D\uFF1E\uFF5C\uFF5E\uFFE2\uFFE9\uFFEA\uFFEB\uFFEC\uD835\uDEC1\uD835\uDEDB\uD835\uDEFB\uD835\uDF15\uD835\uDF35\uD835\uDF4F\uD835\uDF6F\uD835\uDF89\uD835\uDFA9\uD835\uDFC3'
			},
			{
				'description': 'Unicode character category [So]: symbol, other',
				'decoded': [166, 167, 169, 174, 176, 182, 1154, 1550, 1551, 1758, 1769, 1789, 1790, 2038, 2554, 2928, 3059, 3060, 3061, 3062, 3063, 3064, 3066, 3199, 3449, 3841, 3842, 3843, 3859, 3860, 3861, 3862, 3863, 3866, 3867, 3868, 3869, 3870, 3871, 3892, 3894, 3896, 4030, 4031, 4032, 4033, 4034, 4035, 4036, 4037, 4039, 4040, 4041, 4042, 4043, 4044, 4046, 4047, 4053, 4054, 4055, 4056, 4254, 4255, 4960, 5008, 5009, 5010, 5011, 5012, 5013, 5014, 5015, 5016, 5017, 6464, 6622, 6623, 6624, 6625, 6626, 6627, 6628, 6629, 6630, 6631, 6632, 6633, 6634, 6635, 6636, 6637, 6638, 6639, 6640, 6641, 6642, 6643, 6644, 6645, 6646, 6647, 6648, 6649, 6650, 6651, 6652, 6653, 6654, 6655, 7009, 7010, 7011, 7012, 7013, 7014, 7015, 7016, 7017, 7018, 7028, 7029, 7030, 7031, 7032, 7033, 7034, 7035, 7036, 8448, 8449, 8451, 8452, 8453, 8454, 8456, 8457, 8468, 8470, 8471, 8478, 8479, 8480, 8481, 8482, 8483, 8485, 8487, 8489, 8494, 8506, 8507, 8522, 8524, 8525, 8527, 8597, 8598, 8599, 8600, 8601, 8604, 8605, 8606, 8607, 8609, 8610, 8612, 8613, 8615, 8616, 8617, 8618, 8619, 8620, 8621, 8623, 8624, 8625, 8626, 8627, 8628, 8629, 8630, 8631, 8632, 8633, 8634, 8635, 8636, 8637, 8638, 8639, 8640, 8641, 8642, 8643, 8644, 8645, 8646, 8647, 8648, 8649, 8650, 8651, 8652, 8653, 8656, 8657, 8659, 8661, 8662, 8663, 8664, 8665, 8666, 8667, 8668, 8669, 8670, 8671, 8672, 8673, 8674, 8675, 8676, 8677, 8678, 8679, 8680, 8681, 8682, 8683, 8684, 8685, 8686, 8687, 8688, 8689, 8690, 8691, 8960, 8961, 8962, 8963, 8964, 8965, 8966, 8967, 8972, 8973, 8974, 8975, 8976, 8977, 8978, 8979, 8980, 8981, 8982, 8983, 8984, 8985, 8986, 8987, 8988, 8989, 8990, 8991, 8994, 8995, 8996, 8997, 8998, 8999, 9000, 9003, 9004, 9005, 9006, 9007, 9008, 9009, 9010, 9011, 9012, 9013, 9014, 9015, 9016, 9017, 9018, 9019, 9020, 9021, 9022, 9023, 9024, 9025, 9026, 9027, 9028, 9029, 9030, 9031, 9032, 9033, 9034, 9035, 9036, 9037, 9038, 9039, 9040, 9041, 9042, 9043, 9044, 9045, 9046, 9047, 9048, 9049, 9050, 9051, 9052, 9053, 9054, 9055, 9056, 9057, 9058, 9059, 9060, 9061, 9062, 9063, 9064, 9065, 9066, 9067, 9068, 9069, 9070, 9071, 9072, 9073, 9074, 9075, 9076, 9077, 9078, 9079, 9080, 9081, 9082, 9083, 9085, 9086, 9087, 9088, 9089, 9090, 9091, 9092, 9093, 9094, 9095, 9096, 9097, 9098, 9099, 9100, 9101, 9102, 9103, 9104, 9105, 9106, 9107, 9108, 9109, 9110, 9111, 9112, 9113, 9114, 9140, 9141, 9142, 9143, 9144, 9145, 9146, 9147, 9148, 9149, 9150, 9151, 9152, 9153, 9154, 9155, 9156, 9157, 9158, 9159, 9160, 9161, 9162, 9163, 9164, 9165, 9166, 9167, 9168, 9169, 9170, 9171, 9172, 9173, 9174, 9175, 9176, 9177, 9178, 9179, 9186, 9187, 9188, 9189, 9190, 9191, 9192, 9193, 9194, 9195, 9196, 9197, 9198, 9199, 9200, 9201, 9202, 9203, 9216, 9217, 9218, 9219, 9220, 9221, 9222, 9223, 9224, 9225, 9226, 9227, 9228, 9229, 9230, 9231, 9232, 9233, 9234, 9235, 9236, 9237, 9238, 9239, 9240, 9241, 9242, 9243, 9244, 9245, 9246, 9247, 9248, 9249, 9250, 9251, 9252, 9253, 9254, 9280, 9281, 9282, 9283, 9284, 9285, 9286, 9287, 9288, 9289, 9290, 9372, 9373, 9374, 9375, 9376, 9377, 9378, 9379, 9380, 9381, 9382, 9383, 9384, 9385, 9386, 9387, 9388, 9389, 9390, 9391, 9392, 9393, 9394, 9395, 9396, 9397, 9398, 9399, 9400, 9401, 9402, 9403, 9404, 9405, 9406, 9407, 9408, 9409, 9410, 9411, 9412, 9413, 9414, 9415, 9416, 9417, 9418, 9419, 9420, 9421, 9422, 9423, 9424, 9425, 9426, 9427, 9428, 9429, 9430, 9431, 9432, 9433, 9434, 9435, 9436, 9437, 9438, 9439, 9440, 9441, 9442, 9443, 9444, 9445, 9446, 9447, 9448, 9449, 9472, 9473, 9474, 9475, 9476, 9477, 9478, 9479, 9480, 9481, 9482, 9483, 9484, 9485, 9486, 9487, 9488, 9489, 9490, 9491, 9492, 9493, 9494, 9495, 9496, 9497, 9498, 9499, 9500, 9501, 9502, 9503, 9504, 9505, 9506, 9507, 9508, 9509, 9510, 9511, 9512, 9513, 9514, 9515, 9516, 9517, 9518, 9519, 9520, 9521, 9522, 9523, 9524, 9525, 9526, 9527, 9528, 9529, 9530, 9531, 9532, 9533, 9534, 9535, 9536, 9537, 9538, 9539, 9540, 9541, 9542, 9543, 9544, 9545, 9546, 9547, 9548, 9549, 9550, 9551, 9552, 9553, 9554, 9555, 9556, 9557, 9558, 9559, 9560, 9561, 9562, 9563, 9564, 9565, 9566, 9567, 9568, 9569, 9570, 9571, 9572, 9573, 9574, 9575, 9576, 9577, 9578, 9579, 9580, 9581, 9582, 9583, 9584, 9585, 9586, 9587, 9588, 9589, 9590, 9591, 9592, 9593, 9594, 9595, 9596, 9597, 9598, 9599, 9600, 9601, 9602, 9603, 9604, 9605, 9606, 9607, 9608, 9609, 9610, 9611, 9612, 9613, 9614, 9615, 9616, 9617, 9618, 9619, 9620, 9621, 9622, 9623, 9624, 9625, 9626, 9627, 9628, 9629, 9630, 9631, 9632, 9633, 9634, 9635, 9636, 9637, 9638, 9639, 9640, 9641, 9642, 9643, 9644, 9645, 9646, 9647, 9648, 9649, 9650, 9651, 9652, 9653, 9654, 9656, 9657, 9658, 9659, 9660, 9661, 9662, 9663, 9664, 9666, 9667, 9668, 9669, 9670, 9671, 9672, 9673, 9674, 9675, 9676, 9677, 9678, 9679, 9680, 9681, 9682, 9683, 9684, 9685, 9686, 9687, 9688, 9689, 9690, 9691, 9692, 9693, 9694, 9695, 9696, 9697, 9698, 9699, 9700, 9701, 9702, 9703, 9704, 9705, 9706, 9707, 9708, 9709, 9710, 9711, 9712, 9713, 9714, 9715, 9716, 9717, 9718, 9719, 9728, 9729, 9730, 9731, 9732, 9733, 9734, 9735, 9736, 9737, 9738, 9739, 9740, 9741, 9742, 9743, 9744, 9745, 9746, 9747, 9748, 9749, 9750, 9751, 9752, 9753, 9754, 9755, 9756, 9757, 9758, 9759, 9760, 9761, 9762, 9763, 9764, 9765, 9766, 9767, 9768, 9769, 9770, 9771, 9772, 9773, 9774, 9775, 9776, 9777, 9778, 9779, 9780, 9781, 9782, 9783, 9784, 9785, 9786, 9787, 9788, 9789, 9790, 9791, 9792, 9793, 9794, 9795, 9796, 9797, 9798, 9799, 9800, 9801, 9802, 9803, 9804, 9805, 9806, 9807, 9808, 9809, 9810, 9811, 9812, 9813, 9814, 9815, 9816, 9817, 9818, 9819, 9820, 9821, 9822, 9823, 9824, 9825, 9826, 9827, 9828, 9829, 9830, 9831, 9832, 9833, 9834, 9835, 9836, 9837, 9838, 9840, 9841, 9842, 9843, 9844, 9845, 9846, 9847, 9848, 9849, 9850, 9851, 9852, 9853, 9854, 9855, 9856, 9857, 9858, 9859, 9860, 9861, 9862, 9863, 9864, 9865, 9866, 9867, 9868, 9869, 9870, 9871, 9872, 9873, 9874, 9875, 9876, 9877, 9878, 9879, 9880, 9881, 9882, 9883, 9884, 9885, 9886, 9887, 9888, 9889, 9890, 9891, 9892, 9893, 9894, 9895, 9896, 9897, 9898, 9899, 9900, 9901, 9902, 9903, 9904, 9905, 9906, 9907, 9908, 9909, 9910, 9911, 9912, 9913, 9914, 9915, 9916, 9917, 9918, 9919, 9920, 9921, 9922, 9923, 9924, 9925, 9926, 9927, 9928, 9929, 9930, 9931, 9932, 9933, 9934, 9935, 9936, 9937, 9938, 9939, 9940, 9941, 9942, 9943, 9944, 9945, 9946, 9947, 9948, 9949, 9950, 9951, 9952, 9953, 9954, 9955, 9956, 9957, 9958, 9959, 9960, 9961, 9962, 9963, 9964, 9965, 9966, 9967, 9968, 9969, 9970, 9971, 9972, 9973, 9974, 9975, 9976, 9977, 9978, 9979, 9980, 9981, 9982, 9983, 9985, 9986, 9987, 9988, 9989, 9990, 9991, 9992, 9993, 9994, 9995, 9996, 9997, 9998, 9999, 10000, 10001, 10002, 10003, 10004, 10005, 10006, 10007, 10008, 10009, 10010, 10011, 10012, 10013, 10014, 10015, 10016, 10017, 10018, 10019, 10020, 10021, 10022, 10023, 10024, 10025, 10026, 10027, 10028, 10029, 10030, 10031, 10032, 10033, 10034, 10035, 10036, 10037, 10038, 10039, 10040, 10041, 10042, 10043, 10044, 10045, 10046, 10047, 10048, 10049, 10050, 10051, 10052, 10053, 10054, 10055, 10056, 10057, 10058, 10059, 10060, 10061, 10062, 10063, 10064, 10065, 10066, 10067, 10068, 10069, 10070, 10071, 10072, 10073, 10074, 10075, 10076, 10077, 10078, 10079, 10080, 10081, 10082, 10083, 10084, 10085, 10086, 10087, 10132, 10133, 10134, 10135, 10136, 10137, 10138, 10139, 10140, 10141, 10142, 10143, 10144, 10145, 10146, 10147, 10148, 10149, 10150, 10151, 10152, 10153, 10154, 10155, 10156, 10157, 10158, 10159, 10160, 10161, 10162, 10163, 10164, 10165, 10166, 10167, 10168, 10169, 10170, 10171, 10172, 10173, 10174, 10175, 10240, 10241, 10242, 10243, 10244, 10245, 10246, 10247, 10248, 10249, 10250, 10251, 10252, 10253, 10254, 10255, 10256, 10257, 10258, 10259, 10260, 10261, 10262, 10263, 10264, 10265, 10266, 10267, 10268, 10269, 10270, 10271, 10272, 10273, 10274, 10275, 10276, 10277, 10278, 10279, 10280, 10281, 10282, 10283, 10284, 10285, 10286, 10287, 10288, 10289, 10290, 10291, 10292, 10293, 10294, 10295, 10296, 10297, 10298, 10299, 10300, 10301, 10302, 10303, 10304, 10305, 10306, 10307, 10308, 10309, 10310, 10311, 10312, 10313, 10314, 10315, 10316, 10317, 10318, 10319, 10320, 10321, 10322, 10323, 10324, 10325, 10326, 10327, 10328, 10329, 10330, 10331, 10332, 10333, 10334, 10335, 10336, 10337, 10338, 10339, 10340, 10341, 10342, 10343, 10344, 10345, 10346, 10347, 10348, 10349, 10350, 10351, 10352, 10353, 10354, 10355, 10356, 10357, 10358, 10359, 10360, 10361, 10362, 10363, 10364, 10365, 10366, 10367, 10368, 10369, 10370, 10371, 10372, 10373, 10374, 10375, 10376, 10377, 10378, 10379, 10380, 10381, 10382, 10383, 10384, 10385, 10386, 10387, 10388, 10389, 10390, 10391, 10392, 10393, 10394, 10395, 10396, 10397, 10398, 10399, 10400, 10401, 10402, 10403, 10404, 10405, 10406, 10407, 10408, 10409, 10410, 10411, 10412, 10413, 10414, 10415, 10416, 10417, 10418, 10419, 10420, 10421, 10422, 10423, 10424, 10425, 10426, 10427, 10428, 10429, 10430, 10431, 10432, 10433, 10434, 10435, 10436, 10437, 10438, 10439, 10440, 10441, 10442, 10443, 10444, 10445, 10446, 10447, 10448, 10449, 10450, 10451, 10452, 10453, 10454, 10455, 10456, 10457, 10458, 10459, 10460, 10461, 10462, 10463, 10464, 10465, 10466, 10467, 10468, 10469, 10470, 10471, 10472, 10473, 10474, 10475, 10476, 10477, 10478, 10479, 10480, 10481, 10482, 10483, 10484, 10485, 10486, 10487, 10488, 10489, 10490, 10491, 10492, 10493, 10494, 10495, 11008, 11009, 11010, 11011, 11012, 11013, 11014, 11015, 11016, 11017, 11018, 11019, 11020, 11021, 11022, 11023, 11024, 11025, 11026, 11027, 11028, 11029, 11030, 11031, 11032, 11033, 11034, 11035, 11036, 11037, 11038, 11039, 11040, 11041, 11042, 11043, 11044, 11045, 11046, 11047, 11048, 11049, 11050, 11051, 11052, 11053, 11054, 11055, 11077, 11078, 11088, 11089, 11090, 11091, 11092, 11093, 11094, 11095, 11096, 11097, 11493, 11494, 11495, 11496, 11497, 11498, 11904, 11905, 11906, 11907, 11908, 11909, 11910, 11911, 11912, 11913, 11914, 11915, 11916, 11917, 11918, 11919, 11920, 11921, 11922, 11923, 11924, 11925, 11926, 11927, 11928, 11929, 11931, 11932, 11933, 11934, 11935, 11936, 11937, 11938, 11939, 11940, 11941, 11942, 11943, 11944, 11945, 11946, 11947, 11948, 11949, 11950, 11951, 11952, 11953, 11954, 11955, 11956, 11957, 11958, 11959, 11960, 11961, 11962, 11963, 11964, 11965, 11966, 11967, 11968, 11969, 11970, 11971, 11972, 11973, 11974, 11975, 11976, 11977, 11978, 11979, 11980, 11981, 11982, 11983, 11984, 11985, 11986, 11987, 11988, 11989, 11990, 11991, 11992, 11993, 11994, 11995, 11996, 11997, 11998, 11999, 12000, 12001, 12002, 12003, 12004, 12005, 12006, 12007, 12008, 12009, 12010, 12011, 12012, 12013, 12014, 12015, 12016, 12017, 12018, 12019, 12032, 12033, 12034, 12035, 12036, 12037, 12038, 12039, 12040, 12041, 12042, 12043, 12044, 12045, 12046, 12047, 12048, 12049, 12050, 12051, 12052, 12053, 12054, 12055, 12056, 12057, 12058, 12059, 12060, 12061, 12062, 12063, 12064, 12065, 12066, 12067, 12068, 12069, 12070, 12071, 12072, 12073, 12074, 12075, 12076, 12077, 12078, 12079, 12080, 12081, 12082, 12083, 12084, 12085, 12086, 12087, 12088, 12089, 12090, 12091, 12092, 12093, 12094, 12095, 12096, 12097, 12098, 12099, 12100, 12101, 12102, 12103, 12104, 12105, 12106, 12107, 12108, 12109, 12110, 12111, 12112, 12113, 12114, 12115, 12116, 12117, 12118, 12119, 12120, 12121, 12122, 12123, 12124, 12125, 12126, 12127, 12128, 12129, 12130, 12131, 12132, 12133, 12134, 12135, 12136, 12137, 12138, 12139, 12140, 12141, 12142, 12143, 12144, 12145, 12146, 12147, 12148, 12149, 12150, 12151, 12152, 12153, 12154, 12155, 12156, 12157, 12158, 12159, 12160, 12161, 12162, 12163, 12164, 12165, 12166, 12167, 12168, 12169, 12170, 12171, 12172, 12173, 12174, 12175, 12176, 12177, 12178, 12179, 12180, 12181, 12182, 12183, 12184, 12185, 12186, 12187, 12188, 12189, 12190, 12191, 12192, 12193, 12194, 12195, 12196, 12197, 12198, 12199, 12200, 12201, 12202, 12203, 12204, 12205, 12206, 12207, 12208, 12209, 12210, 12211, 12212, 12213, 12214, 12215, 12216, 12217, 12218, 12219, 12220, 12221, 12222, 12223, 12224, 12225, 12226, 12227, 12228, 12229, 12230, 12231, 12232, 12233, 12234, 12235, 12236, 12237, 12238, 12239, 12240, 12241, 12242, 12243, 12244, 12245, 12272, 12273, 12274, 12275, 12276, 12277, 12278, 12279, 12280, 12281, 12282, 12283, 12292, 12306, 12307, 12320, 12342, 12343, 12350, 12351, 12688, 12689, 12694, 12695, 12696, 12697, 12698, 12699, 12700, 12701, 12702, 12703, 12736, 12737, 12738, 12739, 12740, 12741, 12742, 12743, 12744, 12745, 12746, 12747, 12748, 12749, 12750, 12751, 12752, 12753, 12754, 12755, 12756, 12757, 12758, 12759, 12760, 12761, 12762, 12763, 12764, 12765, 12766, 12767, 12768, 12769, 12770, 12771, 12800, 12801, 12802, 12803, 12804, 12805, 12806, 12807, 12808, 12809, 12810, 12811, 12812, 12813, 12814, 12815, 12816, 12817, 12818, 12819, 12820, 12821, 12822, 12823, 12824, 12825, 12826, 12827, 12828, 12829, 12830, 12842, 12843, 12844, 12845, 12846, 12847, 12848, 12849, 12850, 12851, 12852, 12853, 12854, 12855, 12856, 12857, 12858, 12859, 12860, 12861, 12862, 12863, 12864, 12865, 12866, 12867, 12868, 12869, 12870, 12871, 12872, 12873, 12874, 12875, 12876, 12877, 12878, 12879, 12880, 12896, 12897, 12898, 12899, 12900, 12901, 12902, 12903, 12904, 12905, 12906, 12907, 12908, 12909, 12910, 12911, 12912, 12913, 12914, 12915, 12916, 12917, 12918, 12919, 12920, 12921, 12922, 12923, 12924, 12925, 12926, 12927, 12938, 12939, 12940, 12941, 12942, 12943, 12944, 12945, 12946, 12947, 12948, 12949, 12950, 12951, 12952, 12953, 12954, 12955, 12956, 12957, 12958, 12959, 12960, 12961, 12962, 12963, 12964, 12965, 12966, 12967, 12968, 12969, 12970, 12971, 12972, 12973, 12974, 12975, 12976, 12992, 12993, 12994, 12995, 12996, 12997, 12998, 12999, 13000, 13001, 13002, 13003, 13004, 13005, 13006, 13007, 13008, 13009, 13010, 13011, 13012, 13013, 13014, 13015, 13016, 13017, 13018, 13019, 13020, 13021, 13022, 13023, 13024, 13025, 13026, 13027, 13028, 13029, 13030, 13031, 13032, 13033, 13034, 13035, 13036, 13037, 13038, 13039, 13040, 13041, 13042, 13043, 13044, 13045, 13046, 13047, 13048, 13049, 13050, 13051, 13052, 13053, 13054, 13056, 13057, 13058, 13059, 13060, 13061, 13062, 13063, 13064, 13065, 13066, 13067, 13068, 13069, 13070, 13071, 13072, 13073, 13074, 13075, 13076, 13077, 13078, 13079, 13080, 13081, 13082, 13083, 13084, 13085, 13086, 13087, 13088, 13089, 13090, 13091, 13092, 13093, 13094, 13095, 13096, 13097, 13098, 13099, 13100, 13101, 13102, 13103, 13104, 13105, 13106, 13107, 13108, 13109, 13110, 13111, 13112, 13113, 13114, 13115, 13116, 13117, 13118, 13119, 13120, 13121, 13122, 13123, 13124, 13125, 13126, 13127, 13128, 13129, 13130, 13131, 13132, 13133, 13134, 13135, 13136, 13137, 13138, 13139, 13140, 13141, 13142, 13143, 13144, 13145, 13146, 13147, 13148, 13149, 13150, 13151, 13152, 13153, 13154, 13155, 13156, 13157, 13158, 13159, 13160, 13161, 13162, 13163, 13164, 13165, 13166, 13167, 13168, 13169, 13170, 13171, 13172, 13173, 13174, 13175, 13176, 13177, 13178, 13179, 13180, 13181, 13182, 13183, 13184, 13185, 13186, 13187, 13188, 13189, 13190, 13191, 13192, 13193, 13194, 13195, 13196, 13197, 13198, 13199, 13200, 13201, 13202, 13203, 13204, 13205, 13206, 13207, 13208, 13209, 13210, 13211, 13212, 13213, 13214, 13215, 13216, 13217, 13218, 13219, 13220, 13221, 13222, 13223, 13224, 13225, 13226, 13227, 13228, 13229, 13230, 13231, 13232, 13233, 13234, 13235, 13236, 13237, 13238, 13239, 13240, 13241, 13242, 13243, 13244, 13245, 13246, 13247, 13248, 13249, 13250, 13251, 13252, 13253, 13254, 13255, 13256, 13257, 13258, 13259, 13260, 13261, 13262, 13263, 13264, 13265, 13266, 13267, 13268, 13269, 13270, 13271, 13272, 13273, 13274, 13275, 13276, 13277, 13278, 13279, 13280, 13281, 13282, 13283, 13284, 13285, 13286, 13287, 13288, 13289, 13290, 13291, 13292, 13293, 13294, 13295, 13296, 13297, 13298, 13299, 13300, 13301, 13302, 13303, 13304, 13305, 13306, 13307, 13308, 13309, 13310, 13311, 19904, 19905, 19906, 19907, 19908, 19909, 19910, 19911, 19912, 19913, 19914, 19915, 19916, 19917, 19918, 19919, 19920, 19921, 19922, 19923, 19924, 19925, 19926, 19927, 19928, 19929, 19930, 19931, 19932, 19933, 19934, 19935, 19936, 19937, 19938, 19939, 19940, 19941, 19942, 19943, 19944, 19945, 19946, 19947, 19948, 19949, 19950, 19951, 19952, 19953, 19954, 19955, 19956, 19957, 19958, 19959, 19960, 19961, 19962, 19963, 19964, 19965, 19966, 19967, 42128, 42129, 42130, 42131, 42132, 42133, 42134, 42135, 42136, 42137, 42138, 42139, 42140, 42141, 42142, 42143, 42144, 42145, 42146, 42147, 42148, 42149, 42150, 42151, 42152, 42153, 42154, 42155, 42156, 42157, 42158, 42159, 42160, 42161, 42162, 42163, 42164, 42165, 42166, 42167, 42168, 42169, 42170, 42171, 42172, 42173, 42174, 42175, 42176, 42177, 42178, 42179, 42180, 42181, 42182, 43048, 43049, 43050, 43051, 43062, 43063, 43065, 43639, 43640, 43641, 65021, 65508, 65512, 65517, 65518, 65532, 65533, 65794, 65847, 65848, 65849, 65850, 65851, 65852, 65853, 65854, 65855, 65913, 65914, 65915, 65916, 65917, 65918, 65919, 65920, 65921, 65922, 65923, 65924, 65925, 65926, 65927, 65928, 65929, 65936, 65937, 65938, 65939, 65940, 65941, 65942, 65943, 65944, 65945, 65946, 65947, 66000, 66001, 66002, 66003, 66004, 66005, 66006, 66007, 66008, 66009, 66010, 66011, 66012, 66013, 66014, 66015, 66016, 66017, 66018, 66019, 66020, 66021, 66022, 66023, 66024, 66025, 66026, 66027, 66028, 66029, 66030, 66031, 66032, 66033, 66034, 66035, 66036, 66037, 66038, 66039, 66040, 66041, 66042, 66043, 66044, 118784, 118785, 118786, 118787, 118788, 118789, 118790, 118791, 118792, 118793, 118794, 118795, 118796, 118797, 118798, 118799, 118800, 118801, 118802, 118803, 118804, 118805, 118806, 118807, 118808, 118809, 118810, 118811, 118812, 118813, 118814, 118815, 118816, 118817, 118818, 118819, 118820, 118821, 118822, 118823, 118824, 118825, 118826, 118827, 118828, 118829, 118830, 118831, 118832, 118833, 118834, 118835, 118836, 118837, 118838, 118839, 118840, 118841, 118842, 118843, 118844, 118845, 118846, 118847, 118848, 118849, 118850, 118851, 118852, 118853, 118854, 118855, 118856, 118857, 118858, 118859, 118860, 118861, 118862, 118863, 118864, 118865, 118866, 118867, 118868, 118869, 118870, 118871, 118872, 118873, 118874, 118875, 118876, 118877, 118878, 118879, 118880, 118881, 118882, 118883, 118884, 118885, 118886, 118887, 118888, 118889, 118890, 118891, 118892, 118893, 118894, 118895, 118896, 118897, 118898, 118899, 118900, 118901, 118902, 118903, 118904, 118905, 118906, 118907, 118908, 118909, 118910, 118911, 118912, 118913, 118914, 118915, 118916, 118917, 118918, 118919, 118920, 118921, 118922, 118923, 118924, 118925, 118926, 118927, 118928, 118929, 118930, 118931, 118932, 118933, 118934, 118935, 118936, 118937, 118938, 118939, 118940, 118941, 118942, 118943, 118944, 118945, 118946, 118947, 118948, 118949, 118950, 118951, 118952, 118953, 118954, 118955, 118956, 118957, 118958, 118959, 118960, 118961, 118962, 118963, 118964, 118965, 118966, 118967, 118968, 118969, 118970, 118971, 118972, 118973, 118974, 118975, 118976, 118977, 118978, 118979, 118980, 118981, 118982, 118983, 118984, 118985, 118986, 118987, 118988, 118989, 118990, 118991, 118992, 118993, 118994, 118995, 118996, 118997, 118998, 118999, 119000, 119001, 119002, 119003, 119004, 119005, 119006, 119007, 119008, 119009, 119010, 119011, 119012, 119013, 119014, 119015, 119016, 119017, 119018, 119019, 119020, 119021, 119022, 119023, 119024, 119025, 119026, 119027, 119028, 119029, 119040, 119041, 119042, 119043, 119044, 119045, 119046, 119047, 119048, 119049, 119050, 119051, 119052, 119053, 119054, 119055, 119056, 119057, 119058, 119059, 119060, 119061, 119062, 119063, 119064, 119065, 119066, 119067, 119068, 119069, 119070, 119071, 119072, 119073, 119074, 119075, 119076, 119077, 119078, 119081, 119082, 119083, 119084, 119085, 119086, 119087, 119088, 119089, 119090, 119091, 119092, 119093, 119094, 119095, 119096, 119097, 119098, 119099, 119100, 119101, 119102, 119103, 119104, 119105, 119106, 119107, 119108, 119109, 119110, 119111, 119112, 119113, 119114, 119115, 119116, 119117, 119118, 119119, 119120, 119121, 119122, 119123, 119124, 119125, 119126, 119127, 119128, 119129, 119130, 119131, 119132, 119133, 119134, 119135, 119136, 119137, 119138, 119139, 119140, 119146, 119147, 119148, 119171, 119172, 119180, 119181, 119182, 119183, 119184, 119185, 119186, 119187, 119188, 119189, 119190, 119191, 119192, 119193, 119194, 119195, 119196, 119197, 119198, 119199, 119200, 119201, 119202, 119203, 119204, 119205, 119206, 119207, 119208, 119209, 119214, 119215, 119216, 119217, 119218, 119219, 119220, 119221, 119222, 119223, 119224, 119225, 119226, 119227, 119228, 119229, 119230, 119231, 119232, 119233, 119234, 119235, 119236, 119237, 119238, 119239, 119240, 119241, 119242, 119243, 119244, 119245, 119246, 119247, 119248, 119249, 119250, 119251, 119252, 119253, 119254, 119255, 119256, 119257, 119258, 119259, 119260, 119261, 119296, 119297, 119298, 119299, 119300, 119301, 119302, 119303, 119304, 119305, 119306, 119307, 119308, 119309, 119310, 119311, 119312, 119313, 119314, 119315, 119316, 119317, 119318, 119319, 119320, 119321, 119322, 119323, 119324, 119325, 119326, 119327, 119328, 119329, 119330, 119331, 119332, 119333, 119334, 119335, 119336, 119337, 119338, 119339, 119340, 119341, 119342, 119343, 119344, 119345, 119346, 119347, 119348, 119349, 119350, 119351, 119352, 119353, 119354, 119355, 119356, 119357, 119358, 119359, 119360, 119361, 119365, 119552, 119553, 119554, 119555, 119556, 119557, 119558, 119559, 119560, 119561, 119562, 119563, 119564, 119565, 119566, 119567, 119568, 119569, 119570, 119571, 119572, 119573, 119574, 119575, 119576, 119577, 119578, 119579, 119580, 119581, 119582, 119583, 119584, 119585, 119586, 119587, 119588, 119589, 119590, 119591, 119592, 119593, 119594, 119595, 119596, 119597, 119598, 119599, 119600, 119601, 119602, 119603, 119604, 119605, 119606, 119607, 119608, 119609, 119610, 119611, 119612, 119613, 119614, 119615, 119616, 119617, 119618, 119619, 119620, 119621, 119622, 119623, 119624, 119625, 119626, 119627, 119628, 119629, 119630, 119631, 119632, 119633, 119634, 119635, 119636, 119637, 119638, 126976, 126977, 126978, 126979, 126980, 126981, 126982, 126983, 126984, 126985, 126986, 126987, 126988, 126989, 126990, 126991, 126992, 126993, 126994, 126995, 126996, 126997, 126998, 126999, 127000, 127001, 127002, 127003, 127004, 127005, 127006, 127007, 127008, 127009, 127010, 127011, 127012, 127013, 127014, 127015, 127016, 127017, 127018, 127019, 127024, 127025, 127026, 127027, 127028, 127029, 127030, 127031, 127032, 127033, 127034, 127035, 127036, 127037, 127038, 127039, 127040, 127041, 127042, 127043, 127044, 127045, 127046, 127047, 127048, 127049, 127050, 127051, 127052, 127053, 127054, 127055, 127056, 127057, 127058, 127059, 127060, 127061, 127062, 127063, 127064, 127065, 127066, 127067, 127068, 127069, 127070, 127071, 127072, 127073, 127074, 127075, 127076, 127077, 127078, 127079, 127080, 127081, 127082, 127083, 127084, 127085, 127086, 127087, 127088, 127089, 127090, 127091, 127092, 127093, 127094, 127095, 127096, 127097, 127098, 127099, 127100, 127101, 127102, 127103, 127104, 127105, 127106, 127107, 127108, 127109, 127110, 127111, 127112, 127113, 127114, 127115, 127116, 127117, 127118, 127119, 127120, 127121, 127122, 127123, 127136, 127137, 127138, 127139, 127140, 127141, 127142, 127143, 127144, 127145, 127146, 127147, 127148, 127149, 127150, 127153, 127154, 127155, 127156, 127157, 127158, 127159, 127160, 127161, 127162, 127163, 127164, 127165, 127166, 127169, 127170, 127171, 127172, 127173, 127174, 127175, 127176, 127177, 127178, 127179, 127180, 127181, 127182, 127183, 127185, 127186, 127187, 127188, 127189, 127190, 127191, 127192, 127193, 127194, 127195, 127196, 127197, 127198, 127199, 127248, 127249, 127250, 127251, 127252, 127253, 127254, 127255, 127256, 127257, 127258, 127259, 127260, 127261, 127262, 127263, 127264, 127265, 127266, 127267, 127268, 127269, 127270, 127271, 127272, 127273, 127274, 127275, 127276, 127277, 127278, 127280, 127281, 127282, 127283, 127284, 127285, 127286, 127287, 127288, 127289, 127290, 127291, 127292, 127293, 127294, 127295, 127296, 127297, 127298, 127299, 127300, 127301, 127302, 127303, 127304, 127305, 127306, 127307, 127308, 127309, 127310, 127311, 127312, 127313, 127314, 127315, 127316, 127317, 127318, 127319, 127320, 127321, 127322, 127323, 127324, 127325, 127326, 127327, 127328, 127329, 127330, 127331, 127332, 127333, 127334, 127335, 127336, 127337, 127344, 127345, 127346, 127347, 127348, 127349, 127350, 127351, 127352, 127353, 127354, 127355, 127356, 127357, 127358, 127359, 127360, 127361, 127362, 127363, 127364, 127365, 127366, 127367, 127368, 127369, 127370, 127371, 127372, 127373, 127374, 127375, 127376, 127377, 127378, 127379, 127380, 127381, 127382, 127383, 127384, 127385, 127386, 127462, 127463, 127464, 127465, 127466, 127467, 127468, 127469, 127470, 127471, 127472, 127473, 127474, 127475, 127476, 127477, 127478, 127479, 127480, 127481, 127482, 127483, 127484, 127485, 127486, 127487, 127488, 127489, 127490, 127504, 127505, 127506, 127507, 127508, 127509, 127510, 127511, 127512, 127513, 127514, 127515, 127516, 127517, 127518, 127519, 127520, 127521, 127522, 127523, 127524, 127525, 127526, 127527, 127528, 127529, 127530, 127531, 127532, 127533, 127534, 127535, 127536, 127537, 127538, 127539, 127540, 127541, 127542, 127543, 127544, 127545, 127546, 127552, 127553, 127554, 127555, 127556, 127557, 127558, 127559, 127560, 127568, 127569, 127744, 127745, 127746, 127747, 127748, 127749, 127750, 127751, 127752, 127753, 127754, 127755, 127756, 127757, 127758, 127759, 127760, 127761, 127762, 127763, 127764, 127765, 127766, 127767, 127768, 127769, 127770, 127771, 127772, 127773, 127774, 127775, 127776, 127792, 127793, 127794, 127795, 127796, 127797, 127799, 127800, 127801, 127802, 127803, 127804, 127805, 127806, 127807, 127808, 127809, 127810, 127811, 127812, 127813, 127814, 127815, 127816, 127817, 127818, 127819, 127820, 127821, 127822, 127823, 127824, 127825, 127826, 127827, 127828, 127829, 127830, 127831, 127832, 127833, 127834, 127835, 127836, 127837, 127838, 127839, 127840, 127841, 127842, 127843, 127844, 127845, 127846, 127847, 127848, 127849, 127850, 127851, 127852, 127853, 127854, 127855, 127856, 127857, 127858, 127859, 127860, 127861, 127862, 127863, 127864, 127865, 127866, 127867, 127868, 127872, 127873, 127874, 127875, 127876, 127877, 127878, 127879, 127880, 127881, 127882, 127883, 127884, 127885, 127886, 127887, 127888, 127889, 127890, 127891, 127904, 127905, 127906, 127907, 127908, 127909, 127910, 127911, 127912, 127913, 127914, 127915, 127916, 127917, 127918, 127919, 127920, 127921, 127922, 127923, 127924, 127925, 127926, 127927, 127928, 127929, 127930, 127931, 127932, 127933, 127934, 127935, 127936, 127937, 127938, 127939, 127940, 127942, 127943, 127944, 127945, 127946, 127968, 127969, 127970, 127971, 127972, 127973, 127974, 127975, 127976, 127977, 127978, 127979, 127980, 127981, 127982, 127983, 127984, 128000, 128001, 128002, 128003, 128004, 128005, 128006, 128007, 128008, 128009, 128010, 128011, 128012, 128013, 128014, 128015, 128016, 128017, 128018, 128019, 128020, 128021, 128022, 128023, 128024, 128025, 128026, 128027, 128028, 128029, 128030, 128031, 128032, 128033, 128034, 128035, 128036, 128037, 128038, 128039, 128040, 128041, 128042, 128043, 128044, 128045, 128046, 128047, 128048, 128049, 128050, 128051, 128052, 128053, 128054, 128055, 128056, 128057, 128058, 128059, 128060, 128061, 128062, 128064, 128066, 128067, 128068, 128069, 128070, 128071, 128072, 128073, 128074, 128075, 128076, 128077, 128078, 128079, 128080, 128081, 128082, 128083, 128084, 128085, 128086, 128087, 128088, 128089, 128090, 128091, 128092, 128093, 128094, 128095, 128096, 128097, 128098, 128099, 128100, 128101, 128102, 128103, 128104, 128105, 128106, 128107, 128108, 128109, 128110, 128111, 128112, 128113, 128114, 128115, 128116, 128117, 128118, 128119, 128120, 128121, 128122, 128123, 128124, 128125, 128126, 128127, 128128, 128129, 128130, 128131, 128132, 128133, 128134, 128135, 128136, 128137, 128138, 128139, 128140, 128141, 128142, 128143, 128144, 128145, 128146, 128147, 128148, 128149, 128150, 128151, 128152, 128153, 128154, 128155, 128156, 128157, 128158, 128159, 128160, 128161, 128162, 128163, 128164, 128165, 128166, 128167, 128168, 128169, 128170, 128171, 128172, 128173, 128174, 128175, 128176, 128177, 128178, 128179, 128180, 128181, 128182, 128183, 128184, 128185, 128186, 128187, 128188, 128189, 128190, 128191, 128192, 128193, 128194, 128195, 128196, 128197, 128198, 128199, 128200, 128201, 128202, 128203, 128204, 128205, 128206, 128207, 128208, 128209, 128210, 128211, 128212, 128213, 128214, 128215, 128216, 128217, 128218, 128219, 128220, 128221, 128222, 128223, 128224, 128225, 128226, 128227, 128228, 128229, 128230, 128231, 128232, 128233, 128234, 128235, 128236, 128237, 128238, 128239, 128240, 128241, 128242, 128243, 128244, 128245, 128246, 128247, 128249, 128250, 128251, 128252, 128256, 128257, 128258, 128259, 128260, 128261, 128262, 128263, 128264, 128265, 128266, 128267, 128268, 128269, 128270, 128271, 128272, 128273, 128274, 128275, 128276, 128277, 128278, 128279, 128280, 128281, 128282, 128283, 128284, 128285, 128286, 128287, 128288, 128289, 128290, 128291, 128292, 128293, 128294, 128295, 128296, 128297, 128298, 128299, 128300, 128301, 128302, 128303, 128304, 128305, 128306, 128307, 128308, 128309, 128310, 128311, 128312, 128313, 128314, 128315, 128316, 128317, 128336, 128337, 128338, 128339, 128340, 128341, 128342, 128343, 128344, 128345, 128346, 128347, 128348, 128349, 128350, 128351, 128352, 128353, 128354, 128355, 128356, 128357, 128358, 128359, 128507, 128508, 128509, 128510, 128511, 128513, 128514, 128515, 128516, 128517, 128518, 128519, 128520, 128521, 128522, 128523, 128524, 128525, 128526, 128527, 128528, 128530, 128531, 128532, 128534, 128536, 128538, 128540, 128541, 128542, 128544, 128545, 128546, 128547, 128548, 128549, 128552, 128553, 128554, 128555, 128557, 128560, 128561, 128562, 128563, 128565, 128566, 128567, 128568, 128569, 128570, 128571, 128572, 128573, 128574, 128575, 128576, 128581, 128582, 128583, 128584, 128585, 128586, 128587, 128588, 128589, 128590, 128591, 128640, 128641, 128642, 128643, 128644, 128645, 128646, 128647, 128648, 128649, 128650, 128651, 128652, 128653, 128654, 128655, 128656, 128657, 128658, 128659, 128660, 128661, 128662, 128663, 128664, 128665, 128666, 128667, 128668, 128669, 128670, 128671, 128672, 128673, 128674, 128675, 128676, 128677, 128678, 128679, 128680, 128681, 128682, 128683, 128684, 128685, 128686, 128687, 128688, 128689, 128690, 128691, 128692, 128693, 128694, 128695, 128696, 128697, 128698, 128699, 128700, 128701, 128702, 128703, 128704, 128705, 128706, 128707, 128708, 128709, 128768, 128769, 128770, 128771, 128772, 128773, 128774, 128775, 128776, 128777, 128778, 128779, 128780, 128781, 128782, 128783, 128784, 128785, 128786, 128787, 128788, 128789, 128790, 128791, 128792, 128793, 128794, 128795, 128796, 128797, 128798, 128799, 128800, 128801, 128802, 128803, 128804, 128805, 128806, 128807, 128808, 128809, 128810, 128811, 128812, 128813, 128814, 128815, 128816, 128817, 128818, 128819, 128820, 128821, 128822, 128823, 128824, 128825, 128826, 128827, 128828, 128829, 128830, 128831, 128832, 128833, 128834, 128835, 128836, 128837, 128838, 128839, 128840, 128841, 128842, 128843, 128844, 128845, 128846, 128847, 128848, 128849, 128850, 128851, 128852, 128853, 128854, 128855, 128856, 128857, 128858, 128859, 128860, 128861, 128862, 128863, 128864, 128865, 128866, 128867, 128868, 128869, 128870, 128871, 128872, 128873, 128874, 128875, 128876, 128877, 128878, 128879, 128880, 128881, 128882, 128883],
				'encoded': '\xA6\xA7\xA9\xAE\xB0\xB6\u0482\u060E\u060F\u06DE\u06E9\u06FD\u06FE\u07F6\u09FA\u0B70\u0BF3\u0BF4\u0BF5\u0BF6\u0BF7\u0BF8\u0BFA\u0C7F\u0D79\u0F01\u0F02\u0F03\u0F13\u0F14\u0F15\u0F16\u0F17\u0F1A\u0F1B\u0F1C\u0F1D\u0F1E\u0F1F\u0F34\u0F36\u0F38\u0FBE\u0FBF\u0FC0\u0FC1\u0FC2\u0FC3\u0FC4\u0FC5\u0FC7\u0FC8\u0FC9\u0FCA\u0FCB\u0FCC\u0FCE\u0FCF\u0FD5\u0FD6\u0FD7\u0FD8\u109E\u109F\u1360\u1390\u1391\u1392\u1393\u1394\u1395\u1396\u1397\u1398\u1399\u1940\u19DE\u19DF\u19E0\u19E1\u19E2\u19E3\u19E4\u19E5\u19E6\u19E7\u19E8\u19E9\u19EA\u19EB\u19EC\u19ED\u19EE\u19EF\u19F0\u19F1\u19F2\u19F3\u19F4\u19F5\u19F6\u19F7\u19F8\u19F9\u19FA\u19FB\u19FC\u19FD\u19FE\u19FF\u1B61\u1B62\u1B63\u1B64\u1B65\u1B66\u1B67\u1B68\u1B69\u1B6A\u1B74\u1B75\u1B76\u1B77\u1B78\u1B79\u1B7A\u1B7B\u1B7C\u2100\u2101\u2103\u2104\u2105\u2106\u2108\u2109\u2114\u2116\u2117\u211E\u211F\u2120\u2121\u2122\u2123\u2125\u2127\u2129\u212E\u213A\u213B\u214A\u214C\u214D\u214F\u2195\u2196\u2197\u2198\u2199\u219C\u219D\u219E\u219F\u21A1\u21A2\u21A4\u21A5\u21A7\u21A8\u21A9\u21AA\u21AB\u21AC\u21AD\u21AF\u21B0\u21B1\u21B2\u21B3\u21B4\u21B5\u21B6\u21B7\u21B8\u21B9\u21BA\u21BB\u21BC\u21BD\u21BE\u21BF\u21C0\u21C1\u21C2\u21C3\u21C4\u21C5\u21C6\u21C7\u21C8\u21C9\u21CA\u21CB\u21CC\u21CD\u21D0\u21D1\u21D3\u21D5\u21D6\u21D7\u21D8\u21D9\u21DA\u21DB\u21DC\u21DD\u21DE\u21DF\u21E0\u21E1\u21E2\u21E3\u21E4\u21E5\u21E6\u21E7\u21E8\u21E9\u21EA\u21EB\u21EC\u21ED\u21EE\u21EF\u21F0\u21F1\u21F2\u21F3\u2300\u2301\u2302\u2303\u2304\u2305\u2306\u2307\u230C\u230D\u230E\u230F\u2310\u2311\u2312\u2313\u2314\u2315\u2316\u2317\u2318\u2319\u231A\u231B\u231C\u231D\u231E\u231F\u2322\u2323\u2324\u2325\u2326\u2327\u2328\u232B\u232C\u232D\u232E\u232F\u2330\u2331\u2332\u2333\u2334\u2335\u2336\u2337\u2338\u2339\u233A\u233B\u233C\u233D\u233E\u233F\u2340\u2341\u2342\u2343\u2344\u2345\u2346\u2347\u2348\u2349\u234A\u234B\u234C\u234D\u234E\u234F\u2350\u2351\u2352\u2353\u2354\u2355\u2356\u2357\u2358\u2359\u235A\u235B\u235C\u235D\u235E\u235F\u2360\u2361\u2362\u2363\u2364\u2365\u2366\u2367\u2368\u2369\u236A\u236B\u236C\u236D\u236E\u236F\u2370\u2371\u2372\u2373\u2374\u2375\u2376\u2377\u2378\u2379\u237A\u237B\u237D\u237E\u237F\u2380\u2381\u2382\u2383\u2384\u2385\u2386\u2387\u2388\u2389\u238A\u238B\u238C\u238D\u238E\u238F\u2390\u2391\u2392\u2393\u2394\u2395\u2396\u2397\u2398\u2399\u239A\u23B4\u23B5\u23B6\u23B7\u23B8\u23B9\u23BA\u23BB\u23BC\u23BD\u23BE\u23BF\u23C0\u23C1\u23C2\u23C3\u23C4\u23C5\u23C6\u23C7\u23C8\u23C9\u23CA\u23CB\u23CC\u23CD\u23CE\u23CF\u23D0\u23D1\u23D2\u23D3\u23D4\u23D5\u23D6\u23D7\u23D8\u23D9\u23DA\u23DB\u23E2\u23E3\u23E4\u23E5\u23E6\u23E7\u23E8\u23E9\u23EA\u23EB\u23EC\u23ED\u23EE\u23EF\u23F0\u23F1\u23F2\u23F3\u2400\u2401\u2402\u2403\u2404\u2405\u2406\u2407\u2408\u2409\u240A\u240B\u240C\u240D\u240E\u240F\u2410\u2411\u2412\u2413\u2414\u2415\u2416\u2417\u2418\u2419\u241A\u241B\u241C\u241D\u241E\u241F\u2420\u2421\u2422\u2423\u2424\u2425\u2426\u2440\u2441\u2442\u2443\u2444\u2445\u2446\u2447\u2448\u2449\u244A\u249C\u249D\u249E\u249F\u24A0\u24A1\u24A2\u24A3\u24A4\u24A5\u24A6\u24A7\u24A8\u24A9\u24AA\u24AB\u24AC\u24AD\u24AE\u24AF\u24B0\u24B1\u24B2\u24B3\u24B4\u24B5\u24B6\u24B7\u24B8\u24B9\u24BA\u24BB\u24BC\u24BD\u24BE\u24BF\u24C0\u24C1\u24C2\u24C3\u24C4\u24C5\u24C6\u24C7\u24C8\u24C9\u24CA\u24CB\u24CC\u24CD\u24CE\u24CF\u24D0\u24D1\u24D2\u24D3\u24D4\u24D5\u24D6\u24D7\u24D8\u24D9\u24DA\u24DB\u24DC\u24DD\u24DE\u24DF\u24E0\u24E1\u24E2\u24E3\u24E4\u24E5\u24E6\u24E7\u24E8\u24E9\u2500\u2501\u2502\u2503\u2504\u2505\u2506\u2507\u2508\u2509\u250A\u250B\u250C\u250D\u250E\u250F\u2510\u2511\u2512\u2513\u2514\u2515\u2516\u2517\u2518\u2519\u251A\u251B\u251C\u251D\u251E\u251F\u2520\u2521\u2522\u2523\u2524\u2525\u2526\u2527\u2528\u2529\u252A\u252B\u252C\u252D\u252E\u252F\u2530\u2531\u2532\u2533\u2534\u2535\u2536\u2537\u2538\u2539\u253A\u253B\u253C\u253D\u253E\u253F\u2540\u2541\u2542\u2543\u2544\u2545\u2546\u2547\u2548\u2549\u254A\u254B\u254C\u254D\u254E\u254F\u2550\u2551\u2552\u2553\u2554\u2555\u2556\u2557\u2558\u2559\u255A\u255B\u255C\u255D\u255E\u255F\u2560\u2561\u2562\u2563\u2564\u2565\u2566\u2567\u2568\u2569\u256A\u256B\u256C\u256D\u256E\u256F\u2570\u2571\u2572\u2573\u2574\u2575\u2576\u2577\u2578\u2579\u257A\u257B\u257C\u257D\u257E\u257F\u2580\u2581\u2582\u2583\u2584\u2585\u2586\u2587\u2588\u2589\u258A\u258B\u258C\u258D\u258E\u258F\u2590\u2591\u2592\u2593\u2594\u2595\u2596\u2597\u2598\u2599\u259A\u259B\u259C\u259D\u259E\u259F\u25A0\u25A1\u25A2\u25A3\u25A4\u25A5\u25A6\u25A7\u25A8\u25A9\u25AA\u25AB\u25AC\u25AD\u25AE\u25AF\u25B0\u25B1\u25B2\u25B3\u25B4\u25B5\u25B6\u25B8\u25B9\u25BA\u25BB\u25BC\u25BD\u25BE\u25BF\u25C0\u25C2\u25C3\u25C4\u25C5\u25C6\u25C7\u25C8\u25C9\u25CA\u25CB\u25CC\u25CD\u25CE\u25CF\u25D0\u25D1\u25D2\u25D3\u25D4\u25D5\u25D6\u25D7\u25D8\u25D9\u25DA\u25DB\u25DC\u25DD\u25DE\u25DF\u25E0\u25E1\u25E2\u25E3\u25E4\u25E5\u25E6\u25E7\u25E8\u25E9\u25EA\u25EB\u25EC\u25ED\u25EE\u25EF\u25F0\u25F1\u25F2\u25F3\u25F4\u25F5\u25F6\u25F7\u2600\u2601\u2602\u2603\u2604\u2605\u2606\u2607\u2608\u2609\u260A\u260B\u260C\u260D\u260E\u260F\u2610\u2611\u2612\u2613\u2614\u2615\u2616\u2617\u2618\u2619\u261A\u261B\u261C\u261D\u261E\u261F\u2620\u2621\u2622\u2623\u2624\u2625\u2626\u2627\u2628\u2629\u262A\u262B\u262C\u262D\u262E\u262F\u2630\u2631\u2632\u2633\u2634\u2635\u2636\u2637\u2638\u2639\u263A\u263B\u263C\u263D\u263E\u263F\u2640\u2641\u2642\u2643\u2644\u2645\u2646\u2647\u2648\u2649\u264A\u264B\u264C\u264D\u264E\u264F\u2650\u2651\u2652\u2653\u2654\u2655\u2656\u2657\u2658\u2659\u265A\u265B\u265C\u265D\u265E\u265F\u2660\u2661\u2662\u2663\u2664\u2665\u2666\u2667\u2668\u2669\u266A\u266B\u266C\u266D\u266E\u2670\u2671\u2672\u2673\u2674\u2675\u2676\u2677\u2678\u2679\u267A\u267B\u267C\u267D\u267E\u267F\u2680\u2681\u2682\u2683\u2684\u2685\u2686\u2687\u2688\u2689\u268A\u268B\u268C\u268D\u268E\u268F\u2690\u2691\u2692\u2693\u2694\u2695\u2696\u2697\u2698\u2699\u269A\u269B\u269C\u269D\u269E\u269F\u26A0\u26A1\u26A2\u26A3\u26A4\u26A5\u26A6\u26A7\u26A8\u26A9\u26AA\u26AB\u26AC\u26AD\u26AE\u26AF\u26B0\u26B1\u26B2\u26B3\u26B4\u26B5\u26B6\u26B7\u26B8\u26B9\u26BA\u26BB\u26BC\u26BD\u26BE\u26BF\u26C0\u26C1\u26C2\u26C3\u26C4\u26C5\u26C6\u26C7\u26C8\u26C9\u26CA\u26CB\u26CC\u26CD\u26CE\u26CF\u26D0\u26D1\u26D2\u26D3\u26D4\u26D5\u26D6\u26D7\u26D8\u26D9\u26DA\u26DB\u26DC\u26DD\u26DE\u26DF\u26E0\u26E1\u26E2\u26E3\u26E4\u26E5\u26E6\u26E7\u26E8\u26E9\u26EA\u26EB\u26EC\u26ED\u26EE\u26EF\u26F0\u26F1\u26F2\u26F3\u26F4\u26F5\u26F6\u26F7\u26F8\u26F9\u26FA\u26FB\u26FC\u26FD\u26FE\u26FF\u2701\u2702\u2703\u2704\u2705\u2706\u2707\u2708\u2709\u270A\u270B\u270C\u270D\u270E\u270F\u2710\u2711\u2712\u2713\u2714\u2715\u2716\u2717\u2718\u2719\u271A\u271B\u271C\u271D\u271E\u271F\u2720\u2721\u2722\u2723\u2724\u2725\u2726\u2727\u2728\u2729\u272A\u272B\u272C\u272D\u272E\u272F\u2730\u2731\u2732\u2733\u2734\u2735\u2736\u2737\u2738\u2739\u273A\u273B\u273C\u273D\u273E\u273F\u2740\u2741\u2742\u2743\u2744\u2745\u2746\u2747\u2748\u2749\u274A\u274B\u274C\u274D\u274E\u274F\u2750\u2751\u2752\u2753\u2754\u2755\u2756\u2757\u2758\u2759\u275A\u275B\u275C\u275D\u275E\u275F\u2760\u2761\u2762\u2763\u2764\u2765\u2766\u2767\u2794\u2795\u2796\u2797\u2798\u2799\u279A\u279B\u279C\u279D\u279E\u279F\u27A0\u27A1\u27A2\u27A3\u27A4\u27A5\u27A6\u27A7\u27A8\u27A9\u27AA\u27AB\u27AC\u27AD\u27AE\u27AF\u27B0\u27B1\u27B2\u27B3\u27B4\u27B5\u27B6\u27B7\u27B8\u27B9\u27BA\u27BB\u27BC\u27BD\u27BE\u27BF\u2800\u2801\u2802\u2803\u2804\u2805\u2806\u2807\u2808\u2809\u280A\u280B\u280C\u280D\u280E\u280F\u2810\u2811\u2812\u2813\u2814\u2815\u2816\u2817\u2818\u2819\u281A\u281B\u281C\u281D\u281E\u281F\u2820\u2821\u2822\u2823\u2824\u2825\u2826\u2827\u2828\u2829\u282A\u282B\u282C\u282D\u282E\u282F\u2830\u2831\u2832\u2833\u2834\u2835\u2836\u2837\u2838\u2839\u283A\u283B\u283C\u283D\u283E\u283F\u2840\u2841\u2842\u2843\u2844\u2845\u2846\u2847\u2848\u2849\u284A\u284B\u284C\u284D\u284E\u284F\u2850\u2851\u2852\u2853\u2854\u2855\u2856\u2857\u2858\u2859\u285A\u285B\u285C\u285D\u285E\u285F\u2860\u2861\u2862\u2863\u2864\u2865\u2866\u2867\u2868\u2869\u286A\u286B\u286C\u286D\u286E\u286F\u2870\u2871\u2872\u2873\u2874\u2875\u2876\u2877\u2878\u2879\u287A\u287B\u287C\u287D\u287E\u287F\u2880\u2881\u2882\u2883\u2884\u2885\u2886\u2887\u2888\u2889\u288A\u288B\u288C\u288D\u288E\u288F\u2890\u2891\u2892\u2893\u2894\u2895\u2896\u2897\u2898\u2899\u289A\u289B\u289C\u289D\u289E\u289F\u28A0\u28A1\u28A2\u28A3\u28A4\u28A5\u28A6\u28A7\u28A8\u28A9\u28AA\u28AB\u28AC\u28AD\u28AE\u28AF\u28B0\u28B1\u28B2\u28B3\u28B4\u28B5\u28B6\u28B7\u28B8\u28B9\u28BA\u28BB\u28BC\u28BD\u28BE\u28BF\u28C0\u28C1\u28C2\u28C3\u28C4\u28C5\u28C6\u28C7\u28C8\u28C9\u28CA\u28CB\u28CC\u28CD\u28CE\u28CF\u28D0\u28D1\u28D2\u28D3\u28D4\u28D5\u28D6\u28D7\u28D8\u28D9\u28DA\u28DB\u28DC\u28DD\u28DE\u28DF\u28E0\u28E1\u28E2\u28E3\u28E4\u28E5\u28E6\u28E7\u28E8\u28E9\u28EA\u28EB\u28EC\u28ED\u28EE\u28EF\u28F0\u28F1\u28F2\u28F3\u28F4\u28F5\u28F6\u28F7\u28F8\u28F9\u28FA\u28FB\u28FC\u28FD\u28FE\u28FF\u2B00\u2B01\u2B02\u2B03\u2B04\u2B05\u2B06\u2B07\u2B08\u2B09\u2B0A\u2B0B\u2B0C\u2B0D\u2B0E\u2B0F\u2B10\u2B11\u2B12\u2B13\u2B14\u2B15\u2B16\u2B17\u2B18\u2B19\u2B1A\u2B1B\u2B1C\u2B1D\u2B1E\u2B1F\u2B20\u2B21\u2B22\u2B23\u2B24\u2B25\u2B26\u2B27\u2B28\u2B29\u2B2A\u2B2B\u2B2C\u2B2D\u2B2E\u2B2F\u2B45\u2B46\u2B50\u2B51\u2B52\u2B53\u2B54\u2B55\u2B56\u2B57\u2B58\u2B59\u2CE5\u2CE6\u2CE7\u2CE8\u2CE9\u2CEA\u2E80\u2E81\u2E82\u2E83\u2E84\u2E85\u2E86\u2E87\u2E88\u2E89\u2E8A\u2E8B\u2E8C\u2E8D\u2E8E\u2E8F\u2E90\u2E91\u2E92\u2E93\u2E94\u2E95\u2E96\u2E97\u2E98\u2E99\u2E9B\u2E9C\u2E9D\u2E9E\u2E9F\u2EA0\u2EA1\u2EA2\u2EA3\u2EA4\u2EA5\u2EA6\u2EA7\u2EA8\u2EA9\u2EAA\u2EAB\u2EAC\u2EAD\u2EAE\u2EAF\u2EB0\u2EB1\u2EB2\u2EB3\u2EB4\u2EB5\u2EB6\u2EB7\u2EB8\u2EB9\u2EBA\u2EBB\u2EBC\u2EBD\u2EBE\u2EBF\u2EC0\u2EC1\u2EC2\u2EC3\u2EC4\u2EC5\u2EC6\u2EC7\u2EC8\u2EC9\u2ECA\u2ECB\u2ECC\u2ECD\u2ECE\u2ECF\u2ED0\u2ED1\u2ED2\u2ED3\u2ED4\u2ED5\u2ED6\u2ED7\u2ED8\u2ED9\u2EDA\u2EDB\u2EDC\u2EDD\u2EDE\u2EDF\u2EE0\u2EE1\u2EE2\u2EE3\u2EE4\u2EE5\u2EE6\u2EE7\u2EE8\u2EE9\u2EEA\u2EEB\u2EEC\u2EED\u2EEE\u2EEF\u2EF0\u2EF1\u2EF2\u2EF3\u2F00\u2F01\u2F02\u2F03\u2F04\u2F05\u2F06\u2F07\u2F08\u2F09\u2F0A\u2F0B\u2F0C\u2F0D\u2F0E\u2F0F\u2F10\u2F11\u2F12\u2F13\u2F14\u2F15\u2F16\u2F17\u2F18\u2F19\u2F1A\u2F1B\u2F1C\u2F1D\u2F1E\u2F1F\u2F20\u2F21\u2F22\u2F23\u2F24\u2F25\u2F26\u2F27\u2F28\u2F29\u2F2A\u2F2B\u2F2C\u2F2D\u2F2E\u2F2F\u2F30\u2F31\u2F32\u2F33\u2F34\u2F35\u2F36\u2F37\u2F38\u2F39\u2F3A\u2F3B\u2F3C\u2F3D\u2F3E\u2F3F\u2F40\u2F41\u2F42\u2F43\u2F44\u2F45\u2F46\u2F47\u2F48\u2F49\u2F4A\u2F4B\u2F4C\u2F4D\u2F4E\u2F4F\u2F50\u2F51\u2F52\u2F53\u2F54\u2F55\u2F56\u2F57\u2F58\u2F59\u2F5A\u2F5B\u2F5C\u2F5D\u2F5E\u2F5F\u2F60\u2F61\u2F62\u2F63\u2F64\u2F65\u2F66\u2F67\u2F68\u2F69\u2F6A\u2F6B\u2F6C\u2F6D\u2F6E\u2F6F\u2F70\u2F71\u2F72\u2F73\u2F74\u2F75\u2F76\u2F77\u2F78\u2F79\u2F7A\u2F7B\u2F7C\u2F7D\u2F7E\u2F7F\u2F80\u2F81\u2F82\u2F83\u2F84\u2F85\u2F86\u2F87\u2F88\u2F89\u2F8A\u2F8B\u2F8C\u2F8D\u2F8E\u2F8F\u2F90\u2F91\u2F92\u2F93\u2F94\u2F95\u2F96\u2F97\u2F98\u2F99\u2F9A\u2F9B\u2F9C\u2F9D\u2F9E\u2F9F\u2FA0\u2FA1\u2FA2\u2FA3\u2FA4\u2FA5\u2FA6\u2FA7\u2FA8\u2FA9\u2FAA\u2FAB\u2FAC\u2FAD\u2FAE\u2FAF\u2FB0\u2FB1\u2FB2\u2FB3\u2FB4\u2FB5\u2FB6\u2FB7\u2FB8\u2FB9\u2FBA\u2FBB\u2FBC\u2FBD\u2FBE\u2FBF\u2FC0\u2FC1\u2FC2\u2FC3\u2FC4\u2FC5\u2FC6\u2FC7\u2FC8\u2FC9\u2FCA\u2FCB\u2FCC\u2FCD\u2FCE\u2FCF\u2FD0\u2FD1\u2FD2\u2FD3\u2FD4\u2FD5\u2FF0\u2FF1\u2FF2\u2FF3\u2FF4\u2FF5\u2FF6\u2FF7\u2FF8\u2FF9\u2FFA\u2FFB\u3004\u3012\u3013\u3020\u3036\u3037\u303E\u303F\u3190\u3191\u3196\u3197\u3198\u3199\u319A\u319B\u319C\u319D\u319E\u319F\u31C0\u31C1\u31C2\u31C3\u31C4\u31C5\u31C6\u31C7\u31C8\u31C9\u31CA\u31CB\u31CC\u31CD\u31CE\u31CF\u31D0\u31D1\u31D2\u31D3\u31D4\u31D5\u31D6\u31D7\u31D8\u31D9\u31DA\u31DB\u31DC\u31DD\u31DE\u31DF\u31E0\u31E1\u31E2\u31E3\u3200\u3201\u3202\u3203\u3204\u3205\u3206\u3207\u3208\u3209\u320A\u320B\u320C\u320D\u320E\u320F\u3210\u3211\u3212\u3213\u3214\u3215\u3216\u3217\u3218\u3219\u321A\u321B\u321C\u321D\u321E\u322A\u322B\u322C\u322D\u322E\u322F\u3230\u3231\u3232\u3233\u3234\u3235\u3236\u3237\u3238\u3239\u323A\u323B\u323C\u323D\u323E\u323F\u3240\u3241\u3242\u3243\u3244\u3245\u3246\u3247\u3248\u3249\u324A\u324B\u324C\u324D\u324E\u324F\u3250\u3260\u3261\u3262\u3263\u3264\u3265\u3266\u3267\u3268\u3269\u326A\u326B\u326C\u326D\u326E\u326F\u3270\u3271\u3272\u3273\u3274\u3275\u3276\u3277\u3278\u3279\u327A\u327B\u327C\u327D\u327E\u327F\u328A\u328B\u328C\u328D\u328E\u328F\u3290\u3291\u3292\u3293\u3294\u3295\u3296\u3297\u3298\u3299\u329A\u329B\u329C\u329D\u329E\u329F\u32A0\u32A1\u32A2\u32A3\u32A4\u32A5\u32A6\u32A7\u32A8\u32A9\u32AA\u32AB\u32AC\u32AD\u32AE\u32AF\u32B0\u32C0\u32C1\u32C2\u32C3\u32C4\u32C5\u32C6\u32C7\u32C8\u32C9\u32CA\u32CB\u32CC\u32CD\u32CE\u32CF\u32D0\u32D1\u32D2\u32D3\u32D4\u32D5\u32D6\u32D7\u32D8\u32D9\u32DA\u32DB\u32DC\u32DD\u32DE\u32DF\u32E0\u32E1\u32E2\u32E3\u32E4\u32E5\u32E6\u32E7\u32E8\u32E9\u32EA\u32EB\u32EC\u32ED\u32EE\u32EF\u32F0\u32F1\u32F2\u32F3\u32F4\u32F5\u32F6\u32F7\u32F8\u32F9\u32FA\u32FB\u32FC\u32FD\u32FE\u3300\u3301\u3302\u3303\u3304\u3305\u3306\u3307\u3308\u3309\u330A\u330B\u330C\u330D\u330E\u330F\u3310\u3311\u3312\u3313\u3314\u3315\u3316\u3317\u3318\u3319\u331A\u331B\u331C\u331D\u331E\u331F\u3320\u3321\u3322\u3323\u3324\u3325\u3326\u3327\u3328\u3329\u332A\u332B\u332C\u332D\u332E\u332F\u3330\u3331\u3332\u3333\u3334\u3335\u3336\u3337\u3338\u3339\u333A\u333B\u333C\u333D\u333E\u333F\u3340\u3341\u3342\u3343\u3344\u3345\u3346\u3347\u3348\u3349\u334A\u334B\u334C\u334D\u334E\u334F\u3350\u3351\u3352\u3353\u3354\u3355\u3356\u3357\u3358\u3359\u335A\u335B\u335C\u335D\u335E\u335F\u3360\u3361\u3362\u3363\u3364\u3365\u3366\u3367\u3368\u3369\u336A\u336B\u336C\u336D\u336E\u336F\u3370\u3371\u3372\u3373\u3374\u3375\u3376\u3377\u3378\u3379\u337A\u337B\u337C\u337D\u337E\u337F\u3380\u3381\u3382\u3383\u3384\u3385\u3386\u3387\u3388\u3389\u338A\u338B\u338C\u338D\u338E\u338F\u3390\u3391\u3392\u3393\u3394\u3395\u3396\u3397\u3398\u3399\u339A\u339B\u339C\u339D\u339E\u339F\u33A0\u33A1\u33A2\u33A3\u33A4\u33A5\u33A6\u33A7\u33A8\u33A9\u33AA\u33AB\u33AC\u33AD\u33AE\u33AF\u33B0\u33B1\u33B2\u33B3\u33B4\u33B5\u33B6\u33B7\u33B8\u33B9\u33BA\u33BB\u33BC\u33BD\u33BE\u33BF\u33C0\u33C1\u33C2\u33C3\u33C4\u33C5\u33C6\u33C7\u33C8\u33C9\u33CA\u33CB\u33CC\u33CD\u33CE\u33CF\u33D0\u33D1\u33D2\u33D3\u33D4\u33D5\u33D6\u33D7\u33D8\u33D9\u33DA\u33DB\u33DC\u33DD\u33DE\u33DF\u33E0\u33E1\u33E2\u33E3\u33E4\u33E5\u33E6\u33E7\u33E8\u33E9\u33EA\u33EB\u33EC\u33ED\u33EE\u33EF\u33F0\u33F1\u33F2\u33F3\u33F4\u33F5\u33F6\u33F7\u33F8\u33F9\u33FA\u33FB\u33FC\u33FD\u33FE\u33FF\u4DC0\u4DC1\u4DC2\u4DC3\u4DC4\u4DC5\u4DC6\u4DC7\u4DC8\u4DC9\u4DCA\u4DCB\u4DCC\u4DCD\u4DCE\u4DCF\u4DD0\u4DD1\u4DD2\u4DD3\u4DD4\u4DD5\u4DD6\u4DD7\u4DD8\u4DD9\u4DDA\u4DDB\u4DDC\u4DDD\u4DDE\u4DDF\u4DE0\u4DE1\u4DE2\u4DE3\u4DE4\u4DE5\u4DE6\u4DE7\u4DE8\u4DE9\u4DEA\u4DEB\u4DEC\u4DED\u4DEE\u4DEF\u4DF0\u4DF1\u4DF2\u4DF3\u4DF4\u4DF5\u4DF6\u4DF7\u4DF8\u4DF9\u4DFA\u4DFB\u4DFC\u4DFD\u4DFE\u4DFF\uA490\uA491\uA492\uA493\uA494\uA495\uA496\uA497\uA498\uA499\uA49A\uA49B\uA49C\uA49D\uA49E\uA49F\uA4A0\uA4A1\uA4A2\uA4A3\uA4A4\uA4A5\uA4A6\uA4A7\uA4A8\uA4A9\uA4AA\uA4AB\uA4AC\uA4AD\uA4AE\uA4AF\uA4B0\uA4B1\uA4B2\uA4B3\uA4B4\uA4B5\uA4B6\uA4B7\uA4B8\uA4B9\uA4BA\uA4BB\uA4BC\uA4BD\uA4BE\uA4BF\uA4C0\uA4C1\uA4C2\uA4C3\uA4C4\uA4C5\uA4C6\uA828\uA829\uA82A\uA82B\uA836\uA837\uA839\uAA77\uAA78\uAA79\uFDFD\uFFE4\uFFE8\uFFED\uFFEE\uFFFC\uFFFD\uD800\uDD02\uD800\uDD37\uD800\uDD38\uD800\uDD39\uD800\uDD3A\uD800\uDD3B\uD800\uDD3C\uD800\uDD3D\uD800\uDD3E\uD800\uDD3F\uD800\uDD79\uD800\uDD7A\uD800\uDD7B\uD800\uDD7C\uD800\uDD7D\uD800\uDD7E\uD800\uDD7F\uD800\uDD80\uD800\uDD81\uD800\uDD82\uD800\uDD83\uD800\uDD84\uD800\uDD85\uD800\uDD86\uD800\uDD87\uD800\uDD88\uD800\uDD89\uD800\uDD90\uD800\uDD91\uD800\uDD92\uD800\uDD93\uD800\uDD94\uD800\uDD95\uD800\uDD96\uD800\uDD97\uD800\uDD98\uD800\uDD99\uD800\uDD9A\uD800\uDD9B\uD800\uDDD0\uD800\uDDD1\uD800\uDDD2\uD800\uDDD3\uD800\uDDD4\uD800\uDDD5\uD800\uDDD6\uD800\uDDD7\uD800\uDDD8\uD800\uDDD9\uD800\uDDDA\uD800\uDDDB\uD800\uDDDC\uD800\uDDDD\uD800\uDDDE\uD800\uDDDF\uD800\uDDE0\uD800\uDDE1\uD800\uDDE2\uD800\uDDE3\uD800\uDDE4\uD800\uDDE5\uD800\uDDE6\uD800\uDDE7\uD800\uDDE8\uD800\uDDE9\uD800\uDDEA\uD800\uDDEB\uD800\uDDEC\uD800\uDDED\uD800\uDDEE\uD800\uDDEF\uD800\uDDF0\uD800\uDDF1\uD800\uDDF2\uD800\uDDF3\uD800\uDDF4\uD800\uDDF5\uD800\uDDF6\uD800\uDDF7\uD800\uDDF8\uD800\uDDF9\uD800\uDDFA\uD800\uDDFB\uD800\uDDFC\uD834\uDC00\uD834\uDC01\uD834\uDC02\uD834\uDC03\uD834\uDC04\uD834\uDC05\uD834\uDC06\uD834\uDC07\uD834\uDC08\uD834\uDC09\uD834\uDC0A\uD834\uDC0B\uD834\uDC0C\uD834\uDC0D\uD834\uDC0E\uD834\uDC0F\uD834\uDC10\uD834\uDC11\uD834\uDC12\uD834\uDC13\uD834\uDC14\uD834\uDC15\uD834\uDC16\uD834\uDC17\uD834\uDC18\uD834\uDC19\uD834\uDC1A\uD834\uDC1B\uD834\uDC1C\uD834\uDC1D\uD834\uDC1E\uD834\uDC1F\uD834\uDC20\uD834\uDC21\uD834\uDC22\uD834\uDC23\uD834\uDC24\uD834\uDC25\uD834\uDC26\uD834\uDC27\uD834\uDC28\uD834\uDC29\uD834\uDC2A\uD834\uDC2B\uD834\uDC2C\uD834\uDC2D\uD834\uDC2E\uD834\uDC2F\uD834\uDC30\uD834\uDC31\uD834\uDC32\uD834\uDC33\uD834\uDC34\uD834\uDC35\uD834\uDC36\uD834\uDC37\uD834\uDC38\uD834\uDC39\uD834\uDC3A\uD834\uDC3B\uD834\uDC3C\uD834\uDC3D\uD834\uDC3E\uD834\uDC3F\uD834\uDC40\uD834\uDC41\uD834\uDC42\uD834\uDC43\uD834\uDC44\uD834\uDC45\uD834\uDC46\uD834\uDC47\uD834\uDC48\uD834\uDC49\uD834\uDC4A\uD834\uDC4B\uD834\uDC4C\uD834\uDC4D\uD834\uDC4E\uD834\uDC4F\uD834\uDC50\uD834\uDC51\uD834\uDC52\uD834\uDC53\uD834\uDC54\uD834\uDC55\uD834\uDC56\uD834\uDC57\uD834\uDC58\uD834\uDC59\uD834\uDC5A\uD834\uDC5B\uD834\uDC5C\uD834\uDC5D\uD834\uDC5E\uD834\uDC5F\uD834\uDC60\uD834\uDC61\uD834\uDC62\uD834\uDC63\uD834\uDC64\uD834\uDC65\uD834\uDC66\uD834\uDC67\uD834\uDC68\uD834\uDC69\uD834\uDC6A\uD834\uDC6B\uD834\uDC6C\uD834\uDC6D\uD834\uDC6E\uD834\uDC6F\uD834\uDC70\uD834\uDC71\uD834\uDC72\uD834\uDC73\uD834\uDC74\uD834\uDC75\uD834\uDC76\uD834\uDC77\uD834\uDC78\uD834\uDC79\uD834\uDC7A\uD834\uDC7B\uD834\uDC7C\uD834\uDC7D\uD834\uDC7E\uD834\uDC7F\uD834\uDC80\uD834\uDC81\uD834\uDC82\uD834\uDC83\uD834\uDC84\uD834\uDC85\uD834\uDC86\uD834\uDC87\uD834\uDC88\uD834\uDC89\uD834\uDC8A\uD834\uDC8B\uD834\uDC8C\uD834\uDC8D\uD834\uDC8E\uD834\uDC8F\uD834\uDC90\uD834\uDC91\uD834\uDC92\uD834\uDC93\uD834\uDC94\uD834\uDC95\uD834\uDC96\uD834\uDC97\uD834\uDC98\uD834\uDC99\uD834\uDC9A\uD834\uDC9B\uD834\uDC9C\uD834\uDC9D\uD834\uDC9E\uD834\uDC9F\uD834\uDCA0\uD834\uDCA1\uD834\uDCA2\uD834\uDCA3\uD834\uDCA4\uD834\uDCA5\uD834\uDCA6\uD834\uDCA7\uD834\uDCA8\uD834\uDCA9\uD834\uDCAA\uD834\uDCAB\uD834\uDCAC\uD834\uDCAD\uD834\uDCAE\uD834\uDCAF\uD834\uDCB0\uD834\uDCB1\uD834\uDCB2\uD834\uDCB3\uD834\uDCB4\uD834\uDCB5\uD834\uDCB6\uD834\uDCB7\uD834\uDCB8\uD834\uDCB9\uD834\uDCBA\uD834\uDCBB\uD834\uDCBC\uD834\uDCBD\uD834\uDCBE\uD834\uDCBF\uD834\uDCC0\uD834\uDCC1\uD834\uDCC2\uD834\uDCC3\uD834\uDCC4\uD834\uDCC5\uD834\uDCC6\uD834\uDCC7\uD834\uDCC8\uD834\uDCC9\uD834\uDCCA\uD834\uDCCB\uD834\uDCCC\uD834\uDCCD\uD834\uDCCE\uD834\uDCCF\uD834\uDCD0\uD834\uDCD1\uD834\uDCD2\uD834\uDCD3\uD834\uDCD4\uD834\uDCD5\uD834\uDCD6\uD834\uDCD7\uD834\uDCD8\uD834\uDCD9\uD834\uDCDA\uD834\uDCDB\uD834\uDCDC\uD834\uDCDD\uD834\uDCDE\uD834\uDCDF\uD834\uDCE0\uD834\uDCE1\uD834\uDCE2\uD834\uDCE3\uD834\uDCE4\uD834\uDCE5\uD834\uDCE6\uD834\uDCE7\uD834\uDCE8\uD834\uDCE9\uD834\uDCEA\uD834\uDCEB\uD834\uDCEC\uD834\uDCED\uD834\uDCEE\uD834\uDCEF\uD834\uDCF0\uD834\uDCF1\uD834\uDCF2\uD834\uDCF3\uD834\uDCF4\uD834\uDCF5\uD834\uDD00\uD834\uDD01\uD834\uDD02\uD834\uDD03\uD834\uDD04\uD834\uDD05\uD834\uDD06\uD834\uDD07\uD834\uDD08\uD834\uDD09\uD834\uDD0A\uD834\uDD0B\uD834\uDD0C\uD834\uDD0D\uD834\uDD0E\uD834\uDD0F\uD834\uDD10\uD834\uDD11\uD834\uDD12\uD834\uDD13\uD834\uDD14\uD834\uDD15\uD834\uDD16\uD834\uDD17\uD834\uDD18\uD834\uDD19\uD834\uDD1A\uD834\uDD1B\uD834\uDD1C\uD834\uDD1D\uD834\uDD1E\uD834\uDD1F\uD834\uDD20\uD834\uDD21\uD834\uDD22\uD834\uDD23\uD834\uDD24\uD834\uDD25\uD834\uDD26\uD834\uDD29\uD834\uDD2A\uD834\uDD2B\uD834\uDD2C\uD834\uDD2D\uD834\uDD2E\uD834\uDD2F\uD834\uDD30\uD834\uDD31\uD834\uDD32\uD834\uDD33\uD834\uDD34\uD834\uDD35\uD834\uDD36\uD834\uDD37\uD834\uDD38\uD834\uDD39\uD834\uDD3A\uD834\uDD3B\uD834\uDD3C\uD834\uDD3D\uD834\uDD3E\uD834\uDD3F\uD834\uDD40\uD834\uDD41\uD834\uDD42\uD834\uDD43\uD834\uDD44\uD834\uDD45\uD834\uDD46\uD834\uDD47\uD834\uDD48\uD834\uDD49\uD834\uDD4A\uD834\uDD4B\uD834\uDD4C\uD834\uDD4D\uD834\uDD4E\uD834\uDD4F\uD834\uDD50\uD834\uDD51\uD834\uDD52\uD834\uDD53\uD834\uDD54\uD834\uDD55\uD834\uDD56\uD834\uDD57\uD834\uDD58\uD834\uDD59\uD834\uDD5A\uD834\uDD5B\uD834\uDD5C\uD834\uDD5D\uD834\uDD5E\uD834\uDD5F\uD834\uDD60\uD834\uDD61\uD834\uDD62\uD834\uDD63\uD834\uDD64\uD834\uDD6A\uD834\uDD6B\uD834\uDD6C\uD834\uDD83\uD834\uDD84\uD834\uDD8C\uD834\uDD8D\uD834\uDD8E\uD834\uDD8F\uD834\uDD90\uD834\uDD91\uD834\uDD92\uD834\uDD93\uD834\uDD94\uD834\uDD95\uD834\uDD96\uD834\uDD97\uD834\uDD98\uD834\uDD99\uD834\uDD9A\uD834\uDD9B\uD834\uDD9C\uD834\uDD9D\uD834\uDD9E\uD834\uDD9F\uD834\uDDA0\uD834\uDDA1\uD834\uDDA2\uD834\uDDA3\uD834\uDDA4\uD834\uDDA5\uD834\uDDA6\uD834\uDDA7\uD834\uDDA8\uD834\uDDA9\uD834\uDDAE\uD834\uDDAF\uD834\uDDB0\uD834\uDDB1\uD834\uDDB2\uD834\uDDB3\uD834\uDDB4\uD834\uDDB5\uD834\uDDB6\uD834\uDDB7\uD834\uDDB8\uD834\uDDB9\uD834\uDDBA\uD834\uDDBB\uD834\uDDBC\uD834\uDDBD\uD834\uDDBE\uD834\uDDBF\uD834\uDDC0\uD834\uDDC1\uD834\uDDC2\uD834\uDDC3\uD834\uDDC4\uD834\uDDC5\uD834\uDDC6\uD834\uDDC7\uD834\uDDC8\uD834\uDDC9\uD834\uDDCA\uD834\uDDCB\uD834\uDDCC\uD834\uDDCD\uD834\uDDCE\uD834\uDDCF\uD834\uDDD0\uD834\uDDD1\uD834\uDDD2\uD834\uDDD3\uD834\uDDD4\uD834\uDDD5\uD834\uDDD6\uD834\uDDD7\uD834\uDDD8\uD834\uDDD9\uD834\uDDDA\uD834\uDDDB\uD834\uDDDC\uD834\uDDDD\uD834\uDE00\uD834\uDE01\uD834\uDE02\uD834\uDE03\uD834\uDE04\uD834\uDE05\uD834\uDE06\uD834\uDE07\uD834\uDE08\uD834\uDE09\uD834\uDE0A\uD834\uDE0B\uD834\uDE0C\uD834\uDE0D\uD834\uDE0E\uD834\uDE0F\uD834\uDE10\uD834\uDE11\uD834\uDE12\uD834\uDE13\uD834\uDE14\uD834\uDE15\uD834\uDE16\uD834\uDE17\uD834\uDE18\uD834\uDE19\uD834\uDE1A\uD834\uDE1B\uD834\uDE1C\uD834\uDE1D\uD834\uDE1E\uD834\uDE1F\uD834\uDE20\uD834\uDE21\uD834\uDE22\uD834\uDE23\uD834\uDE24\uD834\uDE25\uD834\uDE26\uD834\uDE27\uD834\uDE28\uD834\uDE29\uD834\uDE2A\uD834\uDE2B\uD834\uDE2C\uD834\uDE2D\uD834\uDE2E\uD834\uDE2F\uD834\uDE30\uD834\uDE31\uD834\uDE32\uD834\uDE33\uD834\uDE34\uD834\uDE35\uD834\uDE36\uD834\uDE37\uD834\uDE38\uD834\uDE39\uD834\uDE3A\uD834\uDE3B\uD834\uDE3C\uD834\uDE3D\uD834\uDE3E\uD834\uDE3F\uD834\uDE40\uD834\uDE41\uD834\uDE45\uD834\uDF00\uD834\uDF01\uD834\uDF02\uD834\uDF03\uD834\uDF04\uD834\uDF05\uD834\uDF06\uD834\uDF07\uD834\uDF08\uD834\uDF09\uD834\uDF0A\uD834\uDF0B\uD834\uDF0C\uD834\uDF0D\uD834\uDF0E\uD834\uDF0F\uD834\uDF10\uD834\uDF11\uD834\uDF12\uD834\uDF13\uD834\uDF14\uD834\uDF15\uD834\uDF16\uD834\uDF17\uD834\uDF18\uD834\uDF19\uD834\uDF1A\uD834\uDF1B\uD834\uDF1C\uD834\uDF1D\uD834\uDF1E\uD834\uDF1F\uD834\uDF20\uD834\uDF21\uD834\uDF22\uD834\uDF23\uD834\uDF24\uD834\uDF25\uD834\uDF26\uD834\uDF27\uD834\uDF28\uD834\uDF29\uD834\uDF2A\uD834\uDF2B\uD834\uDF2C\uD834\uDF2D\uD834\uDF2E\uD834\uDF2F\uD834\uDF30\uD834\uDF31\uD834\uDF32\uD834\uDF33\uD834\uDF34\uD834\uDF35\uD834\uDF36\uD834\uDF37\uD834\uDF38\uD834\uDF39\uD834\uDF3A\uD834\uDF3B\uD834\uDF3C\uD834\uDF3D\uD834\uDF3E\uD834\uDF3F\uD834\uDF40\uD834\uDF41\uD834\uDF42\uD834\uDF43\uD834\uDF44\uD834\uDF45\uD834\uDF46\uD834\uDF47\uD834\uDF48\uD834\uDF49\uD834\uDF4A\uD834\uDF4B\uD834\uDF4C\uD834\uDF4D\uD834\uDF4E\uD834\uDF4F\uD834\uDF50\uD834\uDF51\uD834\uDF52\uD834\uDF53\uD834\uDF54\uD834\uDF55\uD834\uDF56\uD83C\uDC00\uD83C\uDC01\uD83C\uDC02\uD83C\uDC03\uD83C\uDC04\uD83C\uDC05\uD83C\uDC06\uD83C\uDC07\uD83C\uDC08\uD83C\uDC09\uD83C\uDC0A\uD83C\uDC0B\uD83C\uDC0C\uD83C\uDC0D\uD83C\uDC0E\uD83C\uDC0F\uD83C\uDC10\uD83C\uDC11\uD83C\uDC12\uD83C\uDC13\uD83C\uDC14\uD83C\uDC15\uD83C\uDC16\uD83C\uDC17\uD83C\uDC18\uD83C\uDC19\uD83C\uDC1A\uD83C\uDC1B\uD83C\uDC1C\uD83C\uDC1D\uD83C\uDC1E\uD83C\uDC1F\uD83C\uDC20\uD83C\uDC21\uD83C\uDC22\uD83C\uDC23\uD83C\uDC24\uD83C\uDC25\uD83C\uDC26\uD83C\uDC27\uD83C\uDC28\uD83C\uDC29\uD83C\uDC2A\uD83C\uDC2B\uD83C\uDC30\uD83C\uDC31\uD83C\uDC32\uD83C\uDC33\uD83C\uDC34\uD83C\uDC35\uD83C\uDC36\uD83C\uDC37\uD83C\uDC38\uD83C\uDC39\uD83C\uDC3A\uD83C\uDC3B\uD83C\uDC3C\uD83C\uDC3D\uD83C\uDC3E\uD83C\uDC3F\uD83C\uDC40\uD83C\uDC41\uD83C\uDC42\uD83C\uDC43\uD83C\uDC44\uD83C\uDC45\uD83C\uDC46\uD83C\uDC47\uD83C\uDC48\uD83C\uDC49\uD83C\uDC4A\uD83C\uDC4B\uD83C\uDC4C\uD83C\uDC4D\uD83C\uDC4E\uD83C\uDC4F\uD83C\uDC50\uD83C\uDC51\uD83C\uDC52\uD83C\uDC53\uD83C\uDC54\uD83C\uDC55\uD83C\uDC56\uD83C\uDC57\uD83C\uDC58\uD83C\uDC59\uD83C\uDC5A\uD83C\uDC5B\uD83C\uDC5C\uD83C\uDC5D\uD83C\uDC5E\uD83C\uDC5F\uD83C\uDC60\uD83C\uDC61\uD83C\uDC62\uD83C\uDC63\uD83C\uDC64\uD83C\uDC65\uD83C\uDC66\uD83C\uDC67\uD83C\uDC68\uD83C\uDC69\uD83C\uDC6A\uD83C\uDC6B\uD83C\uDC6C\uD83C\uDC6D\uD83C\uDC6E\uD83C\uDC6F\uD83C\uDC70\uD83C\uDC71\uD83C\uDC72\uD83C\uDC73\uD83C\uDC74\uD83C\uDC75\uD83C\uDC76\uD83C\uDC77\uD83C\uDC78\uD83C\uDC79\uD83C\uDC7A\uD83C\uDC7B\uD83C\uDC7C\uD83C\uDC7D\uD83C\uDC7E\uD83C\uDC7F\uD83C\uDC80\uD83C\uDC81\uD83C\uDC82\uD83C\uDC83\uD83C\uDC84\uD83C\uDC85\uD83C\uDC86\uD83C\uDC87\uD83C\uDC88\uD83C\uDC89\uD83C\uDC8A\uD83C\uDC8B\uD83C\uDC8C\uD83C\uDC8D\uD83C\uDC8E\uD83C\uDC8F\uD83C\uDC90\uD83C\uDC91\uD83C\uDC92\uD83C\uDC93\uD83C\uDCA0\uD83C\uDCA1\uD83C\uDCA2\uD83C\uDCA3\uD83C\uDCA4\uD83C\uDCA5\uD83C\uDCA6\uD83C\uDCA7\uD83C\uDCA8\uD83C\uDCA9\uD83C\uDCAA\uD83C\uDCAB\uD83C\uDCAC\uD83C\uDCAD\uD83C\uDCAE\uD83C\uDCB1\uD83C\uDCB2\uD83C\uDCB3\uD83C\uDCB4\uD83C\uDCB5\uD83C\uDCB6\uD83C\uDCB7\uD83C\uDCB8\uD83C\uDCB9\uD83C\uDCBA\uD83C\uDCBB\uD83C\uDCBC\uD83C\uDCBD\uD83C\uDCBE\uD83C\uDCC1\uD83C\uDCC2\uD83C\uDCC3\uD83C\uDCC4\uD83C\uDCC5\uD83C\uDCC6\uD83C\uDCC7\uD83C\uDCC8\uD83C\uDCC9\uD83C\uDCCA\uD83C\uDCCB\uD83C\uDCCC\uD83C\uDCCD\uD83C\uDCCE\uD83C\uDCCF\uD83C\uDCD1\uD83C\uDCD2\uD83C\uDCD3\uD83C\uDCD4\uD83C\uDCD5\uD83C\uDCD6\uD83C\uDCD7\uD83C\uDCD8\uD83C\uDCD9\uD83C\uDCDA\uD83C\uDCDB\uD83C\uDCDC\uD83C\uDCDD\uD83C\uDCDE\uD83C\uDCDF\uD83C\uDD10\uD83C\uDD11\uD83C\uDD12\uD83C\uDD13\uD83C\uDD14\uD83C\uDD15\uD83C\uDD16\uD83C\uDD17\uD83C\uDD18\uD83C\uDD19\uD83C\uDD1A\uD83C\uDD1B\uD83C\uDD1C\uD83C\uDD1D\uD83C\uDD1E\uD83C\uDD1F\uD83C\uDD20\uD83C\uDD21\uD83C\uDD22\uD83C\uDD23\uD83C\uDD24\uD83C\uDD25\uD83C\uDD26\uD83C\uDD27\uD83C\uDD28\uD83C\uDD29\uD83C\uDD2A\uD83C\uDD2B\uD83C\uDD2C\uD83C\uDD2D\uD83C\uDD2E\uD83C\uDD30\uD83C\uDD31\uD83C\uDD32\uD83C\uDD33\uD83C\uDD34\uD83C\uDD35\uD83C\uDD36\uD83C\uDD37\uD83C\uDD38\uD83C\uDD39\uD83C\uDD3A\uD83C\uDD3B\uD83C\uDD3C\uD83C\uDD3D\uD83C\uDD3E\uD83C\uDD3F\uD83C\uDD40\uD83C\uDD41\uD83C\uDD42\uD83C\uDD43\uD83C\uDD44\uD83C\uDD45\uD83C\uDD46\uD83C\uDD47\uD83C\uDD48\uD83C\uDD49\uD83C\uDD4A\uD83C\uDD4B\uD83C\uDD4C\uD83C\uDD4D\uD83C\uDD4E\uD83C\uDD4F\uD83C\uDD50\uD83C\uDD51\uD83C\uDD52\uD83C\uDD53\uD83C\uDD54\uD83C\uDD55\uD83C\uDD56\uD83C\uDD57\uD83C\uDD58\uD83C\uDD59\uD83C\uDD5A\uD83C\uDD5B\uD83C\uDD5C\uD83C\uDD5D\uD83C\uDD5E\uD83C\uDD5F\uD83C\uDD60\uD83C\uDD61\uD83C\uDD62\uD83C\uDD63\uD83C\uDD64\uD83C\uDD65\uD83C\uDD66\uD83C\uDD67\uD83C\uDD68\uD83C\uDD69\uD83C\uDD70\uD83C\uDD71\uD83C\uDD72\uD83C\uDD73\uD83C\uDD74\uD83C\uDD75\uD83C\uDD76\uD83C\uDD77\uD83C\uDD78\uD83C\uDD79\uD83C\uDD7A\uD83C\uDD7B\uD83C\uDD7C\uD83C\uDD7D\uD83C\uDD7E\uD83C\uDD7F\uD83C\uDD80\uD83C\uDD81\uD83C\uDD82\uD83C\uDD83\uD83C\uDD84\uD83C\uDD85\uD83C\uDD86\uD83C\uDD87\uD83C\uDD88\uD83C\uDD89\uD83C\uDD8A\uD83C\uDD8B\uD83C\uDD8C\uD83C\uDD8D\uD83C\uDD8E\uD83C\uDD8F\uD83C\uDD90\uD83C\uDD91\uD83C\uDD92\uD83C\uDD93\uD83C\uDD94\uD83C\uDD95\uD83C\uDD96\uD83C\uDD97\uD83C\uDD98\uD83C\uDD99\uD83C\uDD9A\uD83C\uDDE6\uD83C\uDDE7\uD83C\uDDE8\uD83C\uDDE9\uD83C\uDDEA\uD83C\uDDEB\uD83C\uDDEC\uD83C\uDDED\uD83C\uDDEE\uD83C\uDDEF\uD83C\uDDF0\uD83C\uDDF1\uD83C\uDDF2\uD83C\uDDF3\uD83C\uDDF4\uD83C\uDDF5\uD83C\uDDF6\uD83C\uDDF7\uD83C\uDDF8\uD83C\uDDF9\uD83C\uDDFA\uD83C\uDDFB\uD83C\uDDFC\uD83C\uDDFD\uD83C\uDDFE\uD83C\uDDFF\uD83C\uDE00\uD83C\uDE01\uD83C\uDE02\uD83C\uDE10\uD83C\uDE11\uD83C\uDE12\uD83C\uDE13\uD83C\uDE14\uD83C\uDE15\uD83C\uDE16\uD83C\uDE17\uD83C\uDE18\uD83C\uDE19\uD83C\uDE1A\uD83C\uDE1B\uD83C\uDE1C\uD83C\uDE1D\uD83C\uDE1E\uD83C\uDE1F\uD83C\uDE20\uD83C\uDE21\uD83C\uDE22\uD83C\uDE23\uD83C\uDE24\uD83C\uDE25\uD83C\uDE26\uD83C\uDE27\uD83C\uDE28\uD83C\uDE29\uD83C\uDE2A\uD83C\uDE2B\uD83C\uDE2C\uD83C\uDE2D\uD83C\uDE2E\uD83C\uDE2F\uD83C\uDE30\uD83C\uDE31\uD83C\uDE32\uD83C\uDE33\uD83C\uDE34\uD83C\uDE35\uD83C\uDE36\uD83C\uDE37\uD83C\uDE38\uD83C\uDE39\uD83C\uDE3A\uD83C\uDE40\uD83C\uDE41\uD83C\uDE42\uD83C\uDE43\uD83C\uDE44\uD83C\uDE45\uD83C\uDE46\uD83C\uDE47\uD83C\uDE48\uD83C\uDE50\uD83C\uDE51\uD83C\uDF00\uD83C\uDF01\uD83C\uDF02\uD83C\uDF03\uD83C\uDF04\uD83C\uDF05\uD83C\uDF06\uD83C\uDF07\uD83C\uDF08\uD83C\uDF09\uD83C\uDF0A\uD83C\uDF0B\uD83C\uDF0C\uD83C\uDF0D\uD83C\uDF0E\uD83C\uDF0F\uD83C\uDF10\uD83C\uDF11\uD83C\uDF12\uD83C\uDF13\uD83C\uDF14\uD83C\uDF15\uD83C\uDF16\uD83C\uDF17\uD83C\uDF18\uD83C\uDF19\uD83C\uDF1A\uD83C\uDF1B\uD83C\uDF1C\uD83C\uDF1D\uD83C\uDF1E\uD83C\uDF1F\uD83C\uDF20\uD83C\uDF30\uD83C\uDF31\uD83C\uDF32\uD83C\uDF33\uD83C\uDF34\uD83C\uDF35\uD83C\uDF37\uD83C\uDF38\uD83C\uDF39\uD83C\uDF3A\uD83C\uDF3B\uD83C\uDF3C\uD83C\uDF3D\uD83C\uDF3E\uD83C\uDF3F\uD83C\uDF40\uD83C\uDF41\uD83C\uDF42\uD83C\uDF43\uD83C\uDF44\uD83C\uDF45\uD83C\uDF46\uD83C\uDF47\uD83C\uDF48\uD83C\uDF49\uD83C\uDF4A\uD83C\uDF4B\uD83C\uDF4C\uD83C\uDF4D\uD83C\uDF4E\uD83C\uDF4F\uD83C\uDF50\uD83C\uDF51\uD83C\uDF52\uD83C\uDF53\uD83C\uDF54\uD83C\uDF55\uD83C\uDF56\uD83C\uDF57\uD83C\uDF58\uD83C\uDF59\uD83C\uDF5A\uD83C\uDF5B\uD83C\uDF5C\uD83C\uDF5D\uD83C\uDF5E\uD83C\uDF5F\uD83C\uDF60\uD83C\uDF61\uD83C\uDF62\uD83C\uDF63\uD83C\uDF64\uD83C\uDF65\uD83C\uDF66\uD83C\uDF67\uD83C\uDF68\uD83C\uDF69\uD83C\uDF6A\uD83C\uDF6B\uD83C\uDF6C\uD83C\uDF6D\uD83C\uDF6E\uD83C\uDF6F\uD83C\uDF70\uD83C\uDF71\uD83C\uDF72\uD83C\uDF73\uD83C\uDF74\uD83C\uDF75\uD83C\uDF76\uD83C\uDF77\uD83C\uDF78\uD83C\uDF79\uD83C\uDF7A\uD83C\uDF7B\uD83C\uDF7C\uD83C\uDF80\uD83C\uDF81\uD83C\uDF82\uD83C\uDF83\uD83C\uDF84\uD83C\uDF85\uD83C\uDF86\uD83C\uDF87\uD83C\uDF88\uD83C\uDF89\uD83C\uDF8A\uD83C\uDF8B\uD83C\uDF8C\uD83C\uDF8D\uD83C\uDF8E\uD83C\uDF8F\uD83C\uDF90\uD83C\uDF91\uD83C\uDF92\uD83C\uDF93\uD83C\uDFA0\uD83C\uDFA1\uD83C\uDFA2\uD83C\uDFA3\uD83C\uDFA4\uD83C\uDFA5\uD83C\uDFA6\uD83C\uDFA7\uD83C\uDFA8\uD83C\uDFA9\uD83C\uDFAA\uD83C\uDFAB\uD83C\uDFAC\uD83C\uDFAD\uD83C\uDFAE\uD83C\uDFAF\uD83C\uDFB0\uD83C\uDFB1\uD83C\uDFB2\uD83C\uDFB3\uD83C\uDFB4\uD83C\uDFB5\uD83C\uDFB6\uD83C\uDFB7\uD83C\uDFB8\uD83C\uDFB9\uD83C\uDFBA\uD83C\uDFBB\uD83C\uDFBC\uD83C\uDFBD\uD83C\uDFBE\uD83C\uDFBF\uD83C\uDFC0\uD83C\uDFC1\uD83C\uDFC2\uD83C\uDFC3\uD83C\uDFC4\uD83C\uDFC6\uD83C\uDFC7\uD83C\uDFC8\uD83C\uDFC9\uD83C\uDFCA\uD83C\uDFE0\uD83C\uDFE1\uD83C\uDFE2\uD83C\uDFE3\uD83C\uDFE4\uD83C\uDFE5\uD83C\uDFE6\uD83C\uDFE7\uD83C\uDFE8\uD83C\uDFE9\uD83C\uDFEA\uD83C\uDFEB\uD83C\uDFEC\uD83C\uDFED\uD83C\uDFEE\uD83C\uDFEF\uD83C\uDFF0\uD83D\uDC00\uD83D\uDC01\uD83D\uDC02\uD83D\uDC03\uD83D\uDC04\uD83D\uDC05\uD83D\uDC06\uD83D\uDC07\uD83D\uDC08\uD83D\uDC09\uD83D\uDC0A\uD83D\uDC0B\uD83D\uDC0C\uD83D\uDC0D\uD83D\uDC0E\uD83D\uDC0F\uD83D\uDC10\uD83D\uDC11\uD83D\uDC12\uD83D\uDC13\uD83D\uDC14\uD83D\uDC15\uD83D\uDC16\uD83D\uDC17\uD83D\uDC18\uD83D\uDC19\uD83D\uDC1A\uD83D\uDC1B\uD83D\uDC1C\uD83D\uDC1D\uD83D\uDC1E\uD83D\uDC1F\uD83D\uDC20\uD83D\uDC21\uD83D\uDC22\uD83D\uDC23\uD83D\uDC24\uD83D\uDC25\uD83D\uDC26\uD83D\uDC27\uD83D\uDC28\uD83D\uDC29\uD83D\uDC2A\uD83D\uDC2B\uD83D\uDC2C\uD83D\uDC2D\uD83D\uDC2E\uD83D\uDC2F\uD83D\uDC30\uD83D\uDC31\uD83D\uDC32\uD83D\uDC33\uD83D\uDC34\uD83D\uDC35\uD83D\uDC36\uD83D\uDC37\uD83D\uDC38\uD83D\uDC39\uD83D\uDC3A\uD83D\uDC3B\uD83D\uDC3C\uD83D\uDC3D\uD83D\uDC3E\uD83D\uDC40\uD83D\uDC42\uD83D\uDC43\uD83D\uDC44\uD83D\uDC45\uD83D\uDC46\uD83D\uDC47\uD83D\uDC48\uD83D\uDC49\uD83D\uDC4A\uD83D\uDC4B\uD83D\uDC4C\uD83D\uDC4D\uD83D\uDC4E\uD83D\uDC4F\uD83D\uDC50\uD83D\uDC51\uD83D\uDC52\uD83D\uDC53\uD83D\uDC54\uD83D\uDC55\uD83D\uDC56\uD83D\uDC57\uD83D\uDC58\uD83D\uDC59\uD83D\uDC5A\uD83D\uDC5B\uD83D\uDC5C\uD83D\uDC5D\uD83D\uDC5E\uD83D\uDC5F\uD83D\uDC60\uD83D\uDC61\uD83D\uDC62\uD83D\uDC63\uD83D\uDC64\uD83D\uDC65\uD83D\uDC66\uD83D\uDC67\uD83D\uDC68\uD83D\uDC69\uD83D\uDC6A\uD83D\uDC6B\uD83D\uDC6C\uD83D\uDC6D\uD83D\uDC6E\uD83D\uDC6F\uD83D\uDC70\uD83D\uDC71\uD83D\uDC72\uD83D\uDC73\uD83D\uDC74\uD83D\uDC75\uD83D\uDC76\uD83D\uDC77\uD83D\uDC78\uD83D\uDC79\uD83D\uDC7A\uD83D\uDC7B\uD83D\uDC7C\uD83D\uDC7D\uD83D\uDC7E\uD83D\uDC7F\uD83D\uDC80\uD83D\uDC81\uD83D\uDC82\uD83D\uDC83\uD83D\uDC84\uD83D\uDC85\uD83D\uDC86\uD83D\uDC87\uD83D\uDC88\uD83D\uDC89\uD83D\uDC8A\uD83D\uDC8B\uD83D\uDC8C\uD83D\uDC8D\uD83D\uDC8E\uD83D\uDC8F\uD83D\uDC90\uD83D\uDC91\uD83D\uDC92\uD83D\uDC93\uD83D\uDC94\uD83D\uDC95\uD83D\uDC96\uD83D\uDC97\uD83D\uDC98\uD83D\uDC99\uD83D\uDC9A\uD83D\uDC9B\uD83D\uDC9C\uD83D\uDC9D\uD83D\uDC9E\uD83D\uDC9F\uD83D\uDCA0\uD83D\uDCA1\uD83D\uDCA2\uD83D\uDCA3\uD83D\uDCA4\uD83D\uDCA5\uD83D\uDCA6\uD83D\uDCA7\uD83D\uDCA8\uD83D\uDCA9\uD83D\uDCAA\uD83D\uDCAB\uD83D\uDCAC\uD83D\uDCAD\uD83D\uDCAE\uD83D\uDCAF\uD83D\uDCB0\uD83D\uDCB1\uD83D\uDCB2\uD83D\uDCB3\uD83D\uDCB4\uD83D\uDCB5\uD83D\uDCB6\uD83D\uDCB7\uD83D\uDCB8\uD83D\uDCB9\uD83D\uDCBA\uD83D\uDCBB\uD83D\uDCBC\uD83D\uDCBD\uD83D\uDCBE\uD83D\uDCBF\uD83D\uDCC0\uD83D\uDCC1\uD83D\uDCC2\uD83D\uDCC3\uD83D\uDCC4\uD83D\uDCC5\uD83D\uDCC6\uD83D\uDCC7\uD83D\uDCC8\uD83D\uDCC9\uD83D\uDCCA\uD83D\uDCCB\uD83D\uDCCC\uD83D\uDCCD\uD83D\uDCCE\uD83D\uDCCF\uD83D\uDCD0\uD83D\uDCD1\uD83D\uDCD2\uD83D\uDCD3\uD83D\uDCD4\uD83D\uDCD5\uD83D\uDCD6\uD83D\uDCD7\uD83D\uDCD8\uD83D\uDCD9\uD83D\uDCDA\uD83D\uDCDB\uD83D\uDCDC\uD83D\uDCDD\uD83D\uDCDE\uD83D\uDCDF\uD83D\uDCE0\uD83D\uDCE1\uD83D\uDCE2\uD83D\uDCE3\uD83D\uDCE4\uD83D\uDCE5\uD83D\uDCE6\uD83D\uDCE7\uD83D\uDCE8\uD83D\uDCE9\uD83D\uDCEA\uD83D\uDCEB\uD83D\uDCEC\uD83D\uDCED\uD83D\uDCEE\uD83D\uDCEF\uD83D\uDCF0\uD83D\uDCF1\uD83D\uDCF2\uD83D\uDCF3\uD83D\uDCF4\uD83D\uDCF5\uD83D\uDCF6\uD83D\uDCF7\uD83D\uDCF9\uD83D\uDCFA\uD83D\uDCFB\uD83D\uDCFC\uD83D\uDD00\uD83D\uDD01\uD83D\uDD02\uD83D\uDD03\uD83D\uDD04\uD83D\uDD05\uD83D\uDD06\uD83D\uDD07\uD83D\uDD08\uD83D\uDD09\uD83D\uDD0A\uD83D\uDD0B\uD83D\uDD0C\uD83D\uDD0D\uD83D\uDD0E\uD83D\uDD0F\uD83D\uDD10\uD83D\uDD11\uD83D\uDD12\uD83D\uDD13\uD83D\uDD14\uD83D\uDD15\uD83D\uDD16\uD83D\uDD17\uD83D\uDD18\uD83D\uDD19\uD83D\uDD1A\uD83D\uDD1B\uD83D\uDD1C\uD83D\uDD1D\uD83D\uDD1E\uD83D\uDD1F\uD83D\uDD20\uD83D\uDD21\uD83D\uDD22\uD83D\uDD23\uD83D\uDD24\uD83D\uDD25\uD83D\uDD26\uD83D\uDD27\uD83D\uDD28\uD83D\uDD29\uD83D\uDD2A\uD83D\uDD2B\uD83D\uDD2C\uD83D\uDD2D\uD83D\uDD2E\uD83D\uDD2F\uD83D\uDD30\uD83D\uDD31\uD83D\uDD32\uD83D\uDD33\uD83D\uDD34\uD83D\uDD35\uD83D\uDD36\uD83D\uDD37\uD83D\uDD38\uD83D\uDD39\uD83D\uDD3A\uD83D\uDD3B\uD83D\uDD3C\uD83D\uDD3D\uD83D\uDD50\uD83D\uDD51\uD83D\uDD52\uD83D\uDD53\uD83D\uDD54\uD83D\uDD55\uD83D\uDD56\uD83D\uDD57\uD83D\uDD58\uD83D\uDD59\uD83D\uDD5A\uD83D\uDD5B\uD83D\uDD5C\uD83D\uDD5D\uD83D\uDD5E\uD83D\uDD5F\uD83D\uDD60\uD83D\uDD61\uD83D\uDD62\uD83D\uDD63\uD83D\uDD64\uD83D\uDD65\uD83D\uDD66\uD83D\uDD67\uD83D\uDDFB\uD83D\uDDFC\uD83D\uDDFD\uD83D\uDDFE\uD83D\uDDFF\uD83D\uDE01\uD83D\uDE02\uD83D\uDE03\uD83D\uDE04\uD83D\uDE05\uD83D\uDE06\uD83D\uDE07\uD83D\uDE08\uD83D\uDE09\uD83D\uDE0A\uD83D\uDE0B\uD83D\uDE0C\uD83D\uDE0D\uD83D\uDE0E\uD83D\uDE0F\uD83D\uDE10\uD83D\uDE12\uD83D\uDE13\uD83D\uDE14\uD83D\uDE16\uD83D\uDE18\uD83D\uDE1A\uD83D\uDE1C\uD83D\uDE1D\uD83D\uDE1E\uD83D\uDE20\uD83D\uDE21\uD83D\uDE22\uD83D\uDE23\uD83D\uDE24\uD83D\uDE25\uD83D\uDE28\uD83D\uDE29\uD83D\uDE2A\uD83D\uDE2B\uD83D\uDE2D\uD83D\uDE30\uD83D\uDE31\uD83D\uDE32\uD83D\uDE33\uD83D\uDE35\uD83D\uDE36\uD83D\uDE37\uD83D\uDE38\uD83D\uDE39\uD83D\uDE3A\uD83D\uDE3B\uD83D\uDE3C\uD83D\uDE3D\uD83D\uDE3E\uD83D\uDE3F\uD83D\uDE40\uD83D\uDE45\uD83D\uDE46\uD83D\uDE47\uD83D\uDE48\uD83D\uDE49\uD83D\uDE4A\uD83D\uDE4B\uD83D\uDE4C\uD83D\uDE4D\uD83D\uDE4E\uD83D\uDE4F\uD83D\uDE80\uD83D\uDE81\uD83D\uDE82\uD83D\uDE83\uD83D\uDE84\uD83D\uDE85\uD83D\uDE86\uD83D\uDE87\uD83D\uDE88\uD83D\uDE89\uD83D\uDE8A\uD83D\uDE8B\uD83D\uDE8C\uD83D\uDE8D\uD83D\uDE8E\uD83D\uDE8F\uD83D\uDE90\uD83D\uDE91\uD83D\uDE92\uD83D\uDE93\uD83D\uDE94\uD83D\uDE95\uD83D\uDE96\uD83D\uDE97\uD83D\uDE98\uD83D\uDE99\uD83D\uDE9A\uD83D\uDE9B\uD83D\uDE9C\uD83D\uDE9D\uD83D\uDE9E\uD83D\uDE9F\uD83D\uDEA0\uD83D\uDEA1\uD83D\uDEA2\uD83D\uDEA3\uD83D\uDEA4\uD83D\uDEA5\uD83D\uDEA6\uD83D\uDEA7\uD83D\uDEA8\uD83D\uDEA9\uD83D\uDEAA\uD83D\uDEAB\uD83D\uDEAC\uD83D\uDEAD\uD83D\uDEAE\uD83D\uDEAF\uD83D\uDEB0\uD83D\uDEB1\uD83D\uDEB2\uD83D\uDEB3\uD83D\uDEB4\uD83D\uDEB5\uD83D\uDEB6\uD83D\uDEB7\uD83D\uDEB8\uD83D\uDEB9\uD83D\uDEBA\uD83D\uDEBB\uD83D\uDEBC\uD83D\uDEBD\uD83D\uDEBE\uD83D\uDEBF\uD83D\uDEC0\uD83D\uDEC1\uD83D\uDEC2\uD83D\uDEC3\uD83D\uDEC4\uD83D\uDEC5\uD83D\uDF00\uD83D\uDF01\uD83D\uDF02\uD83D\uDF03\uD83D\uDF04\uD83D\uDF05\uD83D\uDF06\uD83D\uDF07\uD83D\uDF08\uD83D\uDF09\uD83D\uDF0A\uD83D\uDF0B\uD83D\uDF0C\uD83D\uDF0D\uD83D\uDF0E\uD83D\uDF0F\uD83D\uDF10\uD83D\uDF11\uD83D\uDF12\uD83D\uDF13\uD83D\uDF14\uD83D\uDF15\uD83D\uDF16\uD83D\uDF17\uD83D\uDF18\uD83D\uDF19\uD83D\uDF1A\uD83D\uDF1B\uD83D\uDF1C\uD83D\uDF1D\uD83D\uDF1E\uD83D\uDF1F\uD83D\uDF20\uD83D\uDF21\uD83D\uDF22\uD83D\uDF23\uD83D\uDF24\uD83D\uDF25\uD83D\uDF26\uD83D\uDF27\uD83D\uDF28\uD83D\uDF29\uD83D\uDF2A\uD83D\uDF2B\uD83D\uDF2C\uD83D\uDF2D\uD83D\uDF2E\uD83D\uDF2F\uD83D\uDF30\uD83D\uDF31\uD83D\uDF32\uD83D\uDF33\uD83D\uDF34\uD83D\uDF35\uD83D\uDF36\uD83D\uDF37\uD83D\uDF38\uD83D\uDF39\uD83D\uDF3A\uD83D\uDF3B\uD83D\uDF3C\uD83D\uDF3D\uD83D\uDF3E\uD83D\uDF3F\uD83D\uDF40\uD83D\uDF41\uD83D\uDF42\uD83D\uDF43\uD83D\uDF44\uD83D\uDF45\uD83D\uDF46\uD83D\uDF47\uD83D\uDF48\uD83D\uDF49\uD83D\uDF4A\uD83D\uDF4B\uD83D\uDF4C\uD83D\uDF4D\uD83D\uDF4E\uD83D\uDF4F\uD83D\uDF50\uD83D\uDF51\uD83D\uDF52\uD83D\uDF53\uD83D\uDF54\uD83D\uDF55\uD83D\uDF56\uD83D\uDF57\uD83D\uDF58\uD83D\uDF59\uD83D\uDF5A\uD83D\uDF5B\uD83D\uDF5C\uD83D\uDF5D\uD83D\uDF5E\uD83D\uDF5F\uD83D\uDF60\uD83D\uDF61\uD83D\uDF62\uD83D\uDF63\uD83D\uDF64\uD83D\uDF65\uD83D\uDF66\uD83D\uDF67\uD83D\uDF68\uD83D\uDF69\uD83D\uDF6A\uD83D\uDF6B\uD83D\uDF6C\uD83D\uDF6D\uD83D\uDF6E\uD83D\uDF6F\uD83D\uDF70\uD83D\uDF71\uD83D\uDF72\uD83D\uDF73'
			},
			{
				'description': 'Unicode character category [Zl]: separator, line',
				'decoded': [8232],
				'encoded': '\u2028'
			},
			{
				'description': 'Unicode character category [Zp]: separator, paragraph',
				'decoded': [8233],
				'encoded': '\u2029'
			},
			{
				'description': 'Unicode character category [Zs]: separator, space',
				'decoded': [32, 160, 5760, 6158, 8192, 8193, 8194, 8195, 8196, 8197, 8198, 8199, 8200, 8201, 8202, 8239, 8287, 12288],
				'encoded': ' \xA0\u1680\u180E\u2000\u2001\u2002\u2003\u2004\u2005\u2006\u2007\u2008\u2009\u200A\u202F\u205F\u3000'
			},
			{
				'description': 'U+10000 (supplementary symbol)',
				'decoded': [65536],
				'encoded': '\uD800\uDC00'
			},
			{
				'description': 'U+D800 (high surrogate)',
				'decoded': [55296],
				'encoded': '\uD800'
			},
			{
				'description': 'U+D800 (high surrogate) followed by non-surrogates',
				'decoded': [55296, 97, 98],
				'encoded': '\uD800ab'
			},
			{
				'description': 'U+DC00 (low surrogate)',
				'decoded': [56320],
				'encoded': '\uDC00'
			},
			{
				'description': 'U+DC00 (low surrogate) followed by non-surrogates',
				'decoded': [56320, 97, 98],
				'encoded': '\uDC00ab'
			},
			{
				'description': 'U+1D306 (supplementary symbol)',
				'decoded': [119558],
				'encoded': '\uD834\uDF06'
			},
			{
				'description': 'U+D834 (high surrogate)',
				'decoded': [55348],
				'encoded': '\uD834'
			},
			{
				'description': 'U+DF06 (low surrogate)',
				'decoded': [57094],
				'encoded': '\uDF06'
			}
		],
		'domains': [
			{
				'decoded': 'ma\xF1ana.com',
				'encoded': 'xn--maana-pta.com'
			},
			{
				'decoded': 'b\xFCcher.com',
				'encoded': 'xn--bcher-kva.com'
			},
			{
				'decoded': 'caf\xE9.com',
				'encoded': 'xn--caf-dma.com'
			},
			{
				'decoded': '\u2603-\u2318.com',
				'encoded': 'xn----dqo34k.com'
			},
			{
				'decoded': '\uD400\u2603-\u2318.com',
				'encoded': 'xn----dqo34kn65z.com'
			},
			{
				'description': 'Emoji',
				'decoded': '\uD83D\uDCA9.la',
				'encoded': 'xn--ls8h.la'
			}
		],
		'separators': [
			{
				'description': 'Using U+002E as separator',
				'decoded': 'ma\xF1ana\x2Ecom',
				'encoded': 'xn--maana-pta.com'
			},
			{
				'description': 'Using U+3002 as separator',
				'decoded': 'ma\xF1ana\u3002com',
				'encoded': 'xn--maana-pta.com'
			},
			{
				'description': 'Using U+FF0E as separator',
				'decoded': 'ma\xF1ana\uFF0Ecom',
				'encoded': 'xn--maana-pta.com'
			},
			{
				'description': 'Using U+FF61 as separator',
				'decoded': 'ma\xF1ana\uFF61com',
				'encoded': 'xn--maana-pta.com'
			}
		]
	};

	/*--------------------------------------------------------------------------*/

	// simple `Array#forEach`-like helper function
	function each(array, fn) {
		var length = array.length;
		while (length--) {
			fn(array[length]);
		}
	}

	/*--------------------------------------------------------------------------*/

	test('punycode.ucs2.decode', function() {
		each(testData.ucs2, function(object) {
			deepEqual(punycode.ucs2.decode(object.encoded), object.decoded, object.description);
		});
	});

	test('punycode.ucs2.encode', function() {
		each(testData.ucs2, function(object) {
			equal(punycode.ucs2.encode(object.decoded), object.encoded, object.description);
		});
	});

	test('punycode.decode', function() {
		each(testData.strings, function(object) {
			equal(punycode.decode(object.encoded), object.decoded, object.description);
		});
	});

	test('punycode.encode', function() {
		each(testData.strings, function(object) {
			equal(punycode.encode(object.decoded), object.encoded, object.description);
		});
	});

	test('punycode.toUnicode', function() {
		each(testData.domains, function(object) {
			equal(punycode.toUnicode(object.encoded), object.decoded, object.description);
		});
		/**
		 * Domain names (or other strings) that don't start with `xn--` may not be
		 * converted.
		 */
		each(testData.strings, function(object) {
			var message = 'Domain names (or other strings) that don\'t start with `xn--` may not be converted';
			equal(punycode.toUnicode(object.encoded), object.encoded, message);
			equal(punycode.toUnicode(object.decoded), object.decoded, message);
		});
	});

	test('punycode.toASCII', function() {
		each(testData.domains, function(object) {
			equal(punycode.toASCII(object.decoded), object.encoded, object.description);
		});
		/**
		 * Domain names (or other strings) that are already in ASCII may not be
		 * converted.
		 */
		each(testData.strings, function(object) {
			equal(punycode.toASCII(object.encoded), object.encoded, 'Domain names (or other strings) that are already in ASCII may not be converted');
		});
		/**
		 * IDNA2003 separators must be supported for backwards compatibility.
		 */
		each(testData.separators, function(object) {
			var message = 'IDNA2003 separators must be supported for backwards compatibility';
			equal(punycode.toASCII(object.decoded), object.encoded, message);
		});
	});

	if (root.document && root.require) {
		test('require(\'punycode\')', function() {
			equal((punycode2 || {}).version, punycode.version, 'require(\'punycode\')');
		});
	}

	/*--------------------------------------------------------------------------*/

	// explicitly call `QUnit.start()` in a CLI environment
	if (!root.document) {
		QUnit.start();
	}

}(typeof global == 'object' && global || this));